package ru.yandex.direct.excel.processing.service.internalad;

import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.banner.container.InternalBannerAddOrUpdateItem;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperation;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.ResultConverters;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.identity;
import static ru.yandex.direct.validation.Predicates.not;

@Service
public class AddOrUpdateInternalAdsService {

    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final BannersAddOperationFactory bannersAddOperationFactory;

    @Autowired
    public AddOrUpdateInternalAdsService(BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                         BannersAddOperationFactory bannersAddOperationFactory) {
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
    }

    /**
     * Комплексная операция добавления и обновления баннеров внутренней рекламы
     * Умеет добавлять новый баннер в новую группу (которая будет создана после валидации). Подробнее:
     * {@link #createFullAddOperation(List, Long, ClientId, boolean, boolean)}
     *
     * @param items          список баннеров для добавления и обновления. Если id != null, то данный баннер будет
     *                       обновлен.
     * @param operatorUid    uid оператора
     * @param clientId       id клиента
     * @param saveDraft      сохранить ли как черновик
     * @param validationOnly если true, то изменений в базе не будет
     * @return результат выполнения операций добавления и обновления
     */
    public MassResult<Long> addOrUpdateInternalAds(List<InternalBannerAddOrUpdateItem> items, long operatorUid,
                                                   ClientId clientId, boolean saveDraft, boolean validationOnly) {
        var updateOperationCreator = getUpdateOperationCreator(operatorUid, clientId);
        var addOperationCreator = getAddOperationCreator(operatorUid, clientId, saveDraft, validationOnly);

        SplitAndMergeOperationAggregator<InternalBannerAddOrUpdateItem, Long> aggregatedAddOperation =
                SplitAndMergeOperationAggregator.builder()
                        .addSubOperation(AddOrUpdateInternalAdsService::isAdItemForUpdate, updateOperationCreator)
                        .addSubOperation(not(AddOrUpdateInternalAdsService::isAdItemForUpdate), addOperationCreator)
                        .build();
        if (validationOnly) {
            return aggregatedAddOperation.validate(items);
        }
        return aggregatedAddOperation.execute(items);
    }

    public static boolean isAdItemForUpdate(InternalBannerAddOrUpdateItem item) {
        return isBannerForUpdate(item.getBanner());
    }

    public static boolean isBannerForUpdate(InternalBanner internalBanner) {
        return internalBanner.getId() != null;
    }

    private OperationCreator<InternalBannerAddOrUpdateItem, Operation<Long>> getAddOperationCreator(
            Long operatorUid, ClientId clientId, boolean saveDraft, boolean validationOnly) {
        return inputItems -> new ConvertResultOperation<>(
                createFullAddOperation(inputItems, operatorUid, clientId, saveDraft, validationOnly),
                ResultConverters.massResultValueConverter(identity()));
    }

    /**
     * Создает операцию для создания баннеров
     * Если на вход получили баннеры с группами, то значит возможно будут баннеры, которые добавляются в новые группы в
     * рамках комплексной операции. Новых групп нет в базе в режиме валидации: validationOnly=true, поэтому в операцию
     * передаем информация о группах. А если операцию выполняем не в режиме валидации, то для баннеров проставляем id
     * новых групп
     * Если на вход не получили группы для баннеров, то значит в рамках коплексной операции не добавляем группы
     * Комплексная операция вызывается тут: {@link InternalAdExcelImportService#importFromExcel}
     */
    private BannersAddOperation createFullAddOperation(List<InternalBannerAddOrUpdateItem> inputItems,
                                                       Long operatorUid,
                                                       ClientId clientId, boolean saveDraft,
                                                       boolean validationOnly) {

        List<BannerWithAdGroupId> internalBanners = mapList(inputItems, InternalBannerAddOrUpdateItem::getBanner);

        var operation = bannersAddOperationFactory
                .createFullAddOperation(internalBanners, clientId, operatorUid, saveDraft);

        Map<Integer, AdGroupForBannerOperation> adGroupByIndex = EntryStream.of(inputItems)
                .mapValues(InternalBannerAddOrUpdateItem::getAdGroup)
                .nonNullValues()
                .mapValues(AdGroupForBannerOperation.class::cast)
                .toImmutableMap();
        if (adGroupByIndex.isEmpty()) {
            return operation;
        }

        if (validationOnly) {
            operation.setAdGroupsInfo(adGroupByIndex);
        } else {
            // Проставим id новых груп для их баннеров. Ожидается, что новые группы уже созданы и для них проставлены id
            Map<Integer, Long> adGroupIdByIndex = EntryStream.of(adGroupByIndex)
                    .mapValues(AdGroupForBannerOperation::getId)
                    .nonNullValues()
                    .toImmutableMap();
            operation.setAdGroupIds(adGroupIdByIndex);
        }
        return operation;
    }

    private OperationCreator<InternalBannerAddOrUpdateItem, Operation<Long>> getUpdateOperationCreator(
            Long operatorUid, ClientId clientId) {
        return inputItems -> new ConvertResultOperation<>(
                bannersUpdateOperationFactory.createFullUpdateOperation(
                        mapList(inputItems, AddOrUpdateInternalAdsService::toInternalBannerChanges),
                        operatorUid, clientId, DatabaseMode.ONLY_MYSQL),
                ResultConverters.massResultValueConverter(identity()));
    }

    private static ModelChanges<BannerWithSystemFields> toInternalBannerChanges(InternalBannerAddOrUpdateItem item) {
        InternalBanner banner = item.getBanner();
        var modelChanges = new ModelChanges<>(banner.getId(), InternalBanner.class)
                .process(banner.getDescription(), InternalBanner.DESCRIPTION)
                .process(banner.getTemplateVariables(), InternalBanner.TEMPLATE_VARIABLES)
                .process(banner.getModerationInfo(), InternalBanner.MODERATION_INFO)
                .process(banner.getStatusShow(), InternalBanner.STATUS_SHOW);
        return modelChanges.castModelUp(BannerWithSystemFields.class);
    }

}
