package ru.yandex.direct.excel.processing.service.internalad;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.crypta.AudienceType;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope;
import ru.yandex.direct.core.entity.retargeting.model.Goal;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@Service
@ParametersAreNonnullByDefault
public class CryptaSegmentDictionariesService {
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final Supplier<Map<Long, Goal>> segmentMap;

    @Autowired
    public CryptaSegmentDictionariesService(CryptaSegmentRepository cryptaSegmentRepository) {
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.segmentMap = Suppliers.memoizeWithExpiration(this::getSegmentMap, 30, TimeUnit.MINUTES);
    }

    private Map<Long, Goal> getSegmentMap() {
        return cryptaSegmentRepository.getAll(CryptaGoalScope.INTERNAL_AD);
    }

    @Nullable
    public Goal getGoalIdByKeywordAndType(String keyword, String keywordValue) {
        return StreamEx.ofValues(segmentMap.get())
                .filter(goal -> goal.getKeyword() != null && goal.getKeywordValue() != null
                        && goal.getKeyword().equals(keyword) && goal.getKeywordValue().equals(keywordValue))
                .findFirst()
                .orElse(null);
    }

    @Nullable
    public Goal getCryptaByGoalId(Long goalId) {
        return segmentMap.get().get(goalId);
    }

    public List<Long> getCryptaGoalIdsByParentId(Long parentId) {
        return filterAndMapList(segmentMap.get().values(),
                goal -> parentId.equals(goal.getParentId()),
                Goal::getId);
    }

    /**
     * Находит цель по parentId и name.
     * Для поля name делается регистронезависимая проверка. Ожидается, что в базе у нужных целей уникальные имена без
     * учета регистра
     */
    @Nullable
    public Goal getGoalIdByParentIdAndName(Long parentId, String name) {
        return StreamEx.ofValues(segmentMap.get())
                .filter(goal -> parentId.equals(goal.getParentId()) && name.equalsIgnoreCase(goal.getName()))
                .findFirst()
                .orElse(null);
    }

    public List<String> getGenderValues() {
        return getCryptaNamesByParentId(AudienceType.GENDER.getTypedValue());
    }

    public List<String> getAgeValues() {
        return getCryptaNamesByParentId(AudienceType.AGE.getTypedValue());
    }

    public List<String> getIncomeValues() {
        return getCryptaNamesByParentId(AudienceType.INCOME.getTypedValue());
    }

    private List<String> getCryptaNamesByParentId(Long parentId) {
        return filterAndMapList(segmentMap.get().values(),
                goal -> parentId.equals(goal.getParentId()),
                Goal::getName);
    }

}
