package ru.yandex.direct.excel.processing.service.internalad;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddOrUpdateItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.banner.container.InternalBannerAddOrUpdateItem;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.ExcelImportResult;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdGroupRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.SheetDescriptor;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator.getMassResultWithAggregatedValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class InternalAdExcelImportService {

    private static final boolean SAVE_DRAFT = false;

    private final AddOrUpdateInternalAdGroupsService addOrUpdateInternalAdGroupsService;
    private final AddOrUpdateInternalAdsService addOrUpdateInternalAdsService;

    @Autowired
    public InternalAdExcelImportService(AddOrUpdateInternalAdGroupsService addOrUpdateInternalAdGroupsService,
                                        AddOrUpdateInternalAdsService addOrUpdateInternalAdsService) {
        this.addOrUpdateInternalAdGroupsService = addOrUpdateInternalAdGroupsService;
        this.addOrUpdateInternalAdsService = addOrUpdateInternalAdsService;
    }

    /**
     * Импорт групп и объявлений внутренней рекламы из excel
     * NB: Сначала для групп и объявлений вызываются операции для валидации и если ошибок нет, то заново эти операции
     * вызываются для сохранения. То есть в режиме сохранения срабатывает дважды валидация операций. В будущем подумаем
     * как оптимизировать этот момент
     *
     * @param operatorUid          идентификатор оператора
     * @param uidAndClientId       идентификаторы клиента
     * @param validationOnly       флаг для определения что только валидируем или сохраняем данные
     * @param objectTypesForImport тип объектов, которые импортируем
     * @param excelFetchedData     данные полученные из excel-файла
     * @return результат импорта
     */
    public ExcelImportResult importFromExcel(long operatorUid, UidAndClientId uidAndClientId, boolean validationOnly,
                                             Set<ObjectType> objectTypesForImport, ExcelFetchedData excelFetchedData) {
        List<InternalAdGroupAddOrUpdateItem> adGroupAddOrUpdateItems =
                objectTypesForImport.contains(ObjectType.AD_GROUP)
                        ? toInternalAdGroupAddOrUpdateItems(excelFetchedData.getAdGroupsSheet())
                        : Collections.emptyList();
        List<List<Long>> originalAdGroupsGeo = mapList(adGroupAddOrUpdateItems, item -> item.getAdGroup().getGeo());

        List<InternalBannerAddOrUpdateItem> adAddOrUpdateItems =
                objectTypesForImport.contains(ObjectType.AD)
                        ? toInternalBannerAddOrUpdateItems(excelFetchedData.getAdsSheets(), adGroupAddOrUpdateItems)
                        : Collections.emptyList();


        var operationContainer =
                new InternalAdGroupOperationContainer(
                        Applicability.FULL, operatorUid,
                        uidAndClientId, SAVE_DRAFT,
                        RequestSource.EXCEL
                );

        MassResult<Long> adGroupsResult =
                addOrUpdateInternalAdGroupsService.addOrUpdateInternalAdGroups(adGroupAddOrUpdateItems,
                        operationContainer, true);

        MassResult<Long> adsResult = addOrUpdateInternalAdsService.addOrUpdateInternalAds(adAddOrUpdateItems,
                operatorUid, uidAndClientId.getClientId(), SAVE_DRAFT, true);

        if (!validationOnly && !hasValidationIssues(adGroupsResult) && !hasValidationIssues(adsResult)) {
            // проставим регионы из эксель файла, т.к. выше на этапе валидации add-операции могли изменить их
            // временный костыль, надеюсь придумаем как исправить тут: https://st.yandex-team.ru/DIRECT-130674
            StreamEx.of(adGroupAddOrUpdateItems)
                    .zipWith(originalAdGroupsGeo.stream())
                    .forKeyValue((item, originalGeo) -> item.getAdGroup().setGeo(originalGeo));

            adGroupsResult =
                    addOrUpdateInternalAdGroupsService.addOrUpdateInternalAdGroups(adGroupAddOrUpdateItems,
                            operationContainer, false);
            adsResult = addOrUpdateInternalAdsService.addOrUpdateInternalAds(adAddOrUpdateItems,
                    operatorUid, uidAndClientId.getClientId(), SAVE_DRAFT, false);
        }

        return new ExcelImportResult()
                .withAdGroupsResult(getMassResultWithAggregatedValidationResult(adGroupsResult))
                .withAdsResult(getMassResultWithAggregatedValidationResult(adsResult));
    }

    private static List<InternalAdGroupAddOrUpdateItem> toInternalAdGroupAddOrUpdateItems(
            ExcelSheetFetchedData<InternalAdGroupRepresentation> adGroupsSheet) {
        Long campaignId = adGroupsSheet.getSheetDescriptor().getCampaignId();

        return mapList(adGroupsSheet.getObjects(), representation ->
                new InternalAdGroupAddOrUpdateItem()
                        .withAdGroup(extractAndEnrichInternalAdGroup(representation, campaignId))
                        .withAdditionalTargetings(representation.getTargetingRepresentation().getTargetingList())
                        .withRetargetingCondition(representation.getRetargetingConditionRepresentation().getRetargetingCondition()));
    }

    /**
     * Извлекает InternalAdGroup и проставляет необходимые поля для операции
     */
    private static InternalAdGroup extractAndEnrichInternalAdGroup(InternalAdGroupRepresentation representation,
                                                                   Long campaignId) {
        return representation.getAdGroup()
                .withCampaignId(campaignId)
                .withType(AdGroupType.INTERNAL);
    }

    private static List<InternalBannerAddOrUpdateItem> toInternalBannerAddOrUpdateItems(
            List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets,
            List<InternalAdGroupAddOrUpdateItem> adGroupAddOrUpdateItems) {
        List<InternalAdGroup> adGroups = mapList(adGroupAddOrUpdateItems, InternalAdGroupAddOrUpdateItem::getAdGroup);
        Map<String, InternalAdGroup> adGroupByName = listToMap(adGroups, InternalAdGroup::getName);

        Map<Long, InternalAdGroup> adGroupById = StreamEx.of(adGroups)
                .removeBy(InternalAdGroup::getId, null)
                .toMap(InternalAdGroup::getId, Function.identity());

        return flatMap(adsSheets, sheet -> toInternalBannerAddOrUpdateItems(sheet, adGroupById, adGroupByName));
    }

    static List<InternalBannerAddOrUpdateItem> toInternalBannerAddOrUpdateItems(
            ExcelSheetFetchedData<InternalBannerRepresentation> adsSheet,
            Map<Long, InternalAdGroup> adGroupById,
            Map<String, InternalAdGroup> adGroupByName) {
        return mapList(adsSheet.getObjects(), representation -> {
            InternalBanner banner = extractAndEnrichInternalBanner(adsSheet.getSheetDescriptor(), representation);

            //Если у баннера нет id группы значит добавляем новый баннер в новую группу и привязка будет по имени группы
            InternalAdGroup adGroup = representation.getAdGroupId() != null
                    ? adGroupById.get(representation.getAdGroupId())
                    : adGroupByName.get(representation.getAdGroupName());

            return new InternalBannerAddOrUpdateItem()
                    .withBanner(banner)
                    .withAdGroup(adGroup);
        });
    }

    /**
     * Извлекает список InternalBanner и проставляет необходимые поля для операции
     */
    static InternalBanner extractAndEnrichInternalBanner(SheetDescriptor sheetDescriptor,
                                                         InternalBannerRepresentation representation) {
        return representation.getBanner()
                .withCampaignId(sheetDescriptor.getCampaignId())
                .withTemplateId(sheetDescriptor.getTemplateId());
    }

}
