package ru.yandex.direct.excel.processing.service.internalad;

import java.io.InputStream;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.SetUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.AdGroupAdditionalTargetingService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.service.ImageService;
import ru.yandex.direct.core.entity.image.service.ImageUtils;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceRestriction;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.excel.processing.exception.ExcelRuntimeException;
import ru.yandex.direct.excel.processing.exception.ExcelValidationException;
import ru.yandex.direct.excel.processing.model.CampaignInfoWithWorkbook;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdExportParameters;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdGroupRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.SheetDescriptor;
import ru.yandex.direct.excel.processing.model.internalad.mappers.AdGroupMappers;
import ru.yandex.direct.excel.processing.model.internalad.mappers.SheetDescriptorMapper;
import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.SheetRange;
import ru.yandex.direct.excelmapper.SheetRangeImpl;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds.Gen.CAMPAIGN_NOT_FOUND;
import static ru.yandex.direct.core.entity.internalads.Constants.isModeratedTemplate;
import static ru.yandex.direct.excel.processing.model.internalad.mappers.AdGroupMappers.getInternalAdGroupsMapper;
import static ru.yandex.direct.excel.processing.model.internalad.mappers.BannerMappers.getInternalBannersMapper;
import static ru.yandex.direct.excel.processing.model.internalad.mappers.BannerMappers.getInternalBannersMapperOnlyForRead;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdConverter.toAdGroupRepresentations;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdConverter.toBannerRepresentationsByTemplateId;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelUtils.createEmptyWorkbook;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelUtils.loadWorkbook;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelValidationService.validateExcelFetchedData;
import static ru.yandex.direct.excel.processing.utils.StyleUtils.getDefaultStyle;
import static ru.yandex.direct.excel.processing.utils.StyleUtils.getStyleForDescriptor;
import static ru.yandex.direct.excel.processing.utils.StyleUtils.getStyleForTitle;
import static ru.yandex.direct.excel.processing.utils.StyleUtils.setCellWidth;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.ADS_OR_AD_GROUPS_BELONG_TO_DIFFERENT_CAMPAIGNS;
import static ru.yandex.direct.excelmapper.ExcelMappers.maybeListMapper;
import static ru.yandex.direct.excelmapper.SheetRange.HeightMode.MAX_HEIGHT;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.result.DefectIds.OBJECT_NOT_FOUND;

@ParametersAreNonnullByDefault
public class InternalAdExcelService {

    private static final Logger logger = LoggerFactory.getLogger(InternalAdExcelService.class);

    public static final int MAX_AD_GROUP_COUNT = 10_000;
    private static final int TITLES_ROW = 5;
    static final int START_ROW_WITH_OBJECTS_DATA = TITLES_ROW + 1;
    private static final LimitOffset LIMIT_OFFSET = LimitOffset.limited(MAX_AD_GROUP_COUNT);
    private static final SheetDescriptorMapper GROUP_SHEET_DESCRIPTOR_MAPPER =
            new SheetDescriptorMapper(SheetDescriptorMapper.SheetType.AD_GROUP_LIST_SHEET);
    private static final SheetDescriptorMapper BANNER_SHEET_DESCRIPTOR_MAPPER =
            new SheetDescriptorMapper(SheetDescriptorMapper.SheetType.BANNER_SHEET);
    static final String AD_GROUP_SHEET_NAME = "Группы";
    static final String AD_SHEET_NAME_PREFIX = "Объявления с templateId=";
    private static final String REQUIRED_RESOURCE_SYMBOL = "*";

    private final CampaignService campaignService;
    private final AdGroupService adGroupService;
    private final AdGroupAdditionalTargetingService adGroupAdditionalTargetingService;
    private final BannerService bannerService;
    private final ImageService imageService;
    private final TemplateInfoService templateInfoService;
    private final ClientGeoService clientGeoService;
    private final CryptaSegmentDictionariesService cryptaSegmentDictionariesService;
    private final AddExcelValidationDataService addExcelValidationDataService;
    private final RetargetingConditionService retargetingConditionService;
    private final RetargetingService retargetingService;
    private final InternalAdExcelValidationService internalAdExcelValidationService;


    public InternalAdExcelService(CampaignService campaignService, AdGroupService adGroupService,
                                  AdGroupAdditionalTargetingService adGroupAdditionalTargetingService,
                                  BannerService bannerService, ImageService imageService,
                                  TemplateInfoService templateInfoService,
                                  ClientGeoService clientGeoService,
                                  CryptaSegmentDictionariesService cryptaSegmentDictionariesService,
                                  AddExcelValidationDataService addExcelValidationDataService,
                                  RetargetingConditionService retargetingConditionService,
                                  RetargetingService retargetingService,
                                  InternalAdExcelValidationService internalAdExcelValidationService) {
        this.campaignService = campaignService;
        this.adGroupService = adGroupService;
        this.adGroupAdditionalTargetingService = adGroupAdditionalTargetingService;
        this.bannerService = bannerService;
        this.imageService = imageService;
        this.templateInfoService = templateInfoService;
        this.clientGeoService = clientGeoService;
        this.cryptaSegmentDictionariesService = cryptaSegmentDictionariesService;
        this.addExcelValidationDataService = addExcelValidationDataService;
        this.retargetingConditionService = retargetingConditionService;
        this.retargetingService = retargetingService;
        this.internalAdExcelValidationService = internalAdExcelValidationService;
    }

    /**
     * Возвращает данные из эксель файла и проводит валидацию над ними
     * В валидации делаются проверки специфичные данным полученным из экселя
     */
    public Result<ExcelFetchedData> getAndValidateDataFromExcelFile(ClientId clientId,
                                                                    InputStream excelFile,
                                                                    Set<ObjectType> objectTypesForImport) {
        ExcelFetchedData dataFromExcelFile = getDataFromExcelFile(clientId, excelFile);
        ValidationResult<ExcelFetchedData, Defect> validationResult =
                validateExcelFetchedData(dataFromExcelFile, objectTypesForImport);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }

        return Result.successful(dataFromExcelFile);
    }

    public ExcelFetchedData getDataFromExcelFile(ClientId clientId, InputStream excelFileInputStream) {
        Workbook workbook = loadWorkbook(excelFileInputStream);
        var adGroupsSheet = readAdGroupSheet(clientId, workbook.getSheet(AD_GROUP_SHEET_NAME));

        List<ExcelSheetFetchedData<InternalBannerRepresentation>> allAdsSheets = StreamEx.of(workbook.sheetIterator())
                .remove(sheet -> AD_GROUP_SHEET_NAME.equals(sheet.getSheetName()))
                /*
                скрытые листы не читаем, т.к. экспортируем данные для валидации в скрытые листы:
                @see {@link AddExcelValidationDataService.createValidationSheetFor}
                 */
                .remove(sheet -> workbook.isSheetHidden(workbook.getSheetIndex(sheet)))
                .map(this::readBannersSheet)
                .toImmutableList();
        return ExcelFetchedData.create(adGroupsSheet, allAdsSheets);
    }

    private ExcelSheetFetchedData<InternalBannerRepresentation> readBannersSheet(Sheet bannersSheet) {
        SheetDescriptor sheetDescriptor =
                readSheetDescriptor(bannersSheet, SheetDescriptorMapper.SheetType.BANNER_SHEET);
        var mapper = createInternalBannerListMapper(sheetDescriptor.getTemplateId());
        List<String> columnTitles = readColumnTitles(bannersSheet, mapper.getMeta());
        List<InternalBannerRepresentation> adsRepresentations =
                readInternalBannerRepresentations(bannersSheet, mapper);
        return ExcelSheetFetchedData.create(bannersSheet.getSheetName(), sheetDescriptor, adsRepresentations,
                columnTitles, mapper.getMeta().getColumns());
    }

    private static List<String> readColumnTitles(Sheet sheet, MapperMeta mapperMeta) {
        var sheetRange = new SheetRangeImpl(sheet, TITLES_ROW, 0, SheetRange.HeightMode.AUTODETECT_HEIGHT);
        return EntryStream.of(mapperMeta.getColumns())
                .mapKeyValue((column, title) -> MapperUtils
                        .tryReadStringCellValueOrReportCantRead(sheetRange, 0, column, mapperMeta.getColumns()))
                // Названия ресурсов, заканчивающиеся на '*' (обязательные ресурсы) - зааменяются на оригинальные
                // названия (без '*')
                .map(title -> title.endsWith(REQUIRED_RESOURCE_SYMBOL) ? title.substring(0, title.length() - 1) : title)
                .toImmutableList();
    }

    private List<InternalBannerRepresentation> readInternalBannerRepresentations(
            Sheet bannersSheet,
            ExcelMapper<List<InternalBannerRepresentation>> internalBannersMapper) {
        SheetRangeImpl sheetRange = new SheetRangeImpl(bannersSheet, START_ROW_WITH_OBJECTS_DATA, 0,
                SheetRange.HeightMode.AUTODETECT_HEIGHT);
        return internalBannersMapper.read(sheetRange).getValue();
    }

    private ExcelMapper<List<InternalBannerRepresentation>> createInternalBannerListMapper(Long templateId) {
        boolean isModerated = isModeratedTemplate(templateId);
        List<ResourceInfo> templateResources = Optional.ofNullable(templateInfoService.getByTemplateId(templateId))
                .map(InternalTemplateInfo::getResources)
                .orElse(emptyList());
        return maybeListMapper(getInternalBannersMapperOnlyForRead(templateResources, isModerated), true);
    }

    private ExcelSheetFetchedData<InternalAdGroupRepresentation> readAdGroupSheet(ClientId clientId,
                                                                                  Sheet adGroupListSheet) {
        SheetDescriptor sheetDescriptor = readSheetDescriptor(adGroupListSheet,
                SheetDescriptorMapper.SheetType.AD_GROUP_LIST_SHEET);
        var mapper = createInternalAdGroupListMapper(clientId, sheetDescriptor.getCampaignId());
        List<String> columnTitles = readColumnTitles(adGroupListSheet, mapper.getMeta());
        var adGroupsRepresentation = readInternalAdGroupRepresentations(adGroupListSheet, mapper);
        return ExcelSheetFetchedData.create(adGroupListSheet.getSheetName(), sheetDescriptor, adGroupsRepresentation,
                columnTitles, mapper.getMeta().getColumns());
    }

    private ExcelMapper<List<InternalAdGroupRepresentation>> createInternalAdGroupListMapper(ClientId clientId,
                                                                                             Long campaignId) {
        Campaign campaign = retrieveCampaign(clientId, campaignId);
        return maybeListMapper(
                AdGroupMappers.getInternalAdGroupsMapper(campaign.getType(), cryptaSegmentDictionariesService),
                true);
    }

    private List<InternalAdGroupRepresentation> readInternalAdGroupRepresentations(
            Sheet adGroupListSheet,
            ExcelMapper<List<InternalAdGroupRepresentation>> mapper) {
        SheetRangeImpl sheetRange = new SheetRangeImpl(adGroupListSheet, START_ROW_WITH_OBJECTS_DATA, 0,
                SheetRange.HeightMode.AUTODETECT_HEIGHT);
        return mapper.read(sheetRange).getValue();
    }

    Campaign retrieveCampaign(ClientId clientId, Long campaignId) {
        List<Campaign> campaigns = campaignService.getCampaigns(clientId, Set.of(campaignId));
        if (campaigns.isEmpty()) {
            throw ExcelValidationException.create(CAMPAIGN_NOT_FOUND);
        }

        return campaigns.get(0);
    }

    private SheetDescriptor readSheetDescriptor(Sheet sheet, SheetDescriptorMapper.SheetType sheetType) {
        var sheetRange = new SheetRangeImpl(sheet, 0, 0, SheetRange.HeightMode.AUTODETECT_HEIGHT);
        return (sheetType == SheetDescriptorMapper.SheetType.AD_GROUP_LIST_SHEET
                ? GROUP_SHEET_DESCRIPTOR_MAPPER
                : BANNER_SHEET_DESCRIPTOR_MAPPER).read(sheetRange).getValue();
    }

    public CampaignInfoWithWorkbook createWorkbook(ClientId clientId, ExcelFetchedData excelFetchedData,
                                                   boolean hideEmptyColumns) {
        Long campaignId = excelFetchedData.getAdGroupsSheet().getSheetDescriptor().getCampaignId();
        Long placeId = excelFetchedData.getAdGroupsSheet().getSheetDescriptor().getPlaceId();
        Campaign campaign = campaignService.getCampaigns(clientId, Set.of(campaignId)).get(0);

        Workbook workbook = createEmptyWorkbook();
        exportAdGroups(clientId, workbook, campaign, placeId, excelFetchedData.getAdGroupsSheet().getObjects(),
                hideEmptyColumns);

        if (CollectionUtils.isNotEmpty(excelFetchedData.getAdsSheets())) {
            Map<Long, List<InternalBannerRepresentation>> bannerRepresentationsByTemplateId =
                    listToMap(excelFetchedData.getAdsSheets(),
                            sheet -> sheet.getSheetDescriptor().getTemplateId(),
                            ExcelSheetFetchedData::getObjects);

            exportBanners(workbook, clientId, campaignId, placeId, bannerRepresentationsByTemplateId);
        }

        return new CampaignInfoWithWorkbook(campaign.getId(), campaign.getName(), workbook);
    }

    public CampaignInfoWithWorkbook createWorkbook(InternalAdExportParameters exportParameters) {
        List<InternalAdGroup> adGroups = getInternalAdGroups(exportParameters);
        Set<Long> campaignIds = listToSet(adGroups, InternalAdGroup::getCampaignId);
        if (campaignIds.size() > 1) {
            throw ExcelValidationException.create(ADS_OR_AD_GROUPS_BELONG_TO_DIFFERENT_CAMPAIGNS);
        } else if (adGroups.isEmpty() && isEmpty(exportParameters.getCampaignIds())) {
            logger.warn("not found any objects by exportParameters");
            throw ExcelValidationException.create(OBJECT_NOT_FOUND);
        }

        /*
        Если кампания пустая, берем id из запроса.
        Для пустой кампании нужно сделать пустую выгрузку с полями групп для дальнейшего заполнения.
         */
        //noinspection ConstantConditions
        long campaignId = campaignIds.isEmpty()
                ? exportParameters.getCampaignIds().iterator().next()
                : campaignIds.iterator().next();
        ClientId clientId = exportParameters.getClientId();

        Map<Long, Long> placeIdByCampaignId = campaignService.getCampaignInternalPlaces(clientId, Set.of(campaignId));
        if (!placeIdByCampaignId.containsKey(campaignId)) {
            logger.warn("campaign {} not found", campaignId);
            throw ExcelValidationException.create(CAMPAIGN_NOT_FOUND);
        }

        Long placeId = placeIdByCampaignId.get(campaignId);
        Set<Long> adGroupIds = listToSet(adGroups, InternalAdGroup::getId);
        Campaign campaign = campaignService.getCampaigns(clientId, List.of(campaignId)).get(0);
        var retargetingConditionMapByAdGroupId = getRetargetingConditionMapByAdGroupIds(adGroupIds, clientId);
        var targetingsByAdGroup = adGroupAdditionalTargetingService.getTargetingMapByAdGroupIds(clientId, adGroupIds);
        List<InternalAdGroupRepresentation> adGroupRepresentations =
                toAdGroupRepresentations(adGroups, targetingsByAdGroup, cryptaSegmentDictionariesService,
                        retargetingConditionMapByAdGroupId);

        InternalAdExcelValidationService.validateAdGroupRepresentations(adGroupRepresentations);

        var hideEmptyColumns = exportParameters.isHideEmptyColumns();

        Workbook workbook = createEmptyWorkbook();
        exportAdGroups(clientId, workbook, campaign, placeId, adGroupRepresentations, hideEmptyColumns);

        if (exportParameters.isExportAdGroupsWithAds()) {
            List<InternalBanner> banners = getInternalBanners(exportParameters.getAdIds(), adGroupIds);
            var bannerRepresentationsByTemplateId = toBannerRepresentationsByTemplateId(adGroups, banners);

            exportBanners(workbook, clientId, campaignId, placeId, bannerRepresentationsByTemplateId);
        }

        return new CampaignInfoWithWorkbook(campaign.getId(), campaign.getName(), workbook);
    }

    public CampaignInfoWithWorkbook createWorkbookForEmptyCampaign(ClientId clientId, Long campaignId, Long placeId,
                                                                   Long templateId) {
        internalAdExcelValidationService.validateParamsForTemplateExport(clientId, campaignId, placeId, templateId);

        Campaign campaign = campaignService.getCampaigns(clientId, List.of(campaignId)).get(0);

        Workbook workbook = createEmptyWorkbook();
        exportAdGroups(clientId, workbook, campaign, placeId, emptyList(), false);
        exportEmptyBannerTemplate(workbook, campaignId, placeId, templateId);

        return new CampaignInfoWithWorkbook(campaignId, campaign.getName(), workbook);
    }

    private List<InternalAdGroup> getInternalAdGroups(InternalAdExportParameters exportParameters) {
        var adGroupsSelectionCriteria = new AdGroupsSelectionCriteria();
        if (isNotEmpty(exportParameters.getCampaignIds())) {
            adGroupsSelectionCriteria.setCampaignIds(exportParameters.getCampaignIds());
        } else if (isNotEmpty(exportParameters.getAdGroupIds())) {
            adGroupsSelectionCriteria.setAdGroupIds(exportParameters.getAdGroupIds());
        } else if (isNotEmpty(exportParameters.getAdIds())) {
            Map<Long, Long> adGroupIdsByBannerIds = bannerService
                    .getAdGroupIdsByBannerIds(exportParameters.getClientId(), exportParameters.getAdIds());
            Set<Long> adGroupIds = Set.copyOf(adGroupIdsByBannerIds.values());
            adGroupsSelectionCriteria.setAdGroupIds(adGroupIds);
        } else {
            // не должны сюда попасть, т.к. ожидается что в exportParameters будет заполнено одно из полей с id'шниками
            throw new ExcelRuntimeException("not found ids in exportParameters for adGroupsSelectionCriteria");
        }

        List<AdGroup> adGroups = adGroupService.getAdGroupsBySelectionCriteria(
                adGroupsSelectionCriteria, LIMIT_OFFSET, false);

        return StreamEx.of(adGroups)
                .select(InternalAdGroup.class)
                .toImmutableList();
    }

    private Map<Long, RetargetingCondition> getRetargetingConditionMapByAdGroupIds(Set<Long> adGroupIds,
                                                                                   ClientId clientId) {
        var retargetingByAdGroupIds = retargetingService.getRetargetingByAdGroupIds(adGroupIds, clientId);
        var retargetinConditionsByAdGroupIds =
                retargetingConditionService.getRetargetingConditionsByAdGroupIds(clientId, adGroupIds);
        var retargetinConditionMapById = StreamEx.of(retargetinConditionsByAdGroupIds)
                .mapToEntry(RetargetingCondition::getId, identity())
                .grouping();
        return StreamEx.of(retargetingByAdGroupIds)
                .mapToEntry(Retargeting::getAdGroupId,
                        r -> retargetinConditionMapById.get(r.getRetargetingConditionId()).get(0))
                .toMap();
    }

    private void enrichAdGroupsDataForExport(ClientId clientId,
                                             List<InternalAdGroupRepresentation> adGroupRepresentations) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        StreamEx.of(adGroupRepresentations)
                .map(InternalAdGroupRepresentation::getAdGroup)
                .forEach(adGroup -> {
                    List<Long> convertedRegionIds = clientGeoService.convertForWeb(adGroup.getGeo(), geoTree);
                    adGroup.setGeo(convertedRegionIds);
                });
    }

    private List<InternalBanner> getInternalBanners(@Nullable Set<Long> adIds, Set<Long> adGroupIds) {
        if (isNotEmpty(adIds)) {
            List<BannerWithSystemFields> banners = bannerService.getBannersByIds(adIds);
            return StreamEx.of(banners)
                    .select(InternalBanner.class)
                    .toImmutableList();
        } else {
            Map<Long, List<BannerWithSystemFields>> bannersByAdGroups =
                    bannerService.getBannersByAdGroupIds(adGroupIds);
            return StreamEx.of(bannersByAdGroups.values())
                    .flatMap(Collection::stream)
                    .select(InternalBanner.class)
                    .toImmutableList();
        }
    }

    private Map<String, String> getImageUrlByHashes(ClientId clientId,
                                                    List<InternalBanner> internalBanners,
                                                    Map<Long, List<ResourceInfo>> templateResourcesByTemplateId) {
        Map<Long, ResourceInfo> templateResourcesById = StreamEx.ofValues(templateResourcesByTemplateId)
                .flatMap(Collection::stream)
                .mapToEntry(ResourceInfo::getId, identity())
                .toMap();
        Set<String> imageHashes = StreamEx.of(internalBanners)
                .map(InternalBanner::getTemplateVariables)
                .flatMap(Collection::stream)
                .filter(variable -> templateResourcesById.containsKey(variable.getTemplateResourceId())
                        && templateResourcesById.get(variable.getTemplateResourceId()).getType() == ResourceType.IMAGE)
                .map(TemplateVariable::getInternalValue)
                .nonNull()
                .toSet();

        List<BannerImageFormat> bannerImageFormats = imageService.getBannerImageFormats(clientId, imageHashes);

        return StreamEx.of(bannerImageFormats)
                .mapToEntry(BannerImageFormat::getImageHash, identity())
                .mapValues(ImageUtils::generateOrigImageUrl)
                .toImmutableMap();
    }

    private void exportAdGroups(ClientId clientId, Workbook workbook, Campaign campaign, long placeId,
                                List<InternalAdGroupRepresentation> adGroupRepresentations, boolean hideEmptyColumns) {
        enrichAdGroupsDataForExport(clientId, adGroupRepresentations);

        Sheet sheet = workbook.createSheet(AD_GROUP_SHEET_NAME);
        var sheetRange = new SheetRangeImpl(sheet, 0, 0, MAX_HEIGHT);

        // write descriptor
        CellStyle descriptorStyle = getStyleForDescriptor(workbook);
        var sheetDescriptor = new SheetDescriptor()
                .setCampaignId(campaign.getId())
                .setPlaceId(placeId);
        GROUP_SHEET_DESCRIPTOR_MAPPER.write(new StyledSheetRange(sheetRange, descriptorStyle), sheetDescriptor);

        // write titles
        CellStyle titleStyle = getStyleForTitle(workbook);
        ExcelMapper<InternalAdGroupRepresentation> adGroupsMapper =
                getInternalAdGroupsMapper(campaign.getType(), cryptaSegmentDictionariesService);
        List<String> columnTitles = adGroupsMapper.getMeta().getColumns();
        Set<String> requiredColumnTitles = adGroupsMapper.getMeta().getRequiredColumns();
        var titleSheetRange = new StyledSheetRange(sheetRange.makeSubRange(TITLES_ROW, 0), titleStyle);
        EntryStream.of(columnTitles)
                .mapValues(title -> requiredColumnTitles.contains(title) ? title + REQUIRED_RESOURCE_SYMBOL : title)
                .forKeyValue((idx, title) -> titleSheetRange.getCellForWrite(0, idx).setCellValue(title));

        // write groups
        CellStyle defaultStyle = getDefaultStyle(workbook);
        var dataSheetRange = new StyledSheetRange(
                sheetRange.makeSubRange(START_ROW_WITH_OBJECTS_DATA, 0), defaultStyle);
        ExcelMapper<List<InternalAdGroupRepresentation>> adGroupsListMapper =
                maybeListMapper(adGroupsMapper, true);
        adGroupsListMapper.write(dataSheetRange, adGroupRepresentations);
        EntryStream.of(columnTitles).forKeyValue((colIdx, title) -> setCellWidth(sheet, colIdx));

        if (hideEmptyColumns) {
            hideEmptyColumns(sheet, dataSheetRange, columnTitles,
                    Sets.union(requiredColumnTitles, AdGroupMappers.NON_HIDDEN_TITLES));
        }

        addExcelValidationDataService.addAdGroupsValidationData(sheet, START_ROW_WITH_OBJECTS_DATA, columnTitles);
    }

    /**
     * Выгрузить пустой шаблон баннера - со всеми названиями колонок (для переданного шаблона) и незаполненными полями
     * самого баннера
     */
    private void exportEmptyBannerTemplate(Workbook workbook, Long campaignId, Long placeId, Long templateId) {
        InternalTemplateInfo templateInfo = templateInfoService.getByTemplateId(templateId);
        List<ResourceInfo> templateResources = Objects.requireNonNull(templateInfo).getResources();
        Set<Long> requiredResources = getRequiredResourceIds(templateInfo);

        exportBannersPerSheet(workbook, campaignId, placeId, templateId, emptyList(), templateResources, emptyMap(),
                requiredResources);
    }

    private void exportBanners(Workbook workbook, ClientId clientId, Long campaignId, Long placeId,
                               Map<Long, List<InternalBannerRepresentation>> bannerRepresentationsByTemplateId) {
        List<InternalTemplateInfo> templateInfos =
                templateInfoService.getByTemplateIds(bannerRepresentationsByTemplateId.keySet());
        Map<Long, List<ResourceInfo>> templateResourcesByTemplateId =
                listToMap(templateInfos, InternalTemplateInfo::getTemplateId, InternalTemplateInfo::getResources);
        Map<Long, Set<Long>> requiredResourcesByTemplateId = listToMap(
                templateInfos, InternalTemplateInfo::getTemplateId, InternalAdExcelService::getRequiredResourceIds);

        List<InternalBanner> banners = StreamEx.ofValues(bannerRepresentationsByTemplateId)
                .flatMap(Collection::stream)
                .map(InternalBannerRepresentation::getBanner)
                .toList();
        Map<String, String> imageUrlByHashes =
                getImageUrlByHashes(clientId, banners, templateResourcesByTemplateId);

        bannerRepresentationsByTemplateId.forEach((templateId, internalBanners) ->
                exportBannersPerSheet(workbook, campaignId, placeId, templateId, internalBanners,
                        templateResourcesByTemplateId.getOrDefault(templateId, emptyList()), imageUrlByHashes,
                        requiredResourcesByTemplateId.getOrDefault(templateId, emptySet()))
        );
    }

    /**
     * Получить обязательные к заполнению id ресурсов. Определяется на основе {@link ResourceRestriction},
     * переопределенные в {@code TemplateInfoOverrides},
     * поэтому обязательными могут быть ресурсы у которых в базе этого свойства нет
     */
    private static Set<Long> getRequiredResourceIds(InternalTemplateInfo templateInfo) {
        return StreamEx.of(templateInfo.getResourceRestrictions())
                .map(ResourceRestriction::getRequired)
                .reduce(SetUtils::intersection)
                .orElse(emptySet());
    }

    /**
     * Записывает на отдельный лист баннеры сгруппирированные по шаблону
     */
    private void exportBannersPerSheet(Workbook workbook,
                                       long campaignId, long placeId, long templateId,
                                       List<InternalBannerRepresentation> internalBannerRepresentations,
                                       List<ResourceInfo> templateResources,
                                       Map<String, String> imageUrlByHashes,
                                       Set<Long> requiredResourceIds) {
        Sheet sheet = workbook.createSheet(AD_SHEET_NAME_PREFIX + templateId);
        var sheetRange = new SheetRangeImpl(sheet, 0, 0, MAX_HEIGHT);

        // write descriptor
        CellStyle descriptorStyle = getStyleForDescriptor(workbook);
        var sheetDescriptor = new SheetDescriptor()
                .setCampaignId(campaignId)
                .setPlaceId(placeId)
                .setTemplateId(templateId);
        BANNER_SHEET_DESCRIPTOR_MAPPER.write(new StyledSheetRange(sheetRange, descriptorStyle), sheetDescriptor);

        boolean isModerated = isModeratedTemplate(templateId);
        ExcelMapper<InternalBannerRepresentation> internalBannersMapper =
                getInternalBannersMapper(templateResources, imageUrlByHashes, isModerated);

        Set<String> requiredColumnTitles = getRequiredBannerColumnTitles(
                requiredResourceIds, templateResources, internalBannersMapper);

        List<String> columnTitles = internalBannersMapper.getMeta().getColumns();
        writeBannerTitles(sheet, sheetRange, columnTitles, templateResources,
                // Показываем все поля шаблона баннера, в том числе скрытые
                false,
                requiredColumnTitles);

        List<InternalBannerRepresentation> sortedInternalBannerRepresentations =
                StreamEx.of(internalBannerRepresentations)
                        .sorted(Comparator.comparing(r -> r.getBanner().getId(), Comparator.nullsLast(Long::compareTo)))
                        .toList();

        // write banners
        CellStyle defaultStyle = getDefaultStyle(workbook);
        var dataSheetRange = new StyledSheetRange(
                sheetRange.makeSubRange(START_ROW_WITH_OBJECTS_DATA, 0), defaultStyle);
        ExcelMapper<List<InternalBannerRepresentation>> internalBannersListMapper =
                maybeListMapper(internalBannersMapper, true);
        internalBannersListMapper.write(dataSheetRange, sortedInternalBannerRepresentations);
        EntryStream.of(columnTitles).forKeyValue((colIdx, title) -> setCellWidth(sheet, colIdx));

        addExcelValidationDataService.addBannersValidationData(sheet, START_ROW_WITH_OBJECTS_DATA,
                columnTitles, templateId, isModerated);
    }

    private static Set<String> getRequiredBannerColumnTitles(Set<Long> requiredResourceIds,
                                                             List<ResourceInfo> templateResources,
                                                             ExcelMapper<InternalBannerRepresentation> internalBannersMapper) {
        return StreamEx.of(templateResources)
                .filter(resourceInfo -> requiredResourceIds.contains(resourceInfo.getId()))
                .map(ResourceInfo::getLabel)
                .append(internalBannersMapper.getMeta().getRequiredColumns())
                .toSet();
    }

    /**
     * Записывает заголовки столбцов для баннеров.
     * Если не требуется показывать все заголовки баннера, то для скрытых ресурсов помечает столбец как скрытый.
     * Обязательные ресурсы помечаются '*'
     */
    private static void writeBannerTitles(Sheet sheet, SheetRange sheetRange,
                                          List<String> columnTitles,
                                          List<ResourceInfo> templateResources,
                                          boolean hideHiddenResources,
                                          Set<String> requiredColumnTitles) {
        Map<String, Boolean> isResourceHiddenByTitle = listToMap(
                templateResources,
                ResourceInfo::getLabel,
                resourceInfo -> hideHiddenResources && resourceInfo.isHidden()
        );
        CellStyle titleStyle = getStyleForTitle(sheet.getWorkbook());

        var titleSheetRange = new StyledSheetRange(sheetRange.makeSubRange(TITLES_ROW, 0), titleStyle);
        EntryStream.of(columnTitles)
                .forKeyValue((idx, title) -> {
                    Cell cell = titleSheetRange.getCell(0, idx);
                    if (requiredColumnTitles.contains(title)) {
                        title = title + REQUIRED_RESOURCE_SYMBOL;
                    }
                    cell.setCellValue(title);
                    sheet.setColumnHidden(idx, isResourceHiddenByTitle.getOrDefault(title, false));
                });
    }

    private static void hideEmptyColumns(Sheet sheet, SheetRange sheetRange,
                                         List<String> columnTitles, Set<String> requiredColumnTitles) {
        var columnsWithData = sheetRange.getColumnsWithNotEmptyValue();
        EntryStream.of(columnTitles)
                .forKeyValue((idx, title) -> {
                    if (!requiredColumnTitles.contains(title) && !columnsWithData.contains(idx)) {
                        sheet.setColumnHidden(idx, true);
                    }
                });
    }

}
