package ru.yandex.direct.excel.processing.service.internalad;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.core.entity.internalads.service.TemplatePlaceService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.excel.processing.exception.ExcelValidationException;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdGroupRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excel.processing.model.validation.AdGroupIdsParams;
import ru.yandex.direct.excel.processing.service.internalad.validation.AdSheetFetchedDataValidator;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static org.apache.commons.collections4.CollectionUtils.isEqualCollection;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds.Gen.CAMPAIGN_NOT_FOUND;
import static ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefectIds.INVALID_PLACE_ID;
import static ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefectIds.TEMPLATE_NOT_FOUND;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.METRIKA_COUNTER_UPPER_BOUND;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.GOAL;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.unsupportedGoalId;
import static ru.yandex.direct.excel.processing.service.internalad.InternalAdExcelUtils.getCampaignIdsFromAllSheetDescriptors;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.adsOrAdGroupsBelongToDifferentCampaigns;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.duplicateAdGroupTitles;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.inconsistentColumnTitles;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

@ParametersAreNonnullByDefault
public class InternalAdExcelValidationService {

    private static final Logger logger = LoggerFactory.getLogger(InternalAdExcelValidationService.class);

    private static final String TITLE_PATH = "title";

    private final CampaignService campaignService;
    private final TemplatePlaceService templatePlaceService;
    private final TemplateInfoService templateInfoService;

    public InternalAdExcelValidationService(CampaignService campaignService,
                                            TemplatePlaceService templatePlaceService,
                                            TemplateInfoService templateInfoService) {
        this.campaignService = campaignService;
        this.templatePlaceService = templatePlaceService;
        this.templateInfoService = templateInfoService;
    }

    public void validateParamsForTemplateExport(ClientId clientId, Long campaignId, Long placeId, Long templateId) {
        Map<Long, Long> campaignPlaces = campaignService.getCampaignInternalPlaces(clientId, Set.of(campaignId));
        if (!campaignPlaces.containsKey(campaignId)) {
            logger.warn("campaign {} not found", campaignId);
            throw ExcelValidationException.create(CAMPAIGN_NOT_FOUND);
        } else if (!placeId.equals(campaignPlaces.get(campaignId))) {
            logger.warn("wrong place id {}. Actual place id is {} for campaign {}",
                    placeId, campaignPlaces.get(campaignId), campaignId);
            throw ExcelValidationException.create(INVALID_PLACE_ID);
        }

        List<TemplatePlace> templatesByPlaceId = templatePlaceService.getVisibleTemplatesByPlaceIds(Set.of(placeId));
        Set<Long> actualTemplateIds = listToSet(templatesByPlaceId, TemplatePlace::getTemplateId);
        InternalTemplateInfo templateInfo = templateInfoService.getByTemplateId(templateId);
        if (templateInfo == null || !actualTemplateIds.contains(templateId)) {
            logger.warn("template id {} for place {} not found", templateId, placeId);
            throw ExcelValidationException.create(TEMPLATE_NOT_FOUND);
        }
    }

    static ValidationResult<ExcelFetchedData, Defect> validateExcelFetchedData(ExcelFetchedData excelFetchedData,
                                                                               Set<ObjectType> objectTypesForImport) {
        var vb = ItemValidationBuilder.of(excelFetchedData, Defect.class);

        Set<Long> campaignIdsFromSheetDescriptors = getCampaignIdsFromAllSheetDescriptors(excelFetchedData);
        vb.check(checkSameCampaignIds(campaignIdsFromSheetDescriptors));
        validateAdGroups(vb, excelFetchedData.getAdGroupsSheet());
        validateAdsSheets(vb, excelFetchedData.getAdsSheets(),
                excelFetchedData.getAdGroupsSheet().getObjects(), objectTypesForImport);

        return vb.getResult();
    }

    //TODO-perezhoginnik возвращаем 1 дефект с максимальным количеством adGroupId в которых он сработал
    static void validateAdGroupRepresentations(List<InternalAdGroupRepresentation> adGroupRepresentations) {
        var result = new HashMap<DefectId, AdGroupIdsParams>();
        for (var adGroupRepresentation : adGroupRepresentations) {
            try {
                adGroupRepresentation.getRetargetingConditionRepresentation().getGoalContext();
                adGroupRepresentation.getRetargetingConditionRepresentation().getAudience();
                adGroupRepresentation.getRetargetingConditionRepresentation().getAudienceNot();
                adGroupRepresentation.getRetargetingConditionRepresentation().getSocialDemoGender();
                adGroupRepresentation.getRetargetingConditionRepresentation().getSocialDemoAge();
                adGroupRepresentation.getRetargetingConditionRepresentation().getSocialDemoIncome();
                adGroupRepresentation.getRetargetingConditionRepresentation().getCrypta();
                adGroupRepresentation.getRetargetingConditionRepresentation().getCryptaNot();
            } catch (ExcelValidationException e) {
                result.putIfAbsent(e.getDefectId(), new AdGroupIdsParams());
                result.get(e.getDefectId()).getAdGroupIds().add(adGroupRepresentation.getAdGroup().getId());
            }
        }
        if (!result.isEmpty()) {
            var defectId = Collections.max(result.entrySet(),
                    Comparator.comparingInt(entry -> entry.getValue().getAdGroupIds().size())).getKey();
            throw ExcelValidationException.create(defectId, result.get(defectId));
        }
    }

    private static void validateAdGroups(ItemValidationBuilder<ExcelFetchedData, Defect> vb,
                                         ExcelSheetFetchedData<InternalAdGroupRepresentation> adGroupsSheet) {
        vb.list(getRetargetingConditions(adGroupsSheet), ExcelFetchedData.RETARGETING_CONDITIONS)
                .checkEachBy(InternalAdExcelValidationService::validateRetCondition, When.notNull());

        var adGroupVb = vb.item(adGroupsSheet, ExcelFetchedData.AD_GROUPS_PATH);

        adGroupVb.check(checkAdGroupsColumnTitles());
        adGroupVb.list(adGroupsSheet.getObjects(), TITLE_PATH)
                .checkEach(unique(representation -> representation.getAdGroup().getName()), duplicateAdGroupTitles());
    }

    private static void validateAdsSheets(ItemValidationBuilder<ExcelFetchedData, Defect> vb,
                                          List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets,
                                          List<InternalAdGroupRepresentation> adGroupRepresentations,
                                          Set<ObjectType> objectTypesForImport) {
        Set<Long> existingAdGroupIds = mapAndFilterToSet(adGroupRepresentations,
                representation -> representation.getAdGroup().getId(), Objects::nonNull);
        Set<String> namesForNewAdGroups = filterAndMapToSet(adGroupRepresentations,
                representation -> representation.getAdGroup().getId() == null,
                representation -> representation.getAdGroup().getName());
        vb.list(adsSheets, ExcelFetchedData.ADS_PATH)
                .checkEach(checkAdsColumnTitles())
                .checkEachBy(new AdSheetFetchedDataValidator(existingAdGroupIds, namesForNewAdGroups,
                        objectTypesForImport));
    }

    private static Constraint<ExcelSheetFetchedData<InternalAdGroupRepresentation>, Defect> checkAdGroupsColumnTitles() {
        return fetchedData ->
                isEqualCollection(fetchedData.getFetchedColumnTitles(), fetchedData.getMappersColumnTitles())
                        ? null : inconsistentColumnTitles(fetchedData);
    }

    private static Constraint<ExcelSheetFetchedData<InternalBannerRepresentation>, Defect> checkAdsColumnTitles() {
        return fetchedData ->
                isEqualCollection(fetchedData.getFetchedColumnTitles(), fetchedData.getMappersColumnTitles())
                        ? null : inconsistentColumnTitles(fetchedData);
    }

    private static List<RetargetingCondition> getRetargetingConditions(ExcelSheetFetchedData<InternalAdGroupRepresentation> adGroupsSheet) {
        return mapList(adGroupsSheet.getObjects(),
                representation -> representation.getRetargetingConditionRepresentation().getRetargetingCondition());
    }

    private static Constraint<ExcelFetchedData, Defect> checkSameCampaignIds(Set<Long> campaignIds) {
        return Constraint.fromPredicate(excelFetchedData -> campaignIds.size() == 1,
                adsOrAdGroupsBelongToDifferentCampaigns());
    }

    private static ValidationResult<RetargetingCondition, Defect> validateRetCondition(
            RetargetingCondition condition) {
        ModelItemValidationBuilder<RetargetingCondition> vb = ModelItemValidationBuilder.of(condition);
        vb.list(RetargetingCondition.RULES)
                .checkBy(InternalAdExcelValidationService::validateRetConditionRules);
        return vb.getResult();
    }


    private static ValidationResult<List<Rule>, Defect> validateRetConditionRules(List<Rule> rules) {
        ListValidationBuilder<Rule, Defect> vb =
                ListValidationBuilder.<Rule, Defect>of(rules)
                        .checkEachBy(InternalAdExcelValidationService::validateRetConditionRule, When.notNull());
        return vb.getResult();
    }

    private static ValidationResult<Rule, Defect> validateRetConditionRule(
            Rule rule) {
        ModelItemValidationBuilder<Rule> vb = ModelItemValidationBuilder.of(rule);
        vb.list(Rule.GOALS).check(unsupportedGoalIdInRetargetingCondition(), When.isValid());
        return vb.getResult();
    }

    private static Constraint<List<Goal>, Defect> unsupportedGoalIdInRetargetingCondition() {
        return Constraint
                .fromPredicate(g -> g.stream()
                                .filter(Objects::nonNull)
                                .filter(goal -> goal.getType() == GOAL)
                                .noneMatch(goal -> goal.getId() > METRIKA_COUNTER_UPPER_BOUND),
                        unsupportedGoalId());
    }

}
