package ru.yandex.direct.excel.processing.service.internalad.validation;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.excel.processing.validation.defects.Defects.adGroupByIdNotFound;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.adGroupIdRequiredForExistingAd;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.newAdGroupByNameNotFound;
import static ru.yandex.direct.excel.processing.validation.defects.Defects.notAllowedAddNewAdGroupWithNewAdForImportMode;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@ParametersAreNonnullByDefault
public class AdSheetFetchedDataValidator
        implements Validator<ExcelSheetFetchedData<InternalBannerRepresentation>, Defect> {

    public static final String AD_PATH = "ad";
    private final Set<Long> existingAdGroupIds;
    private final Set<String> namesForNewAdGroups;
    private final Set<ObjectType> objectTypesForImport;

    public AdSheetFetchedDataValidator(Set<Long> existingAdGroupIds, Set<String> namesForNewAdGroups,
                                       Set<ObjectType> objectTypesForImport) {
        this.existingAdGroupIds = existingAdGroupIds;
        this.namesForNewAdGroups = namesForNewAdGroups;
        this.objectTypesForImport = objectTypesForImport;
    }

    @Override
    public ValidationResult<ExcelSheetFetchedData<InternalBannerRepresentation>, Defect> apply(
            ExcelSheetFetchedData<InternalBannerRepresentation> fetchedData) {
        var vb = ItemValidationBuilder.of(fetchedData, Defect.class);
        vb.list(fetchedData.getObjects(), AD_PATH)
                .checkEach(notNull())
                .checkEachBy(banner -> validateAd(banner, fetchedData.getSheetName()), When.isValid());
        return vb.getResult();
    }

    private ValidationResult<InternalBannerRepresentation, Defect> validateAd(InternalBannerRepresentation banner,
                                                                              String sheetName) {
        return ItemValidationBuilder.of(banner, Defect.class)
                .check(validateConnectionsWithAdGroup(sheetName))
                .getResult();
    }

    private Constraint<InternalBannerRepresentation, Defect> validateConnectionsWithAdGroup(String sheetName) {
        return representation -> {
            InternalBanner banner = representation.getBanner();
            if (representation.getAdGroupId() == null && banner.getId() != null) {
                return adGroupIdRequiredForExistingAd(sheetName, banner.getId());
            }

            //новая группа с новым баннером
            if (representation.getAdGroupId() == null && banner.getId() == null) {
                if (!objectTypesForImport.contains(ObjectType.AD_GROUP)) {
                    return notAllowedAddNewAdGroupWithNewAdForImportMode(sheetName, representation.getAdGroupName());
                }

                if (!namesForNewAdGroups.contains(representation.getAdGroupName())) {
                    return newAdGroupByNameNotFound(sheetName, representation.getAdGroupName());
                }
            }

            if (representation.getAdGroupId() != null && !existingAdGroupIds.contains(representation.getAdGroupId())) {
                return adGroupByIdNotFound(sheetName, representation.getAdGroupId());
            }

            return null;
        };
    }

}
