package ru.yandex.direct.excel.processing.utils;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.BuiltinFormats;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.SheetUtil;

@ParametersAreNonnullByDefault
public class StyleUtils {

    public static final short TEXT_DATA_FORMAT = (short) BuiltinFormats.getBuiltinFormat("text");

    private StyleUtils() {
    }

    public static CellStyle getDefaultStyle(Workbook workbook) {
        Font defaultFont = workbook.createFont();
        defaultFont.setFontHeightInPoints((short) 12);
        CellStyle defaultStyle = workbook.createCellStyle();
        defaultStyle.setFont(defaultFont);
        defaultStyle.setBorderBottom(BorderStyle.THIN);
        defaultStyle.setBorderTop(BorderStyle.THIN);
        defaultStyle.setBorderLeft(BorderStyle.MEDIUM);
        defaultStyle.setBorderRight(BorderStyle.MEDIUM);

        setTextDataFormat(defaultStyle);
        setGrayStyleColor(defaultStyle);

        return defaultStyle;
    }

    /**
     * Проставляем для ячейки текстовый формат, чтобы MSExcel не пытался автоматически преобразовывать формат ячейки
     */
    private static void setTextDataFormat(CellStyle cellStyle) {
        cellStyle.setDataFormat(TEXT_DATA_FORMAT);
    }

    public static CellStyle getStyleForDescriptor(Workbook workbook) {
        Font descriptorFont = workbook.createFont();
        descriptorFont.setFontHeightInPoints((short) 20);
        CellStyle descriptorStyle = workbook.createCellStyle();
        descriptorStyle.setFont(descriptorFont);

        setTextDataFormat(descriptorStyle);

        return descriptorStyle;
    }

    public static CellStyle getStyleForTitle(Workbook workbook) {
        Font titleFont = workbook.createFont();
        titleFont.setFontHeightInPoints((short) 13);
        titleFont.setBold(true);

        CellStyle titleStyle = workbook.createCellStyle();
        titleStyle.setFont(titleFont);
        titleStyle.setIndention((short) 1);
        titleStyle.setBorderBottom(BorderStyle.MEDIUM);
        titleStyle.setBorderTop(BorderStyle.MEDIUM);
        titleStyle.setBorderLeft(BorderStyle.MEDIUM);
        titleStyle.setBorderRight(BorderStyle.MEDIUM);

        setTextDataFormat(titleStyle);
        setGrayStyleColor(titleStyle);

        return titleStyle;
    }

    private static void setGrayStyleColor(CellStyle titleStyle) {
        titleStyle.setBottomBorderColor(HSSFColor.HSSFColorPredefined.GREY_50_PERCENT.getIndex());
        titleStyle.setTopBorderColor(HSSFColor.HSSFColorPredefined.GREY_50_PERCENT.getIndex());
        titleStyle.setRightBorderColor(HSSFColor.HSSFColorPredefined.GREY_50_PERCENT.getIndex());
        titleStyle.setLeftBorderColor(HSSFColor.HSSFColorPredefined.GREY_50_PERCENT.getIndex());
    }

    public static void setCellWidth(Sheet sheet, int columnIndex) {
        double width = SheetUtil.getColumnWidth(sheet, columnIndex, false);
        if (width < 10) {
            width = 10;
        }
        if (width > 100) {
            width = 100;
        }
        sheet.setColumnWidth(columnIndex, (int) (width * 256));
    }
}
