package ru.yandex.direct.feature.generator;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;
import javax.lang.model.element.Modifier;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.google.common.base.CaseFormat;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.i18n.bundle.TranslationStub;
import ru.yandex.direct.jcommander.Command;
import ru.yandex.direct.jcommander.ParserWithHelp;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.capitalize;

public class FeatureGeneratorTool {
    private static final Logger logger = LoggerFactory.getLogger(FeatureGeneratorTool.class);
    private GenerateCommand generateCommand;

    public static final String README_URL =
            "https://a.yandex-team.ru/arc_vcs/direct/libs-internal/feature-generator/README.md";

    public FeatureGeneratorTool() {
        this.generateCommand = new GenerateCommand();
    }

    static class Feature {
        String nameUpper;
        String nameCamel;
        String description;
        String featureType;
        boolean deprecated;
        boolean isEnabledInE2ETests;

        public Feature(String name, String description, String featureType, boolean deprecated, boolean isEnabledInE2ETests) {
            name = StringUtils.lowerCase(name);
            this.nameUpper = CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.UPPER_UNDERSCORE, name);
            this.nameCamel = CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, name);
            this.description = description;
            this.featureType = featureType;
            this.deprecated = deprecated;
            this.isEnabledInE2ETests = isEnabledInE2ETests;
        }
    }

    public void run(String[] args) throws IOException {
        File configFile = null;
        File backEnumFile = null;
        File frontEnumFile = null;
        File frontendFeaturesTestConfig = null;
        Command command = ParserWithHelp.parseCommand(
                FeatureGeneratorTool.class.getCanonicalName(),
                args,
                new Object[]{},
                generateCommand
        );

        if (command == generateCommand) {
            logger.info("starting command " + generateCommand.getCommandName());
            configFile = new File(generateCommand.yamlFile);
            backEnumFile = new File(generateCommand.backEnumPath);
            frontEnumFile = new File(generateCommand.frontEnumPath);
            frontendFeaturesTestConfig = new File(generateCommand.frontendFeaturesTestConfig);
        }
        if (configFile == null || !configFile.exists()) {
            throw new RuntimeException("Invalid config file");
        }

        Writer back = new StringWriter();
        Writer front = new StringWriter();
        Writer frontendTest = new StringWriter();

        generateCode(configFile, back, front, frontendTest);

        Files.writeString(backEnumFile.toPath(), back.toString(), UTF_8);
        logger.info("backend enum saved {}", backEnumFile);

        Files.writeString(frontEnumFile.toPath(), front.toString(), UTF_8);
        logger.info("frontend enum saved {}", frontEnumFile.getAbsolutePath());

        Files.writeString(frontendFeaturesTestConfig.toPath(), frontendTest.toString(), UTF_8);
        logger.info("frontend features test config saved {}", frontendFeaturesTestConfig.getAbsolutePath());
    }

    /**
     * Сгеренировать код фронта и бека из конфига в соответствующие Writer-ы
     * @param configFile файл с конфигом
     * @param back сюда пишется код для бека
     * @param front сюда пишется код для фронта
     * @param frontendTest сюда пишется код для конфигурации тестов e2e
     * @throws IOException в случае отсутствия конфига
     */
    void generateCode(File configFile, Writer back, Writer front, Writer frontendTest) throws IOException {
        var textKey = "text";
        var statusKey = "status";
        var typeKey = "type";
        var deprecated = "deprecated";
        var isEnabledInE2E = "enabled_in_e2e";

        var mapper = new ObjectMapper(new YAMLFactory());
        mapper.findAndRegisterModules();
        Map<String, Map<String, String>> lst = mapper.readValue(configFile, Map.class);

        var features = EntryStream.of(lst)
                .mapKeyValue((k, v) -> {
                    var status = v.get(statusKey);
                    var text = v.get(textKey);
                    var type = v.get(typeKey);
                    var e2eEnabled = v.get(isEnabledInE2E);

                    var isEnabledInE2ETests = e2eEnabled != null && e2eEnabled.equals("true");

                    String featureType;
                    if (type != null) {
                        featureType = StringUtils.upperCase(type);
                    } else {
                        featureType = "TEMP";
                    }
                    return new Feature(k, text, featureType, deprecated.equals(status), isEnabledInE2ETests);
                }).toList();

        logger.info("{} features found", features.size());

        var humanReadableName = "humanReadableName";
        var featureType = "featureType";
        var name = "name";

        var pack = "ru.yandex.direct.feature";
        var featureName = "featureName";
        var featureByName = "FEATURE_NAME_BY_STRING";
        var featureNameTranslations = "FeatureNameTranslations";

        var featureTypeClassName = ClassName.get(pack, capitalize(featureType));

        var featureEnum =
                TypeSpec.enumBuilder(capitalize(featureName))
                        .addMethod(MethodSpec.constructorBuilder()
                                .addParameter(featureTypeClassName, featureType)
                                .addParameter(Translatable.class, humanReadableName)
                                .addStatement("this.$N = $N", featureType, featureType)
                                .addStatement("this.$N = this.toString().toLowerCase()", name)
                                .addStatement("this.$N = $N", humanReadableName, humanReadableName).build()
                        )
                        .addField(featureTypeClassName, featureType, Modifier.PRIVATE, Modifier.FINAL)
                        .addField(String.class, name, Modifier.PRIVATE, Modifier.FINAL)
                        .addField(Translatable.class, humanReadableName, Modifier.PRIVATE, Modifier.FINAL)
                        .addField(FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(Map.class),
                                        ClassName.get(String.class), ClassName.get("", capitalize(featureName))),
                                featureByName,
                                Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL).build())
                        .addStaticBlock(CodeBlock.builder()
                                .addStatement(featureByName + " = new $T<>()", HashMap.class)
                                .beginControlFlow("for ($N $N : $N.values())", capitalize(featureName), featureName,
                                        capitalize(featureName))
                                .add("$N.put($N.getName(), $N);\n", featureByName, featureName, featureName)
                                .endControlFlow()
                                .build())
                        .addMethod(MethodSpec.methodBuilder("get" + capitalize(name))
                                .addModifiers(Modifier.PUBLIC)
                                .addStatement("return " + name)
                                .returns(String.class)
                                .build())
                        .addMethod(MethodSpec.methodBuilder("get" + capitalize(featureType))
                                .addModifiers(Modifier.PUBLIC)
                                .addStatement("return " + featureType)
                                .returns(featureTypeClassName)
                                .build())
                        .addMethod(MethodSpec.methodBuilder("get" + capitalize(humanReadableName))
                                .addModifiers(Modifier.PUBLIC)
                                .addStatement("return " + humanReadableName)
                                .returns(Translatable.class)
                                .build())
                        .addMethod(MethodSpec.methodBuilder("fromString")
                                .addAnnotation(Nullable.class)
                                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                                .returns(ClassName.get("", capitalize(featureName)))
                                .addParameter(String.class, name)
                                .addStatement("return $L.get($L)", featureByName, name)
                                .build())
                        .addModifiers(Modifier.PUBLIC);

        var translatable = TypeSpec.interfaceBuilder(featureNameTranslations)
                .addSuperinterface(TranslationBundle.class)
                .addModifiers(Modifier.PROTECTED)
                .addField(
                        FieldSpec.builder(ClassName.get("", featureNameTranslations),
                                        "INSTANCE")
                                .addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
                                .initializer("$T.implement($L.class)", I18NBundle.class, featureNameTranslations)
                                .build());

        for (Feature feature : features) {
            var enumValueSpecBuilder = TypeSpec.anonymousClassBuilder("$T.$L, " +
                            featureNameTranslations + ".INSTANCE.$L()",
                    featureTypeClassName,
                    feature.featureType,
                    feature.nameCamel);
            if (feature.deprecated) {
                enumValueSpecBuilder.addAnnotation(Deprecated.class);
            }
            featureEnum.addEnumConstant(feature.nameUpper, enumValueSpecBuilder.build());
            translatable.addMethod(
                    MethodSpec.methodBuilder(feature.nameCamel)
                            .addAnnotation(AnnotationSpec.builder(TranslationStub.class)
                                    .addMember("value", "$S", feature.description).build())
                            .addModifiers(Modifier.ABSTRACT, Modifier.PUBLIC)
                            .returns(Translatable.class)
                            .build()
            );
        }

        featureEnum.addType(translatable.build());

        String comment = "Auto-generated by feature-generator, source: "
                + Paths.get(configFile.getName()).getFileName();

        back.append("// CHECKSTYLE:OFF \n");
        back.append("// For modification please use " + README_URL + "\n");
        back.append("// ").append(comment).append("\n");
        JavaFile javaFile = JavaFile.builder(pack, featureEnum.build())
                .skipJavaLangImports(true)
                .indent("    ")
                .build();
        javaFile.writeTo(back);

        front.write("// For modification please use " + README_URL + "\n");
        front.write("/* tslint:disable */\n");
        front.write("/* eslint-disable */\n");
        front.write("// " + comment + "\n");
        front.write("export type " + capitalize(featureName) + " =\n");
        var prefix = "    | ";
        for (int i = 0; i < features.size(); i++) {
            var feature = features.get(i);
            front.write(prefix + "\'" + feature.nameUpper + "\'");
            if(i == features.size() - 1) {
                front.write(";");
            }
            front.write(" // ");
            if (feature.deprecated) {
                front.write("@deprecated ");
            }
            front.write(feature.description + "\n");
        }

        var featureTestConfigName = "featuresTestConfig";

        frontendTest.write("// For modification please use " + README_URL + "\n");
        frontendTest.write("/* tslint:disable */\n");
        frontendTest.write("/* eslint-disable */\n");
        frontendTest.write("// " + comment + "\n");
        frontendTest.write("import { " + capitalize(featureName) + " } from './features';\n");
        frontendTest.write("export const " + featureTestConfigName + ": Record<" + capitalize(featureName) +", boolean> = {\n");

        for (Feature feature: features) {
            frontendTest.write("    " + feature.nameUpper + ": " + (feature.isEnabledInE2ETests ? "true" : "false") + ",\n");
        }

        frontendTest.write("};\n");
    }
}
