package ru.yandex.direct.libs.graphql;

import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import graphql.ExceptionWhileDataFetching;
import graphql.ExecutionResult;
import graphql.GraphQLError;
import graphql.GraphqlErrorHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.CommonTranslations;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.libs.graphql.model.GraphqlResponse;
import ru.yandex.direct.tracing.Trace;

import static java.lang.String.format;

/**
 * Форматирование ошибок выполнения graphql-запроса
 */
@ParametersAreNonnullByDefault
public class GraphqlHelper {
    public static final String REQ_ID_KEY = "reqId";
    private static final String BETA_ERROR_DISCLAIMER =
            " (you're seeing this message because it is development environment)";
    private static final String MESSAGE_KEY = "message";
    private static final Logger logger = LoggerFactory.getLogger(GraphqlHelper.class);

    private final EnvironmentType environmentType;
    private final TranslationService translationService;

    @Autowired
    public GraphqlHelper(
            EnvironmentType environmentType,
            TranslationService translationService) {
        this.environmentType = environmentType;
        this.translationService = translationService;
    }

    public GraphqlResponse formatResponse(ExecutionResult executionResult, boolean withSupportMessage) {
        addReqId(executionResult.getData());

        return new GraphqlResponse()
                .withData(executionResult.getData())
                .withExtensions(executionResult.getExtensions())
                .withErrors(executionResult.getErrors().stream()
                        .map(e -> this.sanitize(e, withSupportMessage))
                        .collect(Collectors.toList())
                );
    }

    public static void addReqId(@Nullable Map<String, Object> data) {
        if (data != null) {
            data.put(REQ_ID_KEY, String.valueOf(Trace.current().getSpanId()));
        }
    }

    private Map<String, Object> sanitize(GraphQLError error, boolean withSupportMessage) {
        Map<String, Object> sanitaizedError = GraphqlErrorHelper.toSpecification(error);
        Object message = sanitaizedError.get(MESSAGE_KEY);
        if (error instanceof ExceptionWhileDataFetching) {
            Throwable cause = ((ExceptionWhileDataFetching) error).getException();
            if (cause instanceof InvocationTargetException) {
                cause = ((InvocationTargetException) cause).getTargetException();
            }

            logger.error("Got exception during execution", cause);
            if (cause instanceof GraphqlPublicException) {
                message = translationService.translate(((GraphqlPublicException) cause).getDetailedMessage());
            } else if (environmentType.isBeta()) {
                // На бете тоже покажем сообщение, чтобы было проще отлаживаться
                message = cause.getMessage() + BETA_ERROR_DISCLAIMER;
            } else {
                message = translationService.translate(CommonTranslations.INSTANCE.serviceInternalError());
            }
        }

        if (withSupportMessage) {
            String supportMessage = translationService.translate(
                    GraphqlTranslations.INSTANCE
                            .supportDataMessage(format("%s = %d", REQ_ID_KEY, Trace.current().getSpanId()))
            );

            sanitaizedError.put(MESSAGE_KEY, format("%s. %s", message, supportMessage));
        } else {
            sanitaizedError.put(MESSAGE_KEY, Objects.toString(message));
        }
        addReqId(sanitaizedError);
        return sanitaizedError;
    }
}
