package ru.yandex.direct.grid.core.entity.banner.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBannersDisplayHrefStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerDisplayHref;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerTurboLanding;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSitelink;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerTurbolandingParams.BANNER_TURBOLANDING_PARAMS;
import static ru.yandex.direct.dbschema.ppc.tables.SitelinksLinks.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.tables.SitelinksSetToLink.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.tables.Turbolandings.TURBOLANDINGS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class GridBannerAdditionsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqReaderWithSupplier<GdiSitelink> sitelinksReader;
    private final JooqReaderWithSupplier<TurboLanding> sitelinksTurbolandingReader;
    private final JooqReaderWithSupplier<GdiBannerDisplayHref> bannerDisplayHrefReader;
    private final JooqReaderWithSupplier<GdiBannerTurboLanding> bannerTurboLandingReader;

    @Autowired
    public GridBannerAdditionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.sitelinksReader = JooqReaderWithSupplierBuilder.builder(GdiSitelink::new)
                .readProperty(GdiSitelink.SITELINKS_SET_ID, fromField(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .readProperty(GdiSitelink.TITLE, fromField(SITELINKS_LINKS.TITLE))
                .readProperty(GdiSitelink.HREF, fromField(SITELINKS_LINKS.HREF))
                .readProperty(GdiSitelink.DESCRIPTION, fromField(SITELINKS_LINKS.DESCRIPTION))
                .readProperty(GdiSitelink.TURBOLANDING_ID, fromField(SITELINKS_LINKS.TL_ID))
                .build();

        this.sitelinksTurbolandingReader = JooqReaderWithSupplierBuilder.builder(TurboLanding::new)
                .readProperty(TurboLanding.ID, fromField(TURBOLANDINGS.TL_ID))
                .readProperty(TurboLanding.CLIENT_ID, fromField(TURBOLANDINGS.CLIENT_ID))
                .readProperty(TurboLanding.NAME, fromField(TURBOLANDINGS.NAME))
                .readProperty(TurboLanding.URL, fromField(TURBOLANDINGS.HREF))
                .build();

        this.bannerDisplayHrefReader = JooqReaderWithSupplierBuilder.builder(GdiBannerDisplayHref::new)
                .readProperty(GdiBannerDisplayHref.HREF, fromField(BANNER_DISPLAY_HREFS.DISPLAY_HREF))
                .readProperty(GdiBannerDisplayHref.STATUS_MODERATE, fromField(BANNER_DISPLAY_HREFS.STATUS_MODERATE)
                        .by(GdiBannerBannersDisplayHrefStatusModerate::fromSource))
                .build();

        this.bannerTurboLandingReader = JooqReaderWithSupplierBuilder.builder(GdiBannerTurboLanding::new)
                .readProperty(GdiBannerTurboLanding.ID, fromField(TURBOLANDINGS.TL_ID))
                .readProperty(GdiBannerTurboLanding.CLIENT_ID, fromField(TURBOLANDINGS.CLIENT_ID))
                .readProperty(GdiBannerTurboLanding.NAME, fromField(TURBOLANDINGS.NAME))
                .readProperty(GdiBannerTurboLanding.URL, fromField(TURBOLANDINGS.HREF)
                        .by(RepositoryUtils::nullableStringFromNotNullDbField))
                .readProperty(GdiBannerTurboLanding.TURBO_SITE_HREF, fromField(TURBOLANDINGS.TURBO_SITE_HREF)
                        .by(RepositoryUtils::nullableStringFromNotNullDbField))
                .readProperty(GdiBannerTurboLanding.PREVIEW_HREF, fromField(TURBOLANDINGS.PREVIEW_HREF)
                        .by(RepositoryUtils::nullableStringFromNotNullDbField))
                .readProperty(GdiBannerTurboLanding.STATUS_MODERATE, fromField(BANNER_TURBOLANDINGS.STATUS_MODERATE)
                        .by(BannerTurboLandingStatusModerate::fromSource))
                .build();
    }

    public Map<Long, List<GdiSitelink>> getSitelinks(int shard, Collection<GdiBanner> banners) {
        Map<Long, Long> bannerIdToSitelinkSetId = StreamEx.of(banners)
                .mapToEntry(GdiBanner::getSitelinksSetId)
                .mapKeys(GdiBanner::getId)
                .nonNullValues()
                .toMap();

        List<GdiSitelink> sitelinks = dslContextProvider.ppc(shard)
                .select(sitelinksReader.getFieldsToRead())
                .select(sitelinksTurbolandingReader.getFieldsToRead())
                .from(SITELINKS_SET_TO_LINK)
                .innerJoin(SITELINKS_LINKS).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(SITELINKS_LINKS.TL_ID))
                .where(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.in(bannerIdToSitelinkSetId.values()))
                .fetch(this::constructGdiSitelinkFromRecord);

        Map<Long, List<GdiSitelink>> sitelinksSetIdToSitelinks = StreamEx.of(sitelinks)
                .groupingBy(GdiSitelink::getSitelinksSetId);
        return EntryStream.of(bannerIdToSitelinkSetId)
                .mapValues(sitelinksSetIdToSitelinks::get)
                .nonNullValues()
                .toMap();
    }

    private GdiSitelink constructGdiSitelinkFromRecord(Record record) {
        GdiSitelink sitelink = sitelinksReader.fromDb(record);

        if (record.get(Tables.SITELINKS_LINKS.TL_ID) != null) {
            TurboLanding turboLanding = sitelinksTurbolandingReader.fromDb(record);
            sitelink.withTurbolanding(turboLanding);
        }
        return sitelink;
    }

    public Map<Long, GdiBannerDisplayHref> getBannersDisplayHref(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_DISPLAY_HREFS.BID, BANNER_DISPLAY_HREFS.DISPLAY_HREF,
                        BANNER_DISPLAY_HREFS.STATUS_MODERATE)
                .from(BANNER_DISPLAY_HREFS)
                .where(BANNER_DISPLAY_HREFS.BID.in(bannerIds))
                .fetchMap(BANNER_DISPLAY_HREFS.BID, bannerDisplayHrefReader::fromDb);
    }

    public Map<Long, String> getBannerTurbolandingParams(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_TURBOLANDING_PARAMS.BID, BANNER_TURBOLANDING_PARAMS.HREF_PARAMS)
                .from(BANNER_TURBOLANDING_PARAMS)
                .where(BANNER_TURBOLANDING_PARAMS.BID.in(bannerIds))
                .fetchMap(BANNER_TURBOLANDING_PARAMS.BID, BANNER_TURBOLANDING_PARAMS.HREF_PARAMS);
    }

    /**
     * По заданному списку bannerIds возвращает map bannerId -> bannerTurboLanding, для банеров с турболендингами
     *
     * @param shard     номер шарда
     * @param bannerIds список bid, для которых нужно получить турболендинги
     */
    public Map<Long, GdiBannerTurboLanding> getBannerTurbolandingsByBannerId(int shard, List<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(bannerTurboLandingReader.getFieldsToRead())
                .select(BANNER_TURBOLANDINGS.BID)
                .from(BANNER_TURBOLANDINGS)
                .leftJoin(Tables.TURBOLANDINGS).on(Tables.TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .where(BANNER_TURBOLANDINGS.BID.in(bannerIds))
                .fetchMap(BANNER_TURBOLANDINGS.BID, bannerTurboLandingReader::fromDb);
    }
}
