package ru.yandex.direct.grid.core.entity.banner.repository;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction;
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerMobileContentInfo;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptyMap;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class GridBannerMobileContentRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqReaderWithSupplier<GdiBannerMobileContentInfo> mapper;

    @Autowired
    public GridBannerMobileContentRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = JooqReaderWithSupplierBuilder.builder(GdiBannerMobileContentInfo::new)
                .readProperty(GdiBannerMobileContentInfo.BANNER_ID, fromField(BANNERS.BID))
                .readProperty(GdiBannerMobileContentInfo.PRIMARY_ACTION,
                        fromField(BANNERS_MOBILE_CONTENT.PRIMARY_ACTION)
                                .by(value -> NewMobileContentPrimaryAction.valueOf(value.getLiteral().toUpperCase())))
                .readProperty(GdiBannerMobileContentInfo.REFLECTED_ATTRS,
                        fromField(BANNERS_MOBILE_CONTENT.REFLECTED_ATTRS)
                        .by(GridBannerMobileContentRepository::reflectedAttrsFromDbConverter))
                .readProperty(GdiBannerMobileContentInfo.IMPRESSION_URL, fromField(BANNERS_MOBILE_CONTENT.IMPRESSION_URL))
                .build();
    }

    public Map<Long, GdiBannerMobileContentInfo> getMobileContentInfoByBannerId(int shard, Set<Long> bannerIds) {
        if (CollectionUtils.isEmpty(bannerIds)) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(BANNERS)
                .leftJoin(BANNERS_MOBILE_CONTENT).on(BANNERS_MOBILE_CONTENT.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, mapper::fromDb);
    }

    private static Set<NewReflectedAttribute> reflectedAttrsFromDbConverter(String value) {
        if (isBlank(value)) {
            return Collections.emptySet();
        }
        return StreamEx.split(value, ",")
                .map(NewReflectedAttribute::fromTypedValue)
                .toSet();
    }
}
