package ru.yandex.direct.grid.core.entity.banner.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Row3;
import org.jooq.Select;
import org.jooq.SelectSelectStep;
import org.jooq.SelectWindowStep;
import org.jooq.TableField;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.ImagesStatusmoderate;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerFilter;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerOrderBy;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerOrderByField;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerPrimaryStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerShowType;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusBsSynced;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusPostModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusSitelinksModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusVCardModeration;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannersWithTotals;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.AdFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProvider;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.stats_from_query.WithTotalsUtils;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.schema.yt.tables.BannerstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.grid.schema.yt.tables.records.BannerstableDirectRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;

import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromStringFieldToEnum;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromYesNoFieldToBoolean;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetNumericSubstring;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetSubstringIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.not;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCaseAndWhitespaceType;
import static ru.yandex.direct.grid.schema.yt.Tables.BANNERSTABLE_DIRECT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.longNodeLocalDate;

/**
 * Репозиторий для получения данных и статистики баннеров из YT
 */
@Repository
@ParametersAreNonnullByDefault
public class GridBannerYtRepository {
    private static final BannerstableDirect BANNERS = BANNERSTABLE_DIRECT.as("B");
    private static final List<TableField<?, ?>> DEFAULT_ORDER = ImmutableList.of(BANNERS.CID, BANNERS.PID, BANNERS.BID);

    private static final Set<String> BASED_ON_CREATIVE_STATUS_TYPES =
            ImmutableSet.of(BannersBannerType.image_ad.getLiteral(), BannersBannerType.cpc_video.getLiteral(),
                    BannersBannerType.cpm_banner.getLiteral(), BannersBannerType.performance.getLiteral());
    private static final Set<String> BASED_ON_IMAGE_STATUS_TYPES =
            ImmutableSet.of(BannersBannerType.image_ad.getLiteral(), BannersBannerType.mcbanner.getLiteral());
    private static final Set<String> BASED_ON_BANNER_STATUS_TYPES =
            Sets.difference(
                    mapSet(new HashSet<>(Arrays.asList(BannersBannerType.values())), BannersBannerType::getLiteral),
                    Sets.union(BASED_ON_CREATIVE_STATUS_TYPES, BASED_ON_IMAGE_STATUS_TYPES));

    private static final Condition PRIMARY_STATUS_ACTIVE_CONDITION =
            BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes.getLiteral())
                    .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No.getLiteral()))
                    .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes.getLiteral()))
                    .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes.getLiteral()))
                    .and((BANNERS.BANNER_TYPE.in(BASED_ON_IMAGE_STATUS_TYPES)
                            .and(BANNERS.BANNER_IMAGES_STATUS_MODERATE.eq(ImagesStatusmoderate.Yes.getLiteral())))
                            .or((BANNERS.BANNER_TYPE.in(BASED_ON_CREATIVE_STATUS_TYPES)
                                    .and(BANNERS.BANNERS_PERFORMANCE_STATUS_MODERATE
                                            .eq(BannersPerformanceStatusmoderate.Yes.getLiteral()))))
                            .or(BANNERS.BANNER_TYPE.in(BASED_ON_BANNER_STATUS_TYPES)));

    private static final Condition PRIMARY_STATUS_DRAFT_CONDITION =
            BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.New.getLiteral())
                    .or(BANNERS.BANNER_TYPE.in(BASED_ON_IMAGE_STATUS_TYPES)
                            .and(BANNERS.BANNER_IMAGES_STATUS_MODERATE.eq(ImagesStatusmoderate.New.getLiteral())))
                    .or(BANNERS.BANNER_TYPE.in(BASED_ON_CREATIVE_STATUS_TYPES)
                            .and(BANNERS.BANNERS_PERFORMANCE_STATUS_MODERATE
                                    .eq(BannersPerformanceStatusmoderate.New.getLiteral())));

    private static final Condition PRIMARY_STATUS_REJECTED_CONDITION =
            BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.No.getLiteral())
                    .or(BANNERS.BANNER_TYPE.in(BASED_ON_IMAGE_STATUS_TYPES)
                            .and(BANNERS.BANNER_IMAGES_STATUS_MODERATE.eq(ImagesStatusmoderate.No.getLiteral())))
                    .or(BANNERS.BANNER_TYPE.in(BASED_ON_CREATIVE_STATUS_TYPES)
                            .and(BANNERS.BANNERS_PERFORMANCE_STATUS_MODERATE
                                    .eq(BannersPerformanceStatusmoderate.No.getLiteral())));

    private static final Condition PRIMARY_STATUS_MODERATION_CONDITION = BANNERS.STATUS_MODERATE.in(
            BannersStatusmoderate.Sending.getLiteral(), BannersStatusmoderate.Sent.getLiteral(),
            BannersStatusmoderate.Ready.getLiteral())
            .and(BANNERS.STATUS_POST_MODERATE
                    .in(BannersStatuspostmoderate.No.getLiteral(), BannersStatuspostmoderate.Rejected.getLiteral()))
            .or(BANNERS.BANNER_TYPE.in(BASED_ON_IMAGE_STATUS_TYPES)
                    .and(BANNERS.BANNER_IMAGES_STATUS_MODERATE
                            .in(ImagesStatusmoderate.Ready.getLiteral(),
                                    ImagesStatusmoderate.Sending.getLiteral(),
                                    ImagesStatusmoderate.Sent.getLiteral())))
            .or(BANNERS.BANNER_TYPE.in(BASED_ON_CREATIVE_STATUS_TYPES)
                    .and(BANNERS.BANNERS_PERFORMANCE_STATUS_MODERATE
                            .in(BannersPerformanceStatusmoderate.Ready.getLiteral(),
                                    BannersPerformanceStatusmoderate.Sending.getLiteral(),
                                    BannersPerformanceStatusmoderate.Sent.getLiteral())));

    /**
     * NB: внутри bannerPrimaryStatusConditionMap() берутся conditions вышеперечисленные.
     * Перемещать инициализацию выше — нельзя
     */
    private static final Map<GdiBannerPrimaryStatus, Condition> BANNER_PRIMARY_STATUS_CONDITION_MAP =
            bannerPrimaryStatusConditionMap();

    private static final JooqFilterProcessor<GdiBannerFilter> FILTER_PROCESSOR =
            getFilterProcessor(true);
    private static final JooqFilterProcessor<GdiBannerFilter> FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS =
            getFilterProcessor(false);

    private static JooqFilterProcessor<GdiBannerFilter> getFilterProcessor(boolean withFilterByStatus) {
        var builder = JooqFilterProcessor.<GdiBannerFilter>builder()
                .withFilter(GdiBannerFilter::getCampaignIdIn, BANNERS.CID::in)
                .withFilter(GdiBannerFilter::getAdGroupIdIn, BANNERS.PID::in)
                .withFilter(GdiBannerFilter::getBannerIdIn, BANNERS.BID::in)
                .withFilter(GdiBannerFilter::getBannerIdNotIn, not(BANNERS.BID::in))
                .withFilter(GdiBannerFilter::getBannerIdContainsAny, inSetNumericSubstring(BANNERS.BID))
                //Фильтр на рекомендации - тот же фильтр на идентификаторы баннеров. Но их много (до 100 тысяч)
                //и yt работает, только когда задается детальное условие на префикс индекса (cid, pid, bid)
                .withFilter(GdiBannerFilter::getRecommendations, recommendations -> {
                    List<Row3<Long, Long, Long>> rows = mapList(recommendations,
                            r -> row(r.getCid(), r.getPid(), r.getBid()));
                    return row(BANNERS.CID, BANNERS.PID, BANNERS.BID).in(rows);
                })
                .withFilter(GdiBannerFilter::getTypeIn,
                        types -> BANNERS.BANNER_TYPE.in(mapList(types, t -> Objects
                                .requireNonNull(t).getLiteral())))
                .withFilter(GdiBannerFilter::getExportIdIn, BANNERS.BANNER_ID::in)
                .withFilter(GdiBannerFilter::getExportIdNotIn, not(BANNERS.BANNER_ID::in))
                .withFilter(GdiBannerFilter::getExportIdContainsAny, inSetNumericSubstring(BANNERS.BANNER_ID))
                .withFilter(GdiBannerFilter::getInternalAdTemplateIdIn, BANNERS.BANNERS_INTERNAL_TEMPLATE_ID::in)
                .withFilter(GdiBannerFilter::getTitleContains, substringIgnoreCase(BANNERS.TITLE))
                .withFilter(GdiBannerFilter::getTitleOrBodyContains, JooqFilterProvider
                        .or(substringIgnoreCaseAndWhitespaceType(BANNERS.TITLE),
                                substringIgnoreCaseAndWhitespaceType(BANNERS.TITLE_EXTENSION),
                                substringIgnoreCaseAndWhitespaceType(BANNERS.BODY)))
                .withFilter(GdiBannerFilter::getTitleIn, inSetIgnoreCase(BANNERS.TITLE))
                .withFilter(GdiBannerFilter::getTitleNotContains, not(substringIgnoreCase(BANNERS.TITLE)))
                .withFilter(GdiBannerFilter::getTitleNotIn, not(inSetIgnoreCase(BANNERS.TITLE)))
                .withFilter(GdiBannerFilter::getInternalAdTitleContains,
                        inSetSubstringIgnoreCase(BANNERS.BANNERS_INTERNAL_DESCRIPTION))
                .withFilter(GdiBannerFilter::getInternalAdTitleIn,
                        inSetIgnoreCase(BANNERS.BANNERS_INTERNAL_DESCRIPTION))
                .withFilter(GdiBannerFilter::getInternalAdTitleNotContains,
                        not(inSetSubstringIgnoreCase(BANNERS.BANNERS_INTERNAL_DESCRIPTION)))
                .withFilter(GdiBannerFilter::getInternalAdTitleNotIn,
                        not(inSetIgnoreCase(BANNERS.BANNERS_INTERNAL_DESCRIPTION)))
                .withFilter(GdiBannerFilter::getTitleExtensionContains,
                        substringIgnoreCase(BANNERS.TITLE_EXTENSION))
                .withFilter(GdiBannerFilter::getTitleExtensionIn, inSetIgnoreCase(BANNERS.TITLE_EXTENSION))
                .withFilter(GdiBannerFilter::getTitleExtensionNotContains,
                        not(substringIgnoreCase(BANNERS.TITLE_EXTENSION)))
                .withFilter(GdiBannerFilter::getTitleExtensionNotIn, not(inSetIgnoreCase(BANNERS.TITLE_EXTENSION)))
                .withFilter(GdiBannerFilter::getBodyContains, substringIgnoreCase(BANNERS.BODY))
                .withFilter(GdiBannerFilter::getBodyIn, inSetIgnoreCase(BANNERS.BODY))
                .withFilter(GdiBannerFilter::getBodyNotContains, not(substringIgnoreCase(BANNERS.BODY)))
                .withFilter(GdiBannerFilter::getBodyNotIn, not(inSetIgnoreCase(BANNERS.BODY)))
                .withFilter(GdiBannerFilter::getHrefContains, substringIgnoreCase(BANNERS.HREF))
                .withFilter(GdiBannerFilter::getHrefIn, inSetIgnoreCase(BANNERS.HREF))
                .withFilter(GdiBannerFilter::getHrefNotContains, not(substringIgnoreCase(BANNERS.HREF)))
                .withFilter(GdiBannerFilter::getHrefNotIn, not(inSetIgnoreCase(BANNERS.HREF)))
                .withFilter(GdiBannerFilter::getImageExists,
                        b -> {
                            Condition hasNoImage = YtDSL.isNull(BANNERS.IMAGES_IMAGE_HASH)
                                    .and(YtDSL.isNull(BANNERS.BANNER_IMAGES_IMAGE_HASH)
                                            .and(YtDSL.isNull(BANNERS.CREATIVE_ID)));
                            return (b ? DSL.not(hasNoImage) : hasNoImage);
                        })
                .withFilter(GdiBannerFilter::getSitelinksExists,
                        b -> (b ? DSL.not(YtDSL.isNull(BANNERS.SITELINKS_SET_ID))
                                : YtDSL.isNull(BANNERS.SITELINKS_SET_ID)))
                .withFilter(GdiBannerFilter::getVcardExists,
                        b -> (b ? DSL.not(YtDSL.isNull(BANNERS.VCARD_ID)) : YtDSL.isNull(BANNERS.VCARD_ID)));
        if (withFilterByStatus) {
            builder
                    .withFilter(GdiBannerFilter::getPrimaryStatusContains,
                            GridBannerYtRepository::getBannerPrimaryStatusConditions)
                    .withFilter(GdiBannerFilter::getArchived, aBoolean -> BANNERS.STATUS_ARCH
                            .eq(GridBannerMapping.booleanFromBannerStatusArch(aBoolean)));
        }
        return builder.build();
    }

    private final YtDynamicSupport ytSupport;
    private final YtFieldMapper<GdiBanner, BannerstableDirectRecord> bannerMapper;
    private final List<OrderField<?>> defaultOrderBy;
    private final Map<GdiBannerOrderByField, Field<?>> sortingMap;
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);

    @Autowired
    public GridBannerYtRepository(YtDynamicSupport gridYtSupport) {
        this.ytSupport = gridYtSupport;

        JooqReaderWithSupplier<GdiBanner> internalReader =
                JooqReaderWithSupplierBuilder.builder(GdiBanner::new)
                        .readProperty(GdiBanner.ID, fromField(BANNERS.BID))
                        .readProperty(GdiBanner.CAMPAIGN_ID, fromField(BANNERS.CID))
                        .readProperty(GdiBanner.GROUP_ID, fromField(BANNERS.PID))
                        .readPropertyForFirst(GdiBanner.TYPE,
                                fromStringFieldToEnum(BANNERS.TYPE, GdiBannerShowType.class))
                        .readPropertyForFirst(GdiBanner.BANNER_TYPE, fromField(BANNERS.BANNER_TYPE)
                                .by(value -> ifNotNull(value, name -> BannersBannerType.valueOf(name.toLowerCase()))))
                        .readPropertyForFirst(GdiBanner.TITLE, fromField(BANNERS.TITLE))
                        .readPropertyForFirst(GdiBanner.TITLE_EXTENSION, fromField(BANNERS.TITLE_EXTENSION))
                        .readPropertyForFirst(GdiBanner.BODY, fromField(BANNERS.BODY))
                        .readPropertyForFirst(GdiBanner.HREF, fromField(BANNERS.HREF))

                        .readPropertyForFirst(GdiBanner.INTERNAL_AD_TITLE,
                                fromField(BANNERS.BANNERS_INTERNAL_DESCRIPTION))
                        .readPropertyForFirst(GdiBanner.INTERNAL_AD_TEMPLATE_ID,
                                fromField(BANNERS.BANNERS_INTERNAL_TEMPLATE_ID))

                        .readPropertyForFirst(GdiBanner.DOMAIN_ID, fromField(BANNERS.DOMAIN_ID))
                        .readPropertyForFirst(GdiBanner.BS_BANNER_ID, fromField(BANNERS.BANNER_ID))

                        .readPropertyForFirst(GdiBanner.STATUS_SHOW, fromYesNoFieldToBoolean(BANNERS.STATUS_SHOW))
                        .readPropertyForFirst(GdiBanner.STATUS_ACTIVE, fromYesNoFieldToBoolean(BANNERS.STATUS_ACTIVE))
                        .readPropertyForFirst(GdiBanner.STATUS_ARCHIVED, fromYesNoFieldToBoolean(BANNERS.STATUS_ARCH))
                        .readPropertyForFirst(GdiBanner.STATUS_MODERATE,
                                fromStringFieldToEnum(BANNERS.STATUS_MODERATE, GdiBannerStatusModerate.class))
                        .readPropertyForFirst(GdiBanner.STATUS_POST_MODERATE,
                                fromStringFieldToEnum(BANNERS.STATUS_POST_MODERATE, GdiBannerStatusPostModerate.class))
                        .readPropertyForFirst(GdiBanner.STATUS_SITELINKS_MODERATE,
                                fromStringFieldToEnum(BANNERS.STATUS_SITELINKS_MODERATE,
                                        GdiBannerStatusSitelinksModerate.class))
                        .readPropertyForFirst(GdiBanner.STATUS_BS_SYNCED,
                                fromStringFieldToEnum(BANNERS.STATUS_BS_SYNCED, GdiBannerStatusBsSynced.class))

                        .readPropertyForFirst(GdiBanner.LAST_CHANGE, fromField(BANNERS.LAST_CHANGE)
                                .by(YtMappingUtils::dateTimeFromString))
                        .readPropertyForFirst(GdiBanner.DOMAIN, fromField(BANNERS.DOMAIN))
                        .readPropertyForFirst(GdiBanner.REVERSE_DOMAIN, fromField(BANNERS.REVERSE_DOMAIN))
                        .readPropertyForFirst(GdiBanner.PHONE_FLAG, fromStringFieldToEnum(BANNERS.PHONEFLAG,
                                GdiBannerStatusVCardModeration.class))

                        .readPropertyForFirst(GdiBanner.VCARD_ID, fromField(BANNERS.VCARD_ID))
                        .readPropertyForFirst(GdiBanner.FLAGS, fromField(BANNERS.FLAGS).by(YtMappingUtils::stringToSet))
                        .readPropertyForFirst(GdiBanner.MOD_FLAGS, fromField(BANNERS.FLAGS).by(BannerFlags::fromSource))
                        .readPropertyForFirst(GdiBanner.SITELINKS_SET_ID, fromField(BANNERS.SITELINKS_SET_ID))
                        .readPropertyForFirst(GdiBanner.OPTS_GEO_FLAG, fromLongFieldToBoolean(BANNERS.OPTS_GEOFLAG))
                        .readPropertyForFirst(GdiBanner.OPTS_NO_DISPLAY_HREF,
                                fromLongFieldToBoolean(BANNERS.OPTS_NO_DISPLAY_HREF))
                        .build();

        bannerMapper = new YtFieldMapper<>(internalReader, BANNERS);

        defaultOrderBy = mapList(DEFAULT_ORDER, bannerMapper::fieldAlias);

        sortingMap = new HashMap<>();
        sortingMap.put(GdiBannerOrderByField.ID, bannerMapper.fieldAlias(BANNERS.BID));
        sortingMap.put(GdiBannerOrderByField.GROUP_ID, bannerMapper.fieldAlias(BANNERS.PID));
        sortingMap.put(GdiBannerOrderByField.CAMPAIGN_ID, bannerMapper.fieldAlias(BANNERS.CID));
        sortingMap.put(GdiBannerOrderByField.BANNER_TYPE, bannerMapper.fieldAlias(BANNERS.BANNER_TYPE));
        sortingMap.put(GdiBannerOrderByField.STATUS, bannerMapper.fieldAlias(BANNERS.STATUS_MODERATE));
        sortingMap.put(GdiBannerOrderByField.TITLE, bannerMapper.fieldAlias(BANNERS.TITLE));
        sortingMap.put(GdiBannerOrderByField.TITLE_EXTENSION, bannerMapper.fieldAlias(BANNERS.TITLE_EXTENSION));
        sortingMap.put(GdiBannerOrderByField.DOMAIN, bannerMapper.fieldAlias(BANNERS.DOMAIN));
        sortingMap.put(GdiBannerOrderByField.HREF, bannerMapper.fieldAlias(BANNERS.HREF));
        sortingMap.put(GdiBannerOrderByField.LAST_CHANGE, bannerMapper.fieldAlias(BANNERS.LAST_CHANGE));
        sortingMap.put(GdiBannerOrderByField.INTERNAL_AD_TEMPLATE_ID,
                bannerMapper.fieldAlias(BANNERS.BANNERS_INTERNAL_TEMPLATE_ID));
        sortingMap.put(GdiBannerOrderByField.INTERNAL_AD_TITLE,
                bannerMapper.fieldAlias(BANNERS.BANNERS_INTERNAL_DESCRIPTION));

        gridStat.addStatOrderingToMap(sortingMap, GdiBannerOrderByField.class);
    }

    private static Condition getBannerPrimaryStatusConditions(Set<GdiBannerPrimaryStatus> primaryStatusSet) {

        List<Condition> conditions = new ArrayList<>();

        if (!primaryStatusSet.isEmpty()) {
            conditions = primaryStatusSet.stream()
                    .map(st -> BANNER_PRIMARY_STATUS_CONDITION_MAP.getOrDefault(st, null))
                    .filter(Objects::nonNull)
                    .collect(toList());
        }
        return (!conditions.isEmpty()) ? DSL.or(conditions) : null;
    }

    private static Map<GdiBannerPrimaryStatus, Condition> bannerPrimaryStatusConditionMap() {
        Map<GdiBannerPrimaryStatus, Condition> bannerPrimaryStatusConditions =
                new EnumMap<>(GdiBannerPrimaryStatus.class);

        bannerPrimaryStatusConditions
                .put(GdiBannerPrimaryStatus.ARCHIVED, BANNERS.STATUS_ARCH.eq(BannersStatusarch.Yes.getLiteral()));
        bannerPrimaryStatusConditions.put(GdiBannerPrimaryStatus.MANUALLY_SUSPENDED,
                BANNERS.STATUS_SHOW.eq(BannersStatusshow.No.getLiteral())
                        .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No.getLiteral())));
        bannerPrimaryStatusConditions.put(GdiBannerPrimaryStatus.ACTIVE, PRIMARY_STATUS_ACTIVE_CONDITION);
        bannerPrimaryStatusConditions.put(GdiBannerPrimaryStatus.MODERATION, PRIMARY_STATUS_MODERATION_CONDITION);
        bannerPrimaryStatusConditions
                .put(GdiBannerPrimaryStatus.MODERATION_REJECTED, PRIMARY_STATUS_REJECTED_CONDITION);
        bannerPrimaryStatusConditions.put(GdiBannerPrimaryStatus.DRAFT, PRIMARY_STATUS_DRAFT_CONDITION);

        return bannerPrimaryStatusConditions;
    }

    /**
     * Получить данные о баннерах
     *
     * @param shard                   шард, в котором хранятся баннеры (исключительно для улучшения результатов запроса)
     * @param filter                  фильтр для выборки данных
     * @param bannerOrderByList       список полей с порядком сортировки данных
     * @param statFrom                начало периода, за который нужно получать статистику
     * @param statTo                  конец периода, за который нужно получать статистику (включительно)
     * @param statByDaysFrom          начало периода, за который нужно получать статистику по дням
     * @param statByDaysTo            конец периода, за который нужно получать статистику по дням (включительно)
     * @param isFlat                  статистика показов: true - РСЯ, false - Поиск, null - оба варината
     * @param limitOffset             верхний предел количества полученных баннеров и смещение относительно начала
     *                                выборки
     * @param goalIds                 идентификаторы целей
     * @param goalIdsForRevenue       идентификаторы целей, по которым считается доход
     * @param adFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля запрошены на
     *                                верхнем уровне
     * @param clientFeatures          все включенные фичи клиента
     * @param disableStatusFilter     не фильтровать по статусам
     */
    public GdiBannersWithTotals getBanners(int shard, GdiBannerFilter filter,
                                           List<GdiBannerOrderBy> bannerOrderByList,
                                           LocalDate statFrom, LocalDate statTo, @Nullable Boolean isFlat,
                                           LimitOffset limitOffset,
                                           Set<Long> goalIds,
                                           @Nullable Set<Long> goalIdsForRevenue,
                                           @Nullable LocalDate statByDaysFrom, @Nullable LocalDate statByDaysTo,
                                           AdFetchedFieldsResolver adFetchedFieldsResolver,
                                           Set<String> clientFeatures,
                                           boolean disableStatusFilter) {
        if (CollectionUtils.isAllEmpty(filter.getCampaignIdIn(), filter.getAdGroupIdIn(),
                filter.getBannerIdIn(), filter.getBannerIdNotIn(), filter.getRecommendations())) {
            return new GdiBannersWithTotals()
                    .withGdiBanners(Collections.emptyList());
        }

        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(bannerMapper.getFieldsToRead())
                .select(bannerMapper.getFieldsToReadForFirst());


        //Читаем статистику, только если она запрошена или есть фильтры по ней или есть сортировка по ним
        boolean hasStatFieldsSort = gridStat.hasStatFieldsSort(bannerOrderByList);
        boolean withStatistic = adFetchedFieldsResolver.getStats() || filter.getStats() != null
                || filter.getGoalStats() != null || hasStatFieldsSort;
        if (withStatistic) {
            selectStatFields(selectStep, statFrom, statTo, isFlat, goalIds, goalIdsForRevenue);
        } else {
            selectStep.from(BANNERS);
        }

        Condition baseCondition =
                (disableStatusFilter ? FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS : FILTER_PROCESSOR).apply(filter);

        if (clientFeatures.contains(FeatureName.CREATIVE_FREE_INTERFACE.getName())) {
            baseCondition = nvl(baseCondition, noCondition())
                    .and(DSL.not(BANNERS.BANNER_TYPE.in(BannersBannerType.performance_main.getLiteral(),
                            BannersBannerType.performance.getLiteral())));
        }

        // Группируем по всем выбираемым полям, так как они не аггрегируемые
        List<Field<?>> groupBy = new ArrayList<>(bannerMapper.getFieldsToRead());
        // И добавляем ключи для ускорения
        groupBy.add(BANNERS.CID_HASH);
        groupBy.add(BANNERS.__SHARD__);

        selectStep.where(baseCondition)
                .groupBy(groupBy);

        SelectWindowStep<Record> windowStep = selectStep;

        Condition havingCondition = gridStat.getStatHavingCondition(filter.getStats());
        if (filter.getGoalStats() != null) {
            havingCondition = gridStat.addGoalStatHavingCondition(havingCondition, filter.getGoalStats());
        }

        if (havingCondition != null) {
            windowStep = selectStep.having(havingCondition);
        }

        List<OrderField<?>> orderFields = gridStat.getOrderFields(sortingMap, bannerOrderByList, defaultOrderBy);

        Select query = windowStep
                .orderBy(orderFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        boolean addWithTotalsToQuery = clientFeatures.contains(FeatureName.ADD_WITH_TOTALS_TO_BANNER_QUERY.getName());
        List<GdiBanner> banners = mapList(ytSupport
                        .selectRows(shard, query, addWithTotalsToQuery && withStatistic).getYTreeRows(),
                (n -> bannerMapper.fromNode(n)
                        .withStat(GridStatNew.addZeros(goalIdsForRevenue != null ?
                                gridStat.extractStatsEntryWithGoalsRevenue(n, goalIds, goalIdsForRevenue) :
                                gridStat.extractStatsEntry(n)))
                        .withGoalStats(gridStat.extractGoalStatEntries(n, goalIds))));

        if (adFetchedFieldsResolver.getStatsByDays() && statByDaysFrom != null && statByDaysTo != null) {
            // Если запрошена статистика по дням, то делаем отдельный запрос для ее получения.
            // Изначально планировалось делать это одним запросом статистики по дням, а потом
            // для получения агрегированной статистики делать эту самую агрегацию руками (а не в базе как сейчас).
            // Однако это оказалось затруднительно из-за поддержки orderBy, having, нетривиальной агрегации.
            // todo подумать над тем, будет ли лучше все-таки делать один запрос для получения сразу всей инфы
            Map<Long, List<GdiEntityStats>> statsByDaysByBannerId =
                    selectStatsByDaysByBannerId(shard, baseCondition, mapList(banners, GdiBanner::getId),
                            statByDaysFrom, statByDaysTo, isFlat);
            for (GdiBanner banner : banners) {
                banner.setStatsByDays(statsByDaysByBannerId.get(banner.getId()));
            }
        }

        // Возвращаем totals из запроса только если кол. баннеров = лимиту (banners.size == limit + total row)
        boolean withTotalStats = WithTotalsUtils
                .withTotalStats(addWithTotalsToQuery, withStatistic, banners, limitOffset);

        GdiBanner bannerWithTotals = !withTotalStats ? null : StreamEx.of(banners)
                .findFirst(banner -> Objects.isNull(banner.getId()))
                .orElse(null);

        return new GdiBannersWithTotals()
                .withGdiBanners(StreamEx.of(banners).filter(banner -> Objects.nonNull(banner.getId())).toList())
                .withTotalStats(bannerWithTotals != null ? bannerWithTotals.getStat() : null);
    }

    private void selectStatFields(SelectSelectStep<Record> selectStep, LocalDate statFrom, LocalDate statTo,
                                  @Nullable Boolean isFlat, @Nullable Set<Long> goalIds,
                                  @Nullable Set<Long> goalIdsForRevenue) {
        Directphrasestatv2Bs statTable = gridStat.getTableData().table();
        TableOnConditionStep<Record> joinStep = gridStat.joinStatColumns(BANNERS,
                row(BANNERS.CID, BANNERS.PID, BANNERS.BID)
                        .eq(statTable.EXPORT_ID, statTable.GROUP_EXPORT_ID, statTable.DIRECT_BANNER_ID),
                statFrom, statTo, isFlat);
        selectStep.select(gridStat.getStatSelectFields());
        if (goalIds != null) {
            joinStep = gridStat.joinGoalStatColumns(joinStep, goalIds);
            selectStep.select(gridStat.getGoalStatFields(goalIds, goalIdsForRevenue));
        }
        selectStep.from(joinStep);
    }

    private Map<Long, List<GdiEntityStats>> selectStatsByDaysByBannerId(
            int shard, Condition baseCondition, List<Long> bannerIds,
            LocalDate statByDaysFrom, LocalDate statByDaysTo, @Nullable Boolean isFlat) {
        if (bannerIds.isEmpty()) {
            return Map.of();
        }
        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(List.of(bannerMapper.fieldAlias(BANNERS.BID)));
        selectStatFields(selectStep, statByDaysFrom, statByDaysTo, isFlat, null, null);
        Field<Long> updateTimeField = gridStat.getTableData().updateTime().as("UpdateTime");
        selectStep.select(updateTimeField);

        List<Field<?>> groupBy = new ArrayList<>();
        groupBy.add(bannerMapper.fieldAlias(BANNERS.BID));
        groupBy.add(BANNERS.CID_HASH);
        groupBy.add(BANNERS.__SHARD__);
        groupBy.add(updateTimeField);
        selectStep.where(baseCondition.and(BANNERS.BID.in(bannerIds)))
                .groupBy(groupBy);

        return ytSupport.selectRows(shard, selectStep).getYTreeRows().stream()
                .collect(Collectors.groupingBy(
                        n -> bannerMapper.fromNode(n).getId(),
                        Collectors.mapping(
                                n -> GridStatNew.addZeros(gridStat.extractStatsEntry(n)
                                        .withDay(longNodeLocalDate(n, updateTimeField.getName()))),
                                toList()
                        )
                ));
    }
}
