package ru.yandex.direct.grid.core.entity.banner.repository;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ListMultimap;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerWithSitelinksForReplace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSourceReplaceDisplayHrefBanner;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.sitelink.service.SitelinkUtils.cloneSitelinks;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDING_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SETS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class GridFindAndReplaceBannerRepository {

    private final DslContextProvider dslContextProvider;
    private final SitelinkSetRepository sitelinkSetRepository;

    private final JooqReaderWithSupplier<GdiBannerWithSitelinksForReplace> bannerReader;
    private final JooqReaderWithSupplier<GdiSourceReplaceDisplayHrefBanner> displayHrefBannerReader;
    private final Collection<Field<?>> bannerFieldsToRead;
    private final Collection<Field<?>> displayHrefBannerFieldsToRead;
    private final BannersUrlHelper bannersUrlHelper;

    @Autowired
    public GridFindAndReplaceBannerRepository(DslContextProvider dslContextProvider,
                                              SitelinkSetRepository sitelinkSetRepository,
                                              BannersUrlHelper bannersUrlHelper) {
        this.dslContextProvider = dslContextProvider;
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.bannersUrlHelper = bannersUrlHelper;
        bannerReader = JooqReaderWithSupplierBuilder.builder(GdiBannerWithSitelinksForReplace::new)
                .readProperty(GdiBannerWithSitelinksForReplace.ID, fromField(BANNERS.BID))
                .readProperty(GdiBannerWithSitelinksForReplace.BANNER_TYPE,
                        fromField(BANNERS.BANNER_TYPE))
                .readProperty(GdiBannerWithSitelinksForReplace.AD_GROUP_TYPE,
                        fromField(PHRASES.ADGROUP_TYPE).by(AdGroupType::fromSource))
                .readProperty(GdiBannerWithSitelinksForReplace.TITLE, fromField(BANNERS.TITLE))
                .readProperty(GdiBannerWithSitelinksForReplace.TITLE_EXTENSION, fromField(BANNERS.TITLE_EXTENSION))
                .readProperty(GdiBannerWithSitelinksForReplace.BODY, fromField(BANNERS.BODY))
                .readProperty(GdiBannerWithSitelinksForReplace.IMAGE_HASH, fromField(IMAGES.IMAGE_HASH))
                .readProperty(GdiBannerWithSitelinksForReplace.HREF, fromField(BANNERS.HREF))
                .readProperty(GdiBannerWithSitelinksForReplace.SITELINK_SET_ID, fromField(BANNERS.SITELINKS_SET_ID))
                .readProperty(GdiBannerWithSitelinksForReplace.TURBOLANDING_PARAMS,
                        fromField(BANNER_TURBOLANDING_PARAMS.HREF_PARAMS))
                .build();
        bannerFieldsToRead = bannerReader.getFieldsToRead();

        displayHrefBannerReader = JooqReaderWithSupplierBuilder.builder(GdiSourceReplaceDisplayHrefBanner::new)
                .readProperty(GdiSourceReplaceDisplayHrefBanner.BANNER_ID, fromField(BANNERS.BID))
                .readProperty(GdiSourceReplaceDisplayHrefBanner.BANNER_TYPE,
                        fromField(BANNERS.BANNER_TYPE))
                .readProperty(GdiSourceReplaceDisplayHrefBanner.SOURCE_DISPLAY_HREF,
                        fromField(BANNER_DISPLAY_HREFS.DISPLAY_HREF))
                .build();
        displayHrefBannerFieldsToRead = displayHrefBannerReader.getFieldsToRead();
    }

    /**
     * Возвращает информацию, необходимую для замены ссылок.
     * В {@link GdiBannerWithSitelinksForReplace#SITELINKS} будет содержаться:
     * - если {@code needSitelinks} == null - пустые списки у всех баннеров
     * - если к баннеру не привязан набор сайтлинков - пустой список
     * - если к баннеру привязан набор сайтлинков - их список
     */
    public List<GdiBannerWithSitelinksForReplace> getBannerWithSitelinks(int shard, ClientId clientId,
                                                                         Collection<Long> bannerIds,
                                                                         boolean needSitelinks,
                                                                         boolean needDisplayHrefs) {
        List<GdiBannerWithSitelinksForReplace> banners = getBanners(shard, clientId, bannerIds);
        banners.forEach(b -> b.withSitelinks(emptyList()).withDisplayHref(null));

        if (needSitelinks) {
            appendSitelinks(shard, banners);
        } else {
            banners.forEach(b -> b.setSitelinks(emptyList()));
        }

        if (needDisplayHrefs) {
            appendDisplayHrefs(shard, banners);
        }

        return banners;
    }

    private void appendDisplayHrefs(int shard, List<GdiBannerWithSitelinksForReplace> banners) {
        var bannerIds = mapList(banners, GdiBannerWithSitelinksForReplace::getId);
        var displayHrefs = dslContextProvider.ppc(shard)
                .select(displayHrefBannerFieldsToRead)
                .from(BANNERS)
                .join(BANNER_DISPLAY_HREFS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .and(BANNERS.BID.in(bannerIds))
                .fetch(displayHrefBannerReader::fromDb);
        var hrefByBannerId = listToMap(displayHrefs, GdiSourceReplaceDisplayHrefBanner::getBannerId);
        banners.forEach(banner -> banner.withDisplayHref(ifNotNull(hrefByBannerId.get(banner.getId()),
                GdiSourceReplaceDisplayHrefBanner::getSourceDisplayHref)));
    }

    private void appendSitelinks(int shard, List<GdiBannerWithSitelinksForReplace> banners) {
        Set<Long> sitelinkSetIds = listToSet(banners, GdiBannerWithSitelinksForReplace::getSitelinkSetId);
        ListMultimap<Long, Sitelink> sitelinkSets =
                sitelinkSetRepository.getSitelinksBySetIds(shard, sitelinkSetIds);
        banners.forEach(b -> b.setSitelinks(cloneSitelinks(sitelinkSets.get(b.getSitelinkSetId()))));
    }

    private List<GdiBannerWithSitelinksForReplace> getBanners(int shard, ClientId clientId,
                                                              Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(bannerFieldsToRead)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .leftJoin(BANNER_TURBOLANDING_PARAMS).on(BANNER_TURBOLANDING_PARAMS.BID.eq(BANNERS.BID))
                .leftJoin(IMAGES).on(IMAGES.BID.eq(BANNERS.BID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()).and(BANNERS.BID.in(bannerIds)))
                .fetch(bannerReader::fromDb);
    }

    /**
     * Получить баннеры с отображаемой ссылкой по списку bannerId и списку отображаемых ссылок.
     *
     * @param shard              - шард
     * @param clientId           - id клиента
     * @param bannerIds          - список id баннеров
     * @param searchDisplayHrefs - список заменяемых отображаемых ссылок для поиска. Если null, то не фильтруем
     * @return список баннеров с отображаемой ссылкой
     */
    public List<GdiSourceReplaceDisplayHrefBanner> getBannersWithDisplayHrefs(
            int shard, ClientId clientId, Collection<Long> bannerIds, @Nullable Collection<String> searchDisplayHrefs) {
        SelectConditionStep<Record> selectStep = dslContextProvider.ppc(shard)
                .select(displayHrefBannerFieldsToRead)
                .from(BANNERS)
                .join(BANNER_DISPLAY_HREFS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(BANNERS.BID.in(bannerIds));

        if (searchDisplayHrefs != null) {
            selectStep.and(BANNER_DISPLAY_HREFS.DISPLAY_HREF.in(searchDisplayHrefs));
        }

        return selectStep.fetch(displayHrefBannerReader::fromDb);
    }

    /**
     * Получить все домены баннеров и соответствующих сайтлинков
     *
     * @param shard     шард
     * @param clientId  id клиента
     * @param bannerIds список id баннеров
     * @return список уникальных доменов
     */
    public Set<String> getBannerAndSitelinksDomains(
            int shard, ClientId clientId, Collection<Long> bannerIds) {
        Result<Record2<String, String>> fetch = dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.DOMAIN, SITELINKS_LINKS.HREF)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .leftJoin(SITELINKS_SETS).on(SITELINKS_SETS.SITELINKS_SET_ID.eq(BANNERS.SITELINKS_SET_ID))
                .leftJoin(SITELINKS_SET_TO_LINK)
                .on(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.eq(SITELINKS_SETS.SITELINKS_SET_ID))
                .leftJoin(SITELINKS_LINKS).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .where(BANNERS.BID.in(bannerIds).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))).fetch();

        return reduceDomainsToSet(fetch);
    }

    /**
     * Получает отображаемые ссылки всех указанных баннеров клиента
     *
     * @param shard     - шард
     * @param clientId  - id клиента
     * @param bannerIds - id баннеров
     * @return - уникальные отображаемые ссылки
     */
    public Set<String> getBannerDisplayHrefs(int shard, ClientId clientId, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_DISPLAY_HREFS.DISPLAY_HREF)
                .from(BANNER_DISPLAY_HREFS)
                .join(BANNERS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.in(bannerIds).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetchSet(BANNER_DISPLAY_HREFS.DISPLAY_HREF);
    }

    /**
     * Сложить список доменов в общий плоский список уникальных доменов.
     * null значения игнорируются.
     *
     * @param fetch исходные данные селекта
     * @return список уникальных доменов
     */
    private Set<String> reduceDomainsToSet(Result<Record2<String, String>> fetch) {
        Set<String> domains = new HashSet<>();
        for (Record record : fetch) {
            String bannerDomain = record.get(BANNERS.DOMAIN);
            String sitelinkDomain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(record.get(SITELINKS_LINKS.HREF));
            domains.add(bannerDomain);
            domains.add(sitelinkDomain);
        }
        domains.remove(null);
        return domains;
    }
}
