package ru.yandex.direct.grid.core.entity.banner.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.image.converter.BannerImageConverter;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.core.entity.banner.model.GdiAbstractBannerImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBannerImagesStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBigKingImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageAvatarsHost;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageNamespace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerLogo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerLogoStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiImagesImage;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_POOL;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGE_ASSET;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerLogos.BANNER_LOGOS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Repository
@ParametersAreNonnullByDefault
public class GridImageRepository {
    private final DslContextProvider dslContextProvider;

    private final JooqReaderWithSupplier<GdiImagesImage> imagesJooqReader;
    private final JooqReaderWithSupplier<GdiBannerImage> bannerImagesJooqReader;
    private final JooqReaderWithSupplier<GdiBannerLogo> bannerLogosJooqReader;
    private final JooqReaderWithSupplier<GdiBannerLogo> bannerLogosPoolJooqReader;
    private final JooqReaderWithSupplier<GdiBannerBigKingImage> bannerBigKingImagesJooqReader;

    @Autowired
    public GridImageRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        imagesJooqReader = getCommonImagesJooqReaderBuilder(GdiImagesImage::new)
                .readProperty(GdiImagesImage.BID, fromField(IMAGES.BID))
                .readProperty(GdiImagesImage.STATUS_MODERATE, fromField(IMAGES.STATUS_MODERATE)
                        .by(GdiBannerImageStatusModerate::fromSource))
                .readProperty(GdiImagesImage.IMAGE_HASH, fromField(IMAGES.IMAGE_HASH))
                .build();
        bannerImagesJooqReader = getCommonImagesJooqReaderBuilder(GdiBannerImage::new)
                .readProperty(GdiBannerImage.BID, fromField(BANNER_IMAGES.BID))
                .readProperty(GdiBannerImage.STATUS_MODERATE, fromField(BANNER_IMAGES.STATUS_MODERATE)
                        .by(GdiBannerBannerImagesStatusModerate::fromSource))
                .readProperty(GdiBannerImage.IMAGE_HASH, fromField(BANNER_IMAGES.IMAGE_HASH))
                .build();
        bannerLogosJooqReader = getCommonImagesJooqReaderBuilder(GdiBannerLogo::new)
                .readProperty(GdiBannerLogo.BID, fromField(Tables.BANNER_LOGOS.BID))
                .readProperty(GdiBannerLogo.STATUS_MODERATE, fromField(BANNER_LOGOS.STATUS_MODERATE)
                        .by(GdiBannerLogoStatusModerate::fromSource))
                .readProperty(GdiBannerLogo.IMAGE_HASH, fromField(BANNER_LOGOS.IMAGE_HASH))
                .build();
        bannerLogosPoolJooqReader = getCommonImagesJooqReaderBuilder(GdiBannerLogo::new)
                .readProperty(GdiBannerLogo.IMAGE_HASH, fromField(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                .build();
        bannerBigKingImagesJooqReader = getCommonImagesJooqReaderBuilder(GdiBannerBigKingImage::new)
                .readProperty(GdiBannerBigKingImage.BID, fromField(BANNER_IMAGE_ASSET.BID))
                .readProperty(GdiBannerBigKingImage.IMAGE_HASH, fromField(BANNER_IMAGE_ASSET.IMAGE_HASH))
                .build();
    }

    private static <M extends GdiAbstractBannerImage> JooqReaderWithSupplierBuilder<M> getCommonImagesJooqReaderBuilder(
            Supplier<M> modelSupplier) {
        return JooqReaderWithSupplierBuilder.builder(modelSupplier)
                .readProperty(GdiAbstractBannerImage.HEIGHT, fromField(BANNER_IMAGES_FORMATS.HEIGHT))
                .readProperty(GdiAbstractBannerImage.WIDTH, fromField(BANNER_IMAGES_FORMATS.WIDTH))
                .readProperty(GdiAbstractBannerImage.MDS_GROUP_ID, fromField(BANNER_IMAGES_FORMATS.MDS_GROUP_ID))
                .readProperty(GdiAbstractBannerImage.AVATARS_HOST, fromField(BANNER_IMAGES_FORMATS.AVATARS_HOST)
                        .by(GdiBannerImageAvatarsHost::fromSource))
                .readProperty(GdiAbstractBannerImage.NAMESPACE, fromField(BANNER_IMAGES_FORMATS.NAMESPACE)
                        .by(GdiBannerImageNamespace::fromSource))
                .readProperty(GdiAbstractBannerImage.IMAGE_TYPE,
                        fromField(BANNER_IMAGES_FORMATS.IMAGE_TYPE).by(ImageType::fromSource))
                .readProperty(GdiAbstractBannerImage.MDS_META,
                        fromField(BANNER_IMAGES_FORMATS.MDS_META).by(BannerImageConverter::toImageMdsMeta))
                .readProperty(GdiAbstractBannerImage.MDS_META_USER_OVERRIDE,
                        fromField(BANNER_IMAGES_POOL.MDS_META_USER_OVERRIDE).by(BannerImageConverter::toImageMdsMeta))
                .readProperty(GdiAbstractBannerImage.NAME, fromField(BANNER_IMAGES_POOL.NAME));
    }

    public Map<Long, GdiImagesImage> getImagesData(int shard, Map<Long, Collection<Long>> clientIdsToBannerIds) {
        List<GdiImagesImage> imageDataResult = new ArrayList<>();
        clientIdsToBannerIds.forEach((clientId, bannerIds) -> imageDataResult
                .addAll(dslContextProvider.ppc(shard).select(imagesJooqReader.getFieldsToRead())
                        .from(IMAGES)
                        .leftJoin(BANNER_IMAGES_FORMATS)
                        .on(IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                        .leftJoin(BANNER_IMAGES_POOL).on(IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                        .where(IMAGES.BID.in(bannerIds)
                                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId)))
                        .fetch(imagesJooqReader::fromDb))
        );

        return listToMap(imageDataResult, GdiImagesImage::getBid);
    }

    public Map<Long, GdiBannerLogo> getBannerLogosData(int shard, Map<Long, Collection<Long>> clientIdsToBannerIds) {
        List<GdiBannerLogo> imageDataResult = new ArrayList<>();
        clientIdsToBannerIds.forEach((clientId, bannerIds) -> imageDataResult
                .addAll(dslContextProvider.ppc(shard).select(bannerLogosJooqReader.getFieldsToRead())
                        .from(BANNER_LOGOS)
                        .leftJoin(BANNER_IMAGES_FORMATS)
                        .on(BANNER_LOGOS.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                        .leftJoin(BANNER_IMAGES_POOL).on(BANNER_LOGOS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                        .where(BANNER_LOGOS.BID.in(bannerIds)
                                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId)))
                        .fetch(bannerLogosJooqReader::fromDb))
        );

        return listToMap(imageDataResult, GdiBannerLogo::getBid);
    }

    public List<GdiBannerLogo> getBannerLogosByHash(int shard, ClientId clientId, Collection<String> imageHashes) {
        return dslContextProvider.ppc(shard).select(bannerLogosPoolJooqReader.getFieldsToRead())
                .from(BANNER_IMAGES_FORMATS)
                .leftJoin(BANNER_IMAGES_POOL).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(imageHashes)
                        .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong())))
                .fetch(bannerLogosPoolJooqReader::fromDb);
    }

    public Map<Long, GdiBannerImage> getBannerImagesData(int shard, Map<Long, Collection<Long>> clientIdsToBannerIds) {
        List<GdiBannerImage> imageDataResult = new ArrayList<>();
        clientIdsToBannerIds.forEach((clientId, bannerIds) -> imageDataResult
                .addAll(dslContextProvider.ppc(shard).select(bannerImagesJooqReader.getFieldsToRead())
                        .from(BANNER_IMAGES)
                        .leftJoin(BANNER_IMAGES_FORMATS)
                        .on(BANNER_IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                        .leftJoin(BANNER_IMAGES_POOL)
                        .on(BANNER_IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                        .where(BANNER_IMAGES.BID.in(bannerIds)
                                .and(BANNER_IMAGES.STATUS_SHOW.eq(BannerImagesStatusshow.Yes))
                                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId)))
                        .fetch(bannerImagesJooqReader::fromDb))
        );

        return listToMap(imageDataResult, GdiBannerImage::getBid);
    }

    public Map<Long, GdiBannerBigKingImage> getBannerBigKingImagesData(int shard, Map<Long, Collection<Long>> clientIdsToBannerIds) {
        List<GdiBannerBigKingImage> imageDataResult = new ArrayList<>();
        clientIdsToBannerIds.forEach((clientId, bannerIds) -> imageDataResult
                .addAll(dslContextProvider.ppc(shard).select(bannerBigKingImagesJooqReader.getFieldsToRead())
                        .from(BANNER_IMAGE_ASSET)
                        .leftJoin(BANNER_IMAGES_FORMATS)
                        .on(BANNER_IMAGE_ASSET.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH))
                        .leftJoin(BANNER_IMAGES_POOL).on(BANNER_IMAGE_ASSET.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                        .where(BANNER_IMAGE_ASSET.BID.in(bannerIds)
                                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId)))
                .fetch(bannerBigKingImagesJooqReader::fromDb))
        );

        return listToMap(imageDataResult, GdiBannerBigKingImage::getBid);
    }
}
