package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.banner.model.Age;
import ru.yandex.direct.core.entity.banner.model.BabyFood;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.AGE;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.BABY_FOOD;

@ParametersAreNonnullByDefault
class AdModelChangesBuilder {
    static String ageFlag = AGE.getKey();
    static String babyFoodFlag = BABY_FOOD.getKey();

    static List<ModelChanges<Banner>> buildModelChangesForBannerWithImageOrBannerImage(
            Map<Long, ? extends Class<? extends Banner>> adClassByAdId,
            String imageHash) {
        return EntryStream.of(adClassByAdId)
                .mapKeyValue((adId, adClass) -> {
                    if (BannerWithBannerImage.class.isAssignableFrom(adClass)) {
                        return buildModelChangesForBannerWithBannerImage(adId, imageHash);
                    } else {
                        return buildModelChangesForBannerWithImage(adId, imageHash);
                    }
                })
                .collect(Collectors.toList());
    }

    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    static List<ModelChanges<BannerWithSystemFields>> buildModelChangesWithFlags(
            Map<Long, BannerFlags> adsFlagsFromDbById,
            Map<Long, ? extends Class<? extends BannerWithSystemFields>> adClassesByAdId,
            Optional<Age> ageValue, Optional<BabyFood> babyFoodValue) {
        return EntryStream.of(adClassesByAdId)
                .mapKeyValue((adId, adClass) ->
                        buildModelChangesWithFlags(adId, adClass, ageValue, babyFoodValue,
                                adsFlagsFromDbById.get(adId)))
                .toList();
    }

    private static ModelChanges<Banner> buildModelChangesForBannerWithBannerImage(
            Long adId,
            String imageHash) {
        return ModelChanges
                .build(adId, BannerWithBannerImage.class, BannerWithBannerImage.IMAGE_HASH, imageHash)
                .castModelUp(Banner.class);
    }

    private static ModelChanges<Banner> buildModelChangesForBannerWithImage(
            Long adId,
            String imageHash) {
        return ModelChanges
                .build(adId, BannerWithImage.class, BannerWithImage.IMAGE_HASH, imageHash)
                .castModelUp(Banner.class);
    }

    /**
     * В Директе можно только менять возрастную метку, но не добавлять-удaлять флаги.
     */
    @SuppressWarnings("unchecked")
    private static ModelChanges<BannerWithSystemFields> buildModelChangesWithFlags(
            Long adId,
            Class<? extends BannerWithSystemFields> adClass,
            @Nullable Optional<Age> ageValue,
            @Nullable Optional<BabyFood> babyFoodValue,
            @Nullable BannerFlags adFlagsFromDb) {
        checkState(BannerWithFlags.class.isAssignableFrom(adClass), "Banner class doesn't support flags");

        if (adFlagsFromDb != null) {
            String babyFoodValueFromDb = adFlagsFromDb.getFlags().get(babyFoodFlag);
            String ageValueFromDb = adFlagsFromDb.getFlags().get(ageFlag);

            if (babyFoodValueFromDb != null && babyFoodValue != null && babyFoodValue.isPresent()) {
                adFlagsFromDb.getFlags().put(babyFoodFlag, babyFoodValue.get().getTypedValue());
            }

            if (ageValueFromDb != null && ageValue != null && ageValue.isPresent()) {
                adFlagsFromDb.getFlags().put(ageFlag, ageValue.get().getValue());
            }
        }

        Map<String, String> resultFlags = new HashMap<>();

        if (adFlagsFromDb != null) {
            resultFlags.putAll(adFlagsFromDb.getFlags());
        }

        return ModelChanges
                .build(adId, adClass, BannerWithFlags.FLAGS, new BannerFlags().withFlags(resultFlags))
                .castModelUp(BannerWithSystemFields.class);
    }
}
