package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Age;
import ru.yandex.direct.core.entity.banner.model.BabyFood;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperation;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.banner.GdiAdMassChangeResult;
import ru.yandex.direct.grid.core.entity.banner.service.validation.GridAdMassChangeValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.BANNER_CLASS_TO_TYPE;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentBannerType;
import static ru.yandex.direct.grid.core.entity.banner.service.AdModelChangesBuilder.buildModelChangesForBannerWithImageOrBannerImage;
import static ru.yandex.direct.grid.core.entity.banner.service.AdModelChangesBuilder.buildModelChangesWithFlags;
import static ru.yandex.direct.grid.core.entity.banner.service.GridAdMassChangeUtils.getIndexMap;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@ParametersAreNonnullByDefault
@Service
public class GridAdsMassChangesService {

    private final ShardHelper shardHelper;
    private final BannerService bannerService;
    private final BannerRepository bannerRepository;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final GridAdMassChangeValidationService gridAdMassChangeValidationService;

    @Autowired
    public GridAdsMassChangesService(ShardHelper shardHelper,
                                     BannerService bannerService,
                                     BannerRepository bannerRepository,
                                     BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                     GridAdMassChangeValidationService gridAdMassChangeValidationService) {
        this.shardHelper = shardHelper;
        this.bannerService = bannerService;
        this.bannerRepository = bannerRepository;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.gridAdMassChangeValidationService = gridAdMassChangeValidationService;
    }

    public GdiAdMassChangeResult addBannerImage(Long opUid, ClientId clientId, List<Long> adIds,
                                                String imageHash) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var banners = bannerRepository.type.getSafely(shard, adIds, BannerWithSystemFields.class);
        ValidationResult<List<BannerWithSystemFields>, Defect> vr =
                gridAdMassChangeValidationService.validateForImagesUpdate(adIds, banners);
        Map<Integer, BannerWithSystemFields> validItemsByIndex = ValidationResult.getValidItemsWithIndex(vr);
        if (validItemsByIndex.isEmpty()) {
            return buildEmptyGdiAdMassChangeResult(vr);
        }

        var bannerClassById = getBannerClassById(vr);

        List<ModelChanges<Banner>> modelChanges =
                buildModelChangesForBannerWithImageOrBannerImage(bannerClassById, imageHash);

        BannersUpdateOperation<Banner> partialUpdateOperation =
                bannersUpdateOperationFactory.createPartialUpdateOperation(ModerationMode.DEFAULT, modelChanges, opUid,
                        clientId, shard, Banner.class, null);

        MassResult<Long> massUpdatedResult = partialUpdateOperation
                .prepareAndApply();
        return buildGdiAdMassChangeResult(validItemsByIndex, massUpdatedResult, vr);
    }

    public GdiAdMassChangeResult addCreative(Long opUid, ClientId clientId, List<Long> adIds,
                                             BannersBannerType bannerTypeRequested,
                                             Set<AdGroupType> adGroupTypeRequested,
                                             Long creativeId, Boolean showTitleAndBody) {
        List<ModelChanges<BannerWithCreative>> modelChanges = adIds.stream()
                .map(adId -> ModelChanges
                        .build(adId, BannerWithCreative.class, BannerWithCreative.CREATIVE_ID, creativeId)
                        .process(showTitleAndBody, BannerWithCreative.SHOW_TITLE_AND_BODY)
                )
                .collect(Collectors.toList());

        BannersUpdateOperation<BannerWithCreative> partialUpdateOperation =
                bannersUpdateOperationFactory.createPartialUpdateOperation(ModerationMode.DEFAULT, modelChanges, opUid,
                        clientId, BannerWithCreative.class,
                        getModelChangesIsApplicableConstraintCreator(bannerTypeRequested, adGroupTypeRequested,
                                BannerWithCreative.class));

        MassResult<Long> massUpdatedResult = partialUpdateOperation
                .prepareAndApply();

        return buildGdiAdMassChangeResult(massUpdatedResult);
    }

    public GdiAdMassChangeResult addSitelinkSet(Long opUid, ClientId clientId, List<Long> adIds, Long sitelinkSetId) {
        List<ModelChanges<BannerWithSitelinks>> modelChanges = adIds.stream()
                .map(adId -> ModelChanges.build(adId, BannerWithSitelinks.class,
                        BannerWithSitelinks.SITELINKS_SET_ID, sitelinkSetId))
                .collect(Collectors.toList());

        MassResult<Long> massUpdatedResult = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, opUid, clientId, BannerWithSitelinks.class)
                .prepareAndApply();

        return buildGdiAdMassChangeResult(massUpdatedResult);
    }

    public GdiAdMassChangeResult addMulticards(Long opUid, ClientId clientId, List<Long> adIds, List<BannerMulticard> multicards) {
        List<ModelChanges<BannerWithMulticardSet>> modelChanges = adIds.stream()
                .map(adId -> ModelChanges.build(adId, BannerWithMulticardSet.class,
                        BannerWithMulticardSet.MULTICARDS, multicards))
                .collect(Collectors.toList());

        MassResult<Long> massUpdatedResult = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, opUid, clientId, BannerWithMulticardSet.class)
                .prepareAndApply();

        return buildGdiAdMassChangeResult(massUpdatedResult);
    }

    public GdiAdMassChangeResult addTurboLanding(Long opUid, ClientId clientId, List<Long> adIds, Long turboLandingId) {
        List<ModelChanges<BannerWithTurboLanding>> modelChanges = adIds.stream()
                .map(adId -> ModelChanges.build(adId, BannerWithTurboLanding.class,
                        BannerWithTurboLanding.TURBO_LANDING_ID, turboLandingId))
                .collect(Collectors.toList());

        MassResult<Long> massUpdatedResult = bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, opUid, clientId, BannerWithTurboLanding.class)
                .prepareAndApply();

        return buildGdiAdMassChangeResult(massUpdatedResult);
    }

    public GdiAdMassChangeResult updateAdAgeFlags(Long opUid, ClientId clientId, List<Long> adIds,
                                                  Optional<Age> ageValue, Optional<BabyFood> babyFoodValue) {
        List<BannerWithSystemFields> banners = bannerService.getBannersByIds(adIds);

        Map<Long, BannerFlags> adsFlagsFromDbById =
                listToMap(banners, BannerWithSystemFields::getId, banner -> nvl(banner.getFlags(),
                        new BannerFlags()));
        ValidationResult<List<BannerWithSystemFields>, Defect> vr =
                gridAdMassChangeValidationService.validateWithAdTypeCheck(adIds,
                        banners, List.of(TextBanner.class, MobileAppBanner.class));
        Map<Integer, BannerWithSystemFields> validItemsByIndex = ValidationResult.getValidItemsWithIndex(vr);

        if (validItemsByIndex.isEmpty()) {
            return buildEmptyGdiAdMassChangeResult(vr);
        }

        Map<Long, ? extends Class<? extends BannerWithSystemFields>> bannerClassesById = getBannerClassById(vr);

        List<ModelChanges<BannerWithSystemFields>> modelChanges = buildModelChangesWithFlags(adsFlagsFromDbById,
                bannerClassesById,
                ageValue, babyFoodValue);

        MassResult<Long> massUpdatedResult = updateBanners(opUid, clientId, modelChanges);

        return buildGdiAdMassChangeResult(validItemsByIndex, massUpdatedResult, vr);
    }

    private Map<Long, ? extends Class<? extends BannerWithSystemFields>> getBannerClassById(ValidationResult<List<BannerWithSystemFields>, Defect> vr) {
        List<BannerWithSystemFields> validItems = ValidationResult.getValidItems(vr);
        return listToMap(validItems, BannerWithSystemFields::getId, BannerWithSystemFields::getClass);
    }

    private MassResult<Long> updateBanners(Long opUid, ClientId clientId,
                                           List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        return bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, opUid, clientId)
                .prepareAndApply();
    }

    private GdiAdMassChangeResult buildEmptyGdiAdMassChangeResult(ValidationResult<List<BannerWithSystemFields>,
            Defect> vr) {
        return new GdiAdMassChangeResult(Collections.emptySet(), vr);
    }

    private GdiAdMassChangeResult buildGdiAdMassChangeResult(Map<Integer, BannerWithSystemFields> validItemsByIndex,
                                                             MassResult<Long> massUpdatedResult,
                                                             ValidationResult<List<BannerWithSystemFields>,
                                                                     Defect> vr) {
        Map<Integer, Integer> destToSourceIndexMap = getIndexMap(validItemsByIndex);
        mergeSubListValidationResults(vr,
                (ValidationResult<List<BannerWithSystemFields>, Defect>) massUpdatedResult.getValidationResult(),
                destToSourceIndexMap);
        //todo ssdmitriev: DIRECT-96727 переиспользовать getSuccessfullyResults
        Set<Long> successfullyAddedIds = StreamEx.of(massUpdatedResult.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .toSet();
        return new GdiAdMassChangeResult(successfullyAddedIds, vr);
    }

    private static GdiAdMassChangeResult buildGdiAdMassChangeResult(MassResult<Long> massUpdatedResult) {
        //noinspection unchecked
        var vr = (ValidationResult<List<BannerWithSystemFields>, Defect>)
                massUpdatedResult.getValidationResult();

        Set<Long> successfullyAddedIds = StreamEx.of(massUpdatedResult.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .toSet();

        return new GdiAdMassChangeResult(successfullyAddedIds, vr);
    }

    private static <T extends Banner> Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
    getModelChangesIsApplicableConstraintCreator(BannersBannerType bannerTypeRequested,
                                                 Set<AdGroupType> adGroupTypeRequested,
                                                 Class<T> modelChangesClass) {
        return container -> Constraint.fromPredicate(mc -> {
            AdGroupForBannerOperation adgroup = container.getBannerIdToAdgroupMap().get(mc.getId());
            return adGroupTypeRequested.contains(adgroup.getType())
                    && BANNER_CLASS_TO_TYPE.get(container.getRuntimeClass(mc.getId())) == bannerTypeRequested;
        }, inconsistentBannerType());
    }

}
