package ru.yandex.direct.grid.core.entity.banner.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithButton;
import ru.yandex.direct.core.entity.banner.model.BannerWithContentPromotion;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet;
import ru.yandex.direct.core.entity.banner.model.BannerWithName;
import ru.yandex.direct.core.entity.banner.model.ButtonAction;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.TurboAppMetaContent;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.banner.service.BannerTurboAppService;
import ru.yandex.direct.core.entity.banner.type.additionalhrefs.BannerAdditionalHrefsRepository;
import ru.yandex.direct.core.entity.banner.type.banneradditions.BannerAdditionsRepository;
import ru.yandex.direct.core.entity.banner.type.measurers.BannerMeasurersRepository;
import ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsRepository;
import ru.yandex.direct.core.entity.banner.type.price.BannerPriceRepository;
import ru.yandex.direct.core.entity.banner.type.turbogallery.BannerTurboGalleriesRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.organization.model.BannerPermalink;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBigKingImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerButton;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerContentPromotion;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerCreative;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerDisplayHref;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerFilter;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerLogo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerMobileContentInfo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerOrderBy;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerTurboLanding;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannersWithTotals;
import ru.yandex.direct.grid.core.entity.banner.model.GdiButtonAction;
import ru.yandex.direct.grid.core.entity.banner.model.GdiImagesImage;
import ru.yandex.direct.grid.core.entity.banner.model.GdiInternalBannerExtraInfo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSitelink;
import ru.yandex.direct.grid.core.entity.banner.repository.GridBannerAdditionsRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridBannerDomainRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridBannerMobileContentRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridBannerRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridBannerYtRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridCreativeRepository;
import ru.yandex.direct.grid.core.entity.banner.repository.GridImageRepository;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.AdFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.group.service.GridAdGroupConstants;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.model.ModelWithId;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerIdFilter;
import static ru.yandex.direct.core.entity.organization.model.PermalinkAssignType.AUTO;
import static ru.yandex.direct.core.entity.organization.model.PermalinkAssignType.MANUAL;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для получения данных и статистики баннеров
 */
@Service
@ParametersAreNonnullByDefault
public class GridBannerService {
    private final ShardHelper shardHelper;
    private final GridBannerYtRepository gridBannerYtRepository;
    private final GridBannerRepository gridBannerRepository;
    private final GridImageRepository gridImageRepository;
    private final GridCreativeRepository gridCreativeRepository;
    private final GridBannerDomainRepository gridBannerDomainRepository;
    private final BannerAdditionsRepository bannerAdditionsRepository;
    private final CalloutRepository calloutRepository;
    private final CreativeRepository creativeRepository;
    private final GridBannerAdditionsRepository gridBannerAdditionsRepository;
    private final BannerTurboGalleriesRepository turboGalleriesRepository;
    private final BannerPixelsRepository bannerPixelsRepository;
    private final BannerPriceRepository bannerPriceRepository;
    private final OrganizationRepository organizationRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final ContentPromotionRepository contentPromotionRepository;
    private final GridInternalBannerExtraInfoService gridInternalBannerExtraInfoService;
    private final BannerMeasurersRepository bannerMeasurersRepository;
    private final ModerateBannerPagesRepository moderateBannerPagesRepository;
    private final BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository;
    private final OrganizationService organizationService;
    private final BannerModerationRepository bannerModerationRepository;
    private final BannerTurboAppService bannerTurboAppService;
    private final GridBannerMobileContentRepository gridBannerMobileContentRepository;
    private final ClientPhoneRepository clientPhoneRepository;
    private final MetrikaGoalsService metrikaGoalsService;

    @Autowired
    public GridBannerService(ShardHelper shardHelper,
                             GridBannerYtRepository gridBannerYtRepository,
                             GridBannerRepository gridBannerRepository,
                             GridImageRepository gridImageRepository,
                             GridCreativeRepository gridCreativeRepository,
                             GridBannerDomainRepository gridBannerDomainRepository,
                             BannerAdditionsRepository bannerAdditionsRepository,
                             CalloutRepository calloutRepository,
                             CreativeRepository creativeRepository,
                             GridBannerAdditionsRepository gridBannerAdditionsRepository,
                             BannerTurboGalleriesRepository turboGalleriesRepository,
                             BannerPixelsRepository bannerPixelsRepository,
                             BannerPriceRepository bannerPriceRepository,
                             OrganizationRepository organizationRepository,
                             BannerTypedRepository bannerTypedRepository,
                             ContentPromotionRepository contentPromotionRepository,
                             GridInternalBannerExtraInfoService gridInternalBannerExtraInfoService,
                             BannerMeasurersRepository bannerMeasurersRepository,
                             ModerateBannerPagesRepository moderateBannerPagesRepository,
                             BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository,
                             OrganizationService organizationService,
                             BannerModerationRepository bannerModerationRepository,
                             GridBannerMobileContentRepository gridBannerMobileContentRepository,
                             ClientPhoneRepository clientPhoneRepository,
                             BannerTurboAppService bannerTurboAppService,
                             MetrikaGoalsService metrikaGoalsService) {
        this.shardHelper = shardHelper;
        this.gridBannerYtRepository = gridBannerYtRepository;
        this.gridBannerRepository = gridBannerRepository;
        this.gridImageRepository = gridImageRepository;
        this.gridCreativeRepository = gridCreativeRepository;
        this.gridBannerDomainRepository = gridBannerDomainRepository;
        this.bannerAdditionsRepository = bannerAdditionsRepository;
        this.calloutRepository = calloutRepository;
        this.creativeRepository = creativeRepository;
        this.gridBannerAdditionsRepository = gridBannerAdditionsRepository;
        this.turboGalleriesRepository = turboGalleriesRepository;
        this.bannerPixelsRepository = bannerPixelsRepository;
        this.bannerPriceRepository = bannerPriceRepository;
        this.organizationRepository = organizationRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.contentPromotionRepository = contentPromotionRepository;
        this.gridInternalBannerExtraInfoService = gridInternalBannerExtraInfoService;
        this.bannerMeasurersRepository = bannerMeasurersRepository;
        this.moderateBannerPagesRepository = moderateBannerPagesRepository;
        this.bannerAdditionalHrefsRepository = bannerAdditionalHrefsRepository;
        this.organizationService = organizationService;
        this.bannerModerationRepository = bannerModerationRepository;
        this.gridBannerMobileContentRepository = gridBannerMobileContentRepository;
        this.bannerTurboAppService = bannerTurboAppService;
        this.clientPhoneRepository = clientPhoneRepository;
        this.metrikaGoalsService = metrikaGoalsService;
    }

    /**
     * Получить данные о баннерах. Фильтруем и сортируем в YT, достаем вместе со статистикой, после чего, из MySQL
     * дочитываем данные необходимые для последующего отображения в интерфейсе.
     * <p>
     * При добавлении новых фильтров в коде нужно так же их учесть в методе:
     * {@link ru.yandex.direct.grid.processing.service.banner.BannerDataConverter#hasAnyCodeFilter}
     *
     * @param shard                   шард, в котором хранятся баннеры (исключительно для улучшения результатов запроса)
     * @param operatorUid             uid оператора
     * @param clientId                идентификатор клиента для которого достаем баннеры
     * @param filter                  настройки фильтрации выбранных баннеров
     * @param bannerOrderByList       настройки упорядочивания баннеров
     * @param statStartDay            начало периода, за который мы получаем статистику по баннеров
     * @param statEndDay              конец периода, за который мы получаем статистику по баннеров
     * @param statByDaysFrom          начало периода, за который нужно получать статистику по дням
     * @param statByDaysTo            конец периода, за который нужно получать статистику по дням (включительно)
     * @param isFlat                  статистика показов: true - РСЯ, false - Поиск, null - оба варината
     * @param goalIds                 идентификаторы целей
     * @param adFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля запрошены на
     * @param disableStatusFilter     не фильтровать по статусам
     */
    public GdiBannersWithTotals getBanners(int shard, Long operatorUid, ClientId clientId, Set<String> clientFeatures,
                                           GdiBannerFilter filter, List<GdiBannerOrderBy> bannerOrderByList,
                                           LocalDate statStartDay, LocalDate statEndDay, @Nullable Boolean isFlat,
                                           Set<Long> goalIds,
                                           @Nullable LocalDate statByDaysFrom, @Nullable LocalDate statByDaysTo,
                                           AdFetchedFieldsResolver adFetchedFieldsResolver,
                                           boolean disableStatusFilter) {
        //Если для групп достигли лимита, то объявления в базе не фильтруем по adGroupIdIn. Подробнее тут: DIRECT-81384
        if (filter.getAdGroupIdIn() != null && filter.getAdGroupIdIn().size() >= GridAdGroupConstants.getMaxGroupRows()) {
            checkState(isNotEmpty(filter.getCampaignIdIn()),
                    "campaignIdIn from filter must be not empty, when count of adGroupIdIn over limit");
            filter.setAdGroupIdIn(null);
        }

        boolean getRevenueOnlyByAvailableGoals =
                clientFeatures.contains(FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS.getName());
        Set<Long> availableGoalIds = null;
        if (CollectionUtils.isNotEmpty(goalIds) && getRevenueOnlyByAvailableGoals) {
            availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                    operatorUid, clientId);
        }
        GdiBannersWithTotals bannersWithTotals = gridBannerYtRepository
                .getBanners(shard, filter, bannerOrderByList, statStartDay, statEndDay, isFlat,
                        limited(GridBannerConstants.getMaxBannerRows()), goalIds, availableGoalIds,
                        statByDaysFrom, statByDaysTo, adFetchedFieldsResolver,
                        clientFeatures, disableStatusFilter);
        List<GdiBanner> banners = bannersWithTotals.getGdiBanners();
        if (banners.isEmpty()) {
            return bannersWithTotals;
        }

        addExternalDataToGdiBanner(shard, banners, clientId, clientFeatures, true);

        if (filter.getTurbolandingsExist() != null) {
            banners = banners.stream()
                    .filter(banner -> filter.getTurbolandingsExist() == hasTurbolandings(banner))
                    .collect(toList());
        }
        return bannersWithTotals.withGdiBanners(banners);
    }

    /**
     * Получить данные о баннерах. Достаем баннеры по идентификатору из MySQL без статистики с данными необходимыми
     * для последующего отображения в интерфейсе.
     *
     * @param shard     шард, в котором хранятся баннеры
     * @param bannerIds идентификаторы баннеров, которые надо прочитать
     */
    public List<GdiBanner> getBannersWithoutStats(int shard, Collection<Long> bannerIds,
                                                  ClientId clientId, Set<String> clientFeatures,
                                                  boolean addAutoAssignedPermalinks) {
        List<GdiBanner> banners = gridBannerRepository.getBanners(shard, bannerIds, clientFeatures);
        if (banners.isEmpty()) {
            return Collections.emptyList();
        }
        banners.forEach(gdiBanner -> gdiBanner.setStat(GridStatNew.addZeros(new GdiEntityStats())));
        addExternalDataToGdiBanner(shard, banners, clientId, clientFeatures, addAutoAssignedPermalinks);
        return banners;
    }


    /**
     * Добавить в баннеры дополнительную информацию о связанных объектах.
     *
     * @param shard                     шард, в котором хранятся баннеры
     * @param banners                   баннеры, которые надо обогатить
     * @param clientId                  идентификатор клиента
     * @param addAutoAssignedPermalinks если {@code true}, то баннеру могут быть добавлены автоматически привязанные
     *                                  организации (будет сделан запрос в справочник для проверки опубликованности)
     */
    void addExternalDataToGdiBanner(int shard, List<GdiBanner> banners,
                                    ClientId clientId, Set<String> clientFeatures,
                                    boolean addAutoAssignedPermalinks) {
        List<Long> bannerIds = mapList(banners, GdiBanner::getId);
        Map<Long, Collection<Long>> clientIdToBannerIds = ImmutableMap.of(clientId.asLong(), bannerIds);
        Map<Long, GdiImagesImage> images = gridImageRepository.getImagesData(shard, clientIdToBannerIds);
        Map<Long, GdiBannerImage> bannerImages = gridImageRepository.getBannerImagesData(shard, clientIdToBannerIds);
        Map<Long, GdiBannerLogo> bannerLogos = gridImageRepository.getBannerLogosData(shard, clientIdToBannerIds);
        Map<Long, GdiBannerBigKingImage> bannerBigKingImages = gridImageRepository.getBannerBigKingImagesData(shard,
                clientIdToBannerIds);
        Map<Long, GdiBannerCreative> gdiCreatives = gridCreativeRepository.getCreativesData(shard, bannerIds);
        Map<Long, Long> bannerIdsToCreativesId = EntryStream.of(gdiCreatives)
                .mapValues(GdiBannerCreative::getCreativeId)
                .toMap();
        Map<Long, Creative> creatives = getCreatives(shard, bannerIdsToCreativesId);
        Map<Long, String> disclaimers = bannerAdditionsRepository.getAdditionDisclaimerByBannerIds(shard, bannerIds);
        Map<Long, GdiBannerDisplayHref> displayHrefs = gridBannerAdditionsRepository.getBannersDisplayHref(shard,
                bannerIds);
        Map<Long, List<GdiSitelink>> sitelinks = gridBannerAdditionsRepository.getSitelinks(shard, banners);
        Map<Long, List<Callout>> callouts = calloutRepository.getExistingCalloutsByBannerIds(shard, bannerIds);
        Map<Long, String> turboGalleryByBannerIds = getTurboGalleryByBannerIds(clientId, bannerIds);
        Map<Long, GdiBannerTurboLanding> bannerTurbolandings =
                gridBannerAdditionsRepository.getBannerTurbolandingsByBannerId(shard, bannerIds);
        Map<Long, String> turbolandingParams = gridBannerAdditionsRepository.getBannerTurbolandingParams(shard,
                bannerIds);
        Map<Long, List<String>> bannerPixels = bannerPixelsRepository.getPixelsByBannerIds(shard, bannerIds);
        Map<Long, BannerPrice> bannerPrices = bannerPriceRepository.getBannerPricesByBannerIds(shard, bannerIds);
        Map<Long, GdiInternalBannerExtraInfo> internalAdExtraInfoByBannerId =
                gridInternalBannerExtraInfoService.getInternalBannerExtraInfoByBannerId(shard, clientId, bannerIds);
        Map<Long, List<BannerMeasurer>> measurersByBannerIds =
                bannerMeasurersRepository.getMeasurersByBannerIds(shard, bannerIds);
        Map<Long, String> tnsIdsByBannerIds = bannerPixelsRepository.getTnsByIds(shard, bannerIds);
        Map<Long, List<ModerateBannerPage>> moderateBannerPages =
                moderateBannerPagesRepository.getModerateBannerPages(shard, bannerIds);

        Map<Long, List<BannerPermalink>> bannerPermalinks =
                organizationRepository.getBannerPermalinkByBannerIds(shard, bannerIds);
        boolean isShowAutoAssignedOrganizations = addAutoAssignedPermalinks
                && clientFeatures.contains(FeatureName.SHOW_AUTO_ASSIGNED_ORGANIZATIONS.getName());
        Map<Long, BannerPermalink> displayedBannerPermalinks =
                getDisplayedBannerPermalinks(bannerPermalinks, isShowAutoAssignedOrganizations);

        Map<Long, List<BannerAdditionalHref>> bannerAdditionalHrefs =
                bannerAdditionalHrefsRepository.getAdditionalHrefs(shard, bannerIds);

        var allBanners = bannerTypedRepository.getSafely(shard, bannerIdFilter(bannerIds), List.of(
                BannerWithButton.class,
                BannerWithContentPromotion.class,
                BannerWithName.class,
                BannerWithMulticardSet.class,
                BannerWithCreative.class
        ));
        var bannersWithButtons =
                StreamEx.of(allBanners).select(BannerWithButton.class).toMap(ModelWithId::getId, identity());
        var bannerWithContentPromotionMap =
                StreamEx.of(allBanners).select(BannerWithContentPromotion.class).toMap(ModelWithId::getId, identity());
        var bannerWithNameMap =
                StreamEx.of(allBanners).select(BannerWithName.class).toMap(ModelWithId::getId, identity());
        var bannerIdsToShowTitleAndBody =
                StreamEx.of(allBanners)
                        .select(BannerWithCreative.class)
                        .filter(t -> nvl(t.getShowTitleAndBody(), false))
                        .map(BannerWithCreative::getId)
                        .toSet();//переделать на yt и выгрузку, чекнуть, что приходят не null в getShowTitleAndBody

        Map<Long, BannerMulticardSetStatusModerate> bannerMulticardsStatusModerate = StreamEx.of(allBanners)
                .select(BannerWithMulticardSet.class)
                .mapToEntry(ModelWithId::getId, BannerWithMulticardSet::getMulticardSetStatusModerate)
                .filterValues(Objects::nonNull)
                .toMap();
        Map<Long, List<BannerMulticard>> bannerMulticards = StreamEx.of(allBanners)
                .select(BannerWithMulticardSet.class)
                .mapToEntry(ModelWithId::getId, BannerWithMulticardSet::getMulticards)
                .filterValues(Objects::nonNull)
                .toMap();

        List<Long> outdoorBannerIds = filterAndMapList(banners, b -> b.getBannerType() == BannersBannerType.cpm_outdoor,
                GdiBanner::getId);
        Map<Long, List<Long>> currentBannersMinusGeo =
                bannerModerationRepository.getCurrentBannersMinusGeo(shard, outdoorBannerIds);

        Map<Long, ContentPromotionContent> contentPromotionByBannerId =
                getContentPromotionContentByBannerId(shard, bannerWithContentPromotionMap.values());

        Map<Long, Long> phoneIdsByBannerIds = clientPhoneRepository.getPhoneIdsByBannerIds(shard, bannerIds);

        boolean isTurboAppsAllowed = clientFeatures.contains(FeatureName.TURBO_APP_ALLOWED.getName());
        Map<Long, TurboAppMetaContent> turboApps;
        if (isTurboAppsAllowed) {
            Map<Long, Long> cidsByBannerIds = StreamEx.of(banners).toMap(GdiBanner::getId, GdiBanner::getCampaignId);
            turboApps = bannerTurboAppService.updateAndGetTurboApps(shard, clientId, cidsByBannerIds);
        } else {
            turboApps = Collections.emptyMap();
        }

        Set<Long> mobileContentBannerIds = StreamEx.of(banners)
                .filter(b -> b.getBannerType() == BannersBannerType.mobile_content)
                .map(GdiBanner::getId)
                .toSet();
        Map<Long, GdiBannerMobileContentInfo> bannerIdToMobileContentInfo =
                gridBannerMobileContentRepository.getMobileContentInfoByBannerId(shard, mobileContentBannerIds);

        banners.forEach(banner -> {
                    banner.withBannerImage(bannerImages.get(banner.getId()))
                            .withImage(images.get(banner.getId()))
                            .withBannerLogo(bannerLogos.get(banner.getId()))
                            .withBigKingImage(bannerBigKingImages.get(banner.getId()))
                            .withCreative(gdiCreatives.get(banner.getId()))
                            .withTypedCreative(creatives.get(banner.getId()))
                            .withShowTitleAndBody(bannerIdsToShowTitleAndBody.contains(banner.getId()))
                            .withDynamicDisclaimer(disclaimers.get(banner.getId()))
                            // todo DIRECT-101180 избавиться от поля linkTail т.к. есть более подробное displayHref
                            .withLinkTail(ifNotNull(displayHrefs.get(banner.getId()), GdiBannerDisplayHref::getHref))
                            .withDisplayHref(displayHrefs.get(banner.getId()))
                            .withSitelinks(sitelinks.get(banner.getId()))
                            .withCallouts(callouts.get(banner.getId()))
                            .withTurboGalleryHref(turboGalleryByBannerIds.get(banner.getId()))
                            .withTurbolanding(bannerTurbolandings.get(banner.getId()))
                            .withTurbolandingParams(turbolandingParams.get(banner.getId()))
                            .withPixels(bannerPixels.get(banner.getId()))
                            .withBannerPrice(bannerPrices.get(banner.getId()))
                            .withInternalAdExtraInfo(internalAdExtraInfoByBannerId.get(banner.getId()))
                            .withMeasurers(measurersByBannerIds.get(banner.getId()))
                            .withTnsId(tnsIdsByBannerIds.get(banner.getId()))
                            .withPlacementPages(moderateBannerPages.get(banner.getId()))
                            .withCurrentMinusGeo(currentBannersMinusGeo.get(banner.getId()))
                            .withBannerContentPromotion(
                                    toGdiBannerContentPromotion(bannerWithContentPromotionMap.get(banner.getId())))
                            .withContentPromotion(contentPromotionByBannerId.get(banner.getId()))
                            .withAdditionalHrefs(bannerAdditionalHrefs.get(banner.getId()))
                            .withTurboApp(turboApps.get(banner.getId()))
                            .withPhoneId(phoneIdsByBannerIds.get(banner.getId()))
                            .withButton(toGdiBannerButton(bannersWithButtons.get(banner.getId())))
                            .withName(ifNotNull(bannerWithNameMap.get(banner.getId()), BannerWithName::getName))
                            .withMulticardSetStatusModerate(bannerMulticardsStatusModerate.get(banner.getId()))
                            .withMulticards(bannerMulticards.get(banner.getId()));

                    BannerPermalink bannerPermalink = displayedBannerPermalinks.get(banner.getId());
                    if (bannerPermalink != null) {
                        banner.setPermalinkId(bannerPermalink.getPermalinkId());
                        banner.setPermalinkAssignType(bannerPermalink.getPermalinkAssignType());
                        banner.setPreferVCardOverPermalink(bannerPermalink.getPreferVCardOverPermalink());
                    }

                    GdiBannerMobileContentInfo bannerMobileContentInfo =
                            bannerIdToMobileContentInfo.get(banner.getId());
                    if (bannerMobileContentInfo != null) {
                        banner.setPrimaryAction(bannerMobileContentInfo.getPrimaryAction());
                        banner.setReflectedAttrs(bannerMobileContentInfo.getReflectedAttrs());
                        banner.setImpressionUrl(bannerMobileContentInfo.getImpressionUrl());
                    }
                }
        );
    }

    private Map<Long, ContentPromotionContent> getContentPromotionContentByBannerId(
            int shard,
            Collection<BannerWithContentPromotion> bannersWithContentPromotion) {
        Map<Long, Long> bidToContentPromotionId = listToMap(bannersWithContentPromotion,
                BannerWithContentPromotion::getId, BannerWithContentPromotion::getContentPromotionId);

        Map<Long, ContentPromotionContent> contentPromotionByContentId =
                contentPromotionRepository.getContentPromotionsByContentIds(shard, bidToContentPromotionId.values());
        return EntryStream.of(bidToContentPromotionId)
                .mapValues(contentPromotionByContentId::get)
                .toImmutableMap();
    }

    private static GdiBannerContentPromotion toGdiBannerContentPromotion(
            @Nullable BannerWithContentPromotion banner) {

        if (banner == null || banner.getContentPromotionId() == null) {
            return null;
        }
        return new GdiBannerContentPromotion()
                .withBid(banner.getId())
                .withVisitUrl(banner.getVisitUrl())
                .withContentPromotionId(banner.getContentPromotionId());
    }

    private static GdiBannerButton toGdiBannerButton(@Nullable BannerWithButton banner) {
        if (banner == null || banner.getButtonAction() == null) {
            return null;
        }

        return new GdiBannerButton()
                .withAction(GdiButtonAction.fromSource(banner.getButtonAction()))
                .withCustomText(banner.getButtonAction() == ButtonAction.CUSTOM_TEXT ? banner.getButtonCaption() : null)
                .withHref(banner.getButtonHref());
    }

    /**
     * Возвращает для каждого баннера организацию, которая должна быть отображена на фронте
     *
     * @param bannerPermalinks                пермалинки баннеров
     * @param isShowAutoAssignedOrganizations если {@code true}, то может быть отображена автоматически
     *                                        привязанная организация (будет сделан запрос в справочник для проверки
     *                                        опубликованности авто-привязанных организаций)
     * @see GridBannerService#chooseDisplayedBannerPermalink
     */
    private Map<Long, BannerPermalink> getDisplayedBannerPermalinks(
            Map<Long, List<BannerPermalink>> bannerPermalinks,
            boolean isShowAutoAssignedOrganizations
    ) {
        if (bannerPermalinks.isEmpty()) {
            return emptyMap();
        }

        Set<Long> availableAutoPermalinks;
        if (isShowAutoAssignedOrganizations) {
            Set<Long> autoAssignedPermalinkIds = EntryStream.of(bannerPermalinks)
                    // Оставляем только баннеры, у которых нету ручного пермалинка
                    .filterValues(permalinks -> permalinks.stream()
                            .noneMatch(p -> p.getPermalinkAssignType() == MANUAL))
                    .values()
                    .flatMap(StreamEx::of)
                    .filter(GridBannerService::canDisplayAutoPermalink)
                    .map(BannerPermalink::getPermalinkId)
                    .toSet();

            // не кидаем исключение если справочник недоступен, т.к. в этом случае все равно хотим показать
            // список баннеров на фронте
            availableAutoPermalinks = organizationService.getAvailablePermalinkIdsNoThrow(autoAssignedPermalinkIds);
        } else {
            availableAutoPermalinks = Collections.emptySet();
        }

        return EntryStream.of(bannerPermalinks)
                .mapValues(permalinks -> chooseDisplayedBannerPermalink(permalinks, availableAutoPermalinks))
                .nonNullValues()
                .toMap();
    }

    /**
     * Возвращает пермалинк, который должен быть отображен на фронте
     * <ul>
     *     <li>Если есть вручную привязанный пермалинк, то отдаём его.</li>
     *     <li>Если такого нет, то отдаём опубликованный автоматически привязанный пермалинк,
     *     привязку которого пользователь не отклонил, с минимальным ID (если такой есть)</li>
     * </ul>
     *
     * @param permalinks          все пермалинки баннера
     * @param availablePermalinks опубликованные авто-привязанные пермалинки
     */
    @Nullable
    static BannerPermalink chooseDisplayedBannerPermalink(Collection<BannerPermalink> permalinks,
                                                          Set<Long> availablePermalinks) {
        var manualPermalink = StreamEx.of(permalinks)
                .findFirst(permalink -> permalink.getPermalinkAssignType() == MANUAL);

        if (manualPermalink.isPresent()) {
            return manualPermalink.get();
        }

        var autoPermalink = StreamEx.of(permalinks)
                .filter(GridBannerService::canDisplayAutoPermalink)
                .filter(permalink -> availablePermalinks.contains(permalink.getPermalinkId()))
                .minBy(BannerPermalink::getPermalinkId);

        return autoPermalink.orElse(null);
    }

    /**
     * Авто-привязанный пермалинк должен отдаваться на фронт, если его привязку раньше не отклоняли
     */
    private static boolean canDisplayAutoPermalink(BannerPermalink permalink) {
        return permalink.getPermalinkAssignType() == AUTO && !permalink.getIsChangeToManualRejected();
    }

    /**
     * Получить множество id баннеров, которые остановлены мониторингом.
     *
     * @param shard     шард, в котором хранятся баннеры
     * @param bannerIds список id баннеров баннеров для которых нужно получить данные по изображениям
     */
    public Set<Long> getMonitoringStoppedBanners(int shard, Collection<Long> bannerIds) {
        return gridBannerDomainRepository.getMonitoringStoppedBanners(shard, bannerIds);
    }

    private boolean hasTurbolandings(GdiBanner banner) {
        if (banner.getTurbolanding() != null) {
            return true;
        }
        return banner.getSitelinks() != null &&
                banner.getSitelinks().stream().anyMatch(s -> s.getTurbolanding() != null);
    }

    public Map<Long, String> getTurboGalleryByBannerIds(ClientId clientId, Collection<Long> bannerIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return turboGalleriesRepository.getTurboGalleriesByBannerIds(shard, bannerIds);
    }

    public Map<Long, Creative> getCreatives(int shard, Collection<Long> bannerIds) {
        Map<Long, Long> bannerIdsToCreativeIds =
                gridCreativeRepository.getBannerIdToCreativeId(shard, bannerIds);

        return getCreatives(shard, bannerIdsToCreativeIds);
    }

    private Map<Long, Creative> getCreatives(int shard, Map<Long, Long> bannerIdsToCreativeIds) {
        List<Creative> creatives = creativeRepository.getCreatives(shard, bannerIdsToCreativeIds.values());
        Map<Long, Creative> creativeIdsToCreatives = listToMap(creatives, Creative::getId);

        return EntryStream.of(bannerIdsToCreativeIds)
                .mapValues(creativeIdsToCreatives::get)
                .toMap();
    }
}
