package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutsStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.StatusModerateBannerPage;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBannerImagesStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBannersCreativeStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerBannersDisplayHrefStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerPrimaryStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerResources;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatus;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusBsSynced;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusPostModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusSitelinksModerate;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerStatusVCardModeration;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис, считающий вычисляемые поля для внешнего представления описания баннера
 */
@ParametersAreNonnullByDefault
public class GridBannerStatusUtils {
    private static final Logger logger = LoggerFactory.getLogger(GridBannerStatusUtils.class);
    private static final ImmutableList<BannersBannerType> BASED_ON_CREATIVE_STATUS_BANNER_CLASSES =
            ImmutableList.of(BannersBannerType.image_ad, BannersBannerType.performance, BannersBannerType.cpm_banner,
                    BannersBannerType.cpm_audio, BannersBannerType.cpc_video);
    private static final ImmutableList<BannersBannerType> BASED_ON_IMAGE_STATUS_BANNER_CLASSES =
            ImmutableList.of(BannersBannerType.image_ad, BannersBannerType.mcbanner);

    /**
     * Получить статус баннера для отправки оператору
     *
     * @param banner описание баннера во внутреннем формате
     */
    public static GdiBannerStatus extractBannerStatus(GdiBanner banner) {
        return new GdiBannerStatus()
                .withPreviousVersionShown(isPreviousVersionShown(banner))
                .withRejectedOnModeration(isRejectedOnModeration(banner))
                .withPlacementPagesRequired(isPlacementPagesRequired(banner))
                .withHasInactiveResources(getInactiveResources(banner))
                .withPrimaryStatus(primaryStatus(banner));
    }

    public static GdiBannerPrimaryStatus primaryStatus(GdiBanner banner) {
        if (banner.getStatusArchived()) {
            return GdiBannerPrimaryStatus.ARCHIVED;
        } else if (isManuallySuspended(banner)) {
            return GdiBannerPrimaryStatus.MANUALLY_SUSPENDED;
        } else if (isDraft(banner)) {
            return GdiBannerPrimaryStatus.DRAFT;
        } else if (isOnModeration(banner)) {
            return GdiBannerPrimaryStatus.MODERATION;
        } else if (isRejectedOnModeration(banner) || isOutdoorBannerDeclined(banner)) {
            return GdiBannerPrimaryStatus.MODERATION_REJECTED;
        } else if (isTemporarilySuspended(banner)) {
            return GdiBannerPrimaryStatus.TEMPORARILY_SUSPENDED;
        } else if (isActive(banner)) {
            return GdiBannerPrimaryStatus.ACTIVE;
        } else {
            logger.warn("Banner unusual status. Returning ACTIVE.", banner);
            return GdiBannerPrimaryStatus.ACTIVE;
        }
    }

    private static boolean isPreviousVersionShown(GdiBanner banner) {
        return banner.getStatusBsSynced() != GdiBannerStatusBsSynced.YES && banner.getStatusActive()
                && banner.getBannerType() != BannersBannerType.cpm_outdoor;
    }

    /**
     * Возвращаем это всегда, когда баннер отклонен на модерации.
     * Даже, если показываем предыдущую версию.
     */
    private static boolean isRejectedOnModeration(GdiBanner banner) {

        boolean bannerDeclined = banner.getStatusModerate() == GdiBannerStatusModerate.NO;
        boolean imageDeclined = extractBannerImageStatusModerate(banner) == GdiBannerImageStatusModerate.NO;
        boolean creativeDeclined =
                (extractBannersCreativeStatusModerate(banner) == GdiBannerBannersCreativeStatusModerate.NO);

        return bannerDeclined || creativeDeclined || imageDeclined;
    }

    private static boolean isOutdoorBannerDeclined(GdiBanner banner) {
        return banner.getBannerType() == BannersBannerType.cpm_outdoor
                && banner.getStatusModerate() == GdiBannerStatusModerate.YES
                && isOutdoorBannerWithStatusModerateYesDeclined(banner);
    }


    private static boolean isOnModeration(GdiBanner banner) {
        boolean bannerOnModeration = (
                Arrays.asList(GdiBannerStatusModerate.SENDING,
                        GdiBannerStatusModerate.SENT,
                        GdiBannerStatusModerate.READY)
                        .contains(banner.getStatusModerate()) &&
                        Arrays.asList(GdiBannerStatusPostModerate.NO, GdiBannerStatusPostModerate.REJECTED)
                                .contains(banner.getStatusPostModerate())
        );

        boolean cpmOutdoorBannerOnModeration =
                banner.getBannerType() == BannersBannerType.cpm_outdoor
                        && (Arrays.asList(GdiBannerStatusModerate.SENDING,
                        GdiBannerStatusModerate.SENT,
                        GdiBannerStatusModerate.READY)
                        .contains(banner.getStatusModerate())
                        || banner.getStatusModerate() == GdiBannerStatusModerate.YES
                        && isOutdoorBannerWithStatusModerateYesOnModeration(banner));

        boolean imageOnModeration = Arrays.asList(GdiBannerImageStatusModerate.READY,
                GdiBannerImageStatusModerate.SENDING,
                GdiBannerImageStatusModerate.SENT)
                .contains(
                        nvl(extractBannerImageStatusModerate(banner), GdiBannerImageStatusModerate.YES)
                );
        boolean creativeOnModeration = Arrays.asList(GdiBannerBannersCreativeStatusModerate.READY,
                GdiBannerBannersCreativeStatusModerate.SENDING,
                GdiBannerBannersCreativeStatusModerate.SENT)
                .contains(
                        nvl(extractBannersCreativeStatusModerate(banner), GdiBannerBannersCreativeStatusModerate.YES)
                );

        return bannerOnModeration || cpmOutdoorBannerOnModeration || creativeOnModeration || imageOnModeration;
    }

    private static boolean isOutdoorBannerWithStatusModerateYesOnModeration(GdiBanner banner) {
        if (banner.getBannerType() == BannersBannerType.cpm_outdoor
                && banner.getStatusModerate() == GdiBannerStatusModerate.YES) {
            List<ModerateBannerPage> operatorModerationResults = nvl(banner.getPlacementPages(), emptyList());
            boolean operatorResultsIsEmpty = isEmpty(operatorModerationResults);
            Set<StatusModerateBannerPage> operatorStatuses = operatorModerationResults.stream()
                    .map(ModerateBannerPage::getStatusModerate)
                    .collect(toSet());
            boolean thereIsYesOperatorModerationResult = operatorStatuses.contains(StatusModerateBannerPage.YES);
            boolean thereAreOnlyNoOperatorResults = singleton(StatusModerateBannerPage.NO).equals(operatorStatuses);

            // 1. операторы не уходят на модерацию + нет минус регионов -> На модерации (не выбрано подходящих щитов)
            // 2. не все операторы отклонили и ни один оператор не принял -> На модерации
            return isPlacementPagesRequired(banner)
                    || !operatorResultsIsEmpty && !thereIsYesOperatorModerationResult && !thereAreOnlyNoOperatorResults;
        } else {
            return false;
        }
    }

    private static boolean isPlacementPagesRequired(GdiBanner banner) {
        List<ModerateBannerPage> operatorModerationResults = nvl(banner.getPlacementPages(), emptyList());
        boolean currentMinusGeoIsEmpty = isEmpty(banner.getCurrentMinusGeo());
        boolean operatorResultsIsEmpty = isEmpty(operatorModerationResults);

        // 1. операторы не уходят на модерацию + нет минус регионов -> На модерации (не выбрано подходящих щитов)
        return banner.getBannerType() == BannersBannerType.cpm_outdoor
                && banner.getStatusModerate() == GdiBannerStatusModerate.YES
                && operatorResultsIsEmpty && currentMinusGeoIsEmpty;
    }

    private static boolean isOutdoorBannerWithStatusModerateYesDeclined(GdiBanner outdoorBanner) {
        List<ModerateBannerPage> operatorModerationResults = nvl(outdoorBanner.getPlacementPages(), emptyList());
        Set<StatusModerateBannerPage> operatorStatuses = operatorModerationResults.stream()
                .map(ModerateBannerPage::getStatusModerate)
                .collect(toSet());

        return singleton(StatusModerateBannerPage.NO).equals(operatorStatuses);
    }

    private static boolean isActive(GdiBanner banner) {
        boolean bannerActive = banner.getStatusModerate() == GdiBannerStatusModerate.YES &&
                banner.getStatusPostModerate() == GdiBannerStatusPostModerate.YES &&
                banner.getStatusShow();
        boolean imageAccepted = nvl(extractBannerImageStatusModerate(banner), GdiBannerImageStatusModerate.YES)
                == GdiBannerImageStatusModerate.YES;
        boolean creativeAccepted =
                nvl(extractBannersCreativeStatusModerate(banner), GdiBannerBannersCreativeStatusModerate.YES)
                        == GdiBannerBannersCreativeStatusModerate.YES;

        return bannerActive && creativeAccepted && imageAccepted;
    }

    private static boolean isDraft(GdiBanner banner) {
        boolean bannerDraft = banner.getStatusModerate() == GdiBannerStatusModerate.NEW;
        boolean imageDraft = extractBannerImageStatusModerate(banner) == GdiBannerImageStatusModerate.NEW;
        boolean creativeDraft =
                (extractBannersCreativeStatusModerate(banner) == GdiBannerBannersCreativeStatusModerate.NEW);

        return bannerDraft || imageDraft || creativeDraft;
    }

    private static boolean isTemporarilySuspended(GdiBanner banner) {
        return nvl(banner.getStatusMonitoringDomainStop(), false);
    }

    private static boolean isManuallySuspended(GdiBanner banner) {
        return !banner.getStatusShow();
    }

    /**
     * Находим все непромодерированные подобъекты баннера
     *
     * @param banner
     * @return
     */
    private static Set<GdiBannerResources> getInactiveResources(GdiBanner banner) {
        Set<GdiBannerResources> inactiveResources = new HashSet<>();

        checkBannerSitelinksRestriction(banner, inactiveResources::add);
        checkBannerVcardsRestriction(banner, inactiveResources::add);
        checkBannerImagesImageRestriction(banner, inactiveResources::add);
        checkBannerBannerImagesImageRestriction(banner, inactiveResources::add);
        checkBannerCreativeRestriction(banner, inactiveResources::add);
        checkBannerTurbolandingRestriction(banner, inactiveResources::add);
        checkBannerDisplayHrefRestriction(banner, inactiveResources::add);
        checkBannerCalloutsRestriction(banner, inactiveResources::add);

        return inactiveResources;
    }

    private static void checkBannerVcardsRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getVcardId() != null && GdiBannerStatusVCardModeration.NO == banner.getPhoneFlag()) {
            c.accept(GdiBannerResources.VCARD);
        }
    }

    private static void checkBannerImagesImageRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getImage() != null
                && banner.getImage().getStatusModerate() == GdiBannerImageStatusModerate.NO) {
            c.accept(GdiBannerResources.IMAGES);
        }
    }

    private static void checkBannerBannerImagesImageRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getBannerImage() != null
                && banner.getBannerImage().getStatusModerate() == GdiBannerBannerImagesStatusModerate.NO) {
            c.accept(GdiBannerResources.BANNER_IMAGES);
        }
    }

    private static void checkBannerCreativeRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getCreative() != null
                && banner.getCreative().getStatusModerate() == GdiBannerBannersCreativeStatusModerate.NO) {
            c.accept(GdiBannerResources.CREATIVES);
        }
    }

    private static void checkBannerTurbolandingRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getTurbolanding() != null
                && banner.getTurbolanding().getStatusModerate() == BannerTurboLandingStatusModerate.NO) {
            c.accept(GdiBannerResources.TURBOLANDINGS);
        }
    }

    private static void checkBannerDisplayHrefRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getDisplayHref() != null
                && banner.getDisplayHref().getStatusModerate() == GdiBannerBannersDisplayHrefStatusModerate.NO) {
            c.accept(GdiBannerResources.DISPLAY_HREF);
        }
    }

    private static void checkBannerSitelinksRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (banner.getSitelinksSetId() != null
                && banner.getStatusSitelinksModerate() == GdiBannerStatusSitelinksModerate.NO) {
            c.accept(GdiBannerResources.SITELINKS);
        }
    }

    private static void checkBannerCalloutsRestriction(GdiBanner banner, Consumer<GdiBannerResources> c) {
        if (bannerHasAnyDeclinedCallouts(banner.getCallouts())) {
            c.accept(GdiBannerResources.CALLOUTS);
        }
    }

    private static boolean bannerHasAnyDeclinedCallouts(@Nullable List<Callout> callouts) {
        return CollectionUtils.isNotEmpty(callouts) &&
                callouts.stream().anyMatch(callout -> CalloutsStatusModerate.NO == callout.getStatusModerate());
    }

    @Nullable
    private static GdiBannerBannersCreativeStatusModerate extractBannersCreativeStatusModerate(GdiBanner banner) {
        if (BASED_ON_CREATIVE_STATUS_BANNER_CLASSES.contains(banner.getBannerType()) && banner.getCreative() != null) {
            return banner.getCreative().getStatusModerate();
        }
        return null;
    }

    @Nullable
    private static GdiBannerImageStatusModerate extractBannerImageStatusModerate(GdiBanner banner) {
        if (BASED_ON_IMAGE_STATUS_BANNER_CLASSES.contains(banner.getBannerType()) && banner.getImage() != null) {
            return banner.getImage().getStatusModerate();
        }
        return null;
    }
}
