package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.banner.model.GdiReplaceDisplayHrefBanner;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSourceReplaceDisplayHrefBanner;
import ru.yandex.direct.grid.core.entity.banner.repository.GridFindAndReplaceBannerRepository;
import ru.yandex.direct.grid.core.entity.banner.service.internal.GridBannerDisplayHrefUpdate;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;

/**
 * Сервис для получения данных поиска и замены отображаемой ссылки баннера
 */
@Service
public class GridFindAndReplaceBannerDisplayHrefService {

    private final ShardHelper shardHelper;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final GridFindAndReplaceBannerRepository repository;

    @Autowired
    public GridFindAndReplaceBannerDisplayHrefService(ShardHelper shardHelper,
                                                      BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                                      GridFindAndReplaceBannerRepository repository) {
        this.shardHelper = shardHelper;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.repository = repository;
    }


    /**
     * Получить баннеры из бд, добавив обертку с новой отображаемой ссылкой.
     *
     * @param clientId           - id клиента
     * @param bannerIds          - список id баннеров для поиска
     * @param searchDisplayHrefs - список отображаемых ссылок для поиска
     * @param newDisplayHref     - новая отображаемая ссылка
     * @return список баннеров с новой ссылкой
     */
    public List<GdiReplaceDisplayHrefBanner> getBanners(
            ClientId clientId, List<Long> bannerIds, List<String> searchDisplayHrefs, String newDisplayHref) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<GdiSourceReplaceDisplayHrefBanner> banners =
                repository.getBannersWithDisplayHrefs(shard, clientId, bannerIds, searchDisplayHrefs);
        return banners.stream().map(
                banner -> new GdiReplaceDisplayHrefBanner()
                        .withBannerId(banner.getBannerId())
                        .withBannerType(banner.getBannerType())
                        .withSourceDisplayHref(banner.getSourceDisplayHref())
                        .withNewDisplayHref(newDisplayHref))
                .collect(Collectors.toList());
    }

    /**
     * Получает отображаемые ссылки всех указанных баннеров клиента
     *
     * @param bannerIds - список id баннеров
     * @param clientId  - id клиента
     * @return - уникальные отображаемые ссылки
     */
    public Set<String> getDisplayHrefs(List<Long> bannerIds, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return repository.getBannerDisplayHrefs(shard, clientId, bannerIds);
    }

    /**
     * Превью изменений баннера. Получаем только результат валидации
     *
     * @param updateBanners список изменений, которые нужно применить к баннеру
     * @param operatorUid   uid оператора
     * @param clientId      id клиента, над которым производится действие
     * @return Результат валидации изменений
     */
    public GridBannerUpdateInfo preview(
            List<GdiReplaceDisplayHrefBanner> updateBanners, Long operatorUid, ClientId clientId) {
        return new GridBannerDisplayHrefUpdate(updateBanners, bannersUpdateOperationFactory, operatorUid, clientId).preview();
    }

    /**
     * Заменить ссылки баннеров
     *
     * @param updateBanners список изменений, которые нужно применить к баннеру
     * @param operatorUid   uid оператора
     * @param clientId      id клиента, над которым производится действие
     * @return Результат валидации изменений
     */
    public GridBannerUpdateInfo replace(
            List<GdiReplaceDisplayHrefBanner> updateBanners, Long operatorUid, ClientId clientId) {
        return new GridBannerDisplayHrefUpdate(updateBanners, bannersUpdateOperationFactory, operatorUid, clientId).replace();
    }

}
