package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerWithSitelinksForReplace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerHrefItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerHrefItemSitelink;
import ru.yandex.direct.grid.core.entity.banner.repository.GridFindAndReplaceBannerRepository;
import ru.yandex.direct.grid.core.entity.banner.service.internal.GridBannerHrefWithSitelinksUpdate;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;

import static java.util.Collections.emptyList;

/**
 * Сервис для получения данных поиска и замены href в баннерах + сайтлинках
 */
@Service
@ParametersAreNonnullByDefault
public class GridFindAndReplaceBannerHrefService {

    private final GridFindAndReplaceBannerRepository repository;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final SitelinkSetService sitelinkSetService;
    private final ShardHelper shardHelper;

    @Autowired
    public GridFindAndReplaceBannerHrefService(
            GridFindAndReplaceBannerRepository repository,
            BannersUpdateOperationFactory bannersUpdateOperationFactory,
            SitelinkSetService sitelinkSetService,
            ShardHelper shardHelper) {
        this.repository = repository;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.sitelinkSetService = sitelinkSetService;
        this.shardHelper = shardHelper;
    }

    /**
     * Возвращает информацию, необходимую для замены ссылок в баннерах и сайтлинках.
     * В итоговом списке будут элементы только для тех баннеров, которые действительно надо обновить
     * (обновить ссылку самого баннера и/или ссылки его сайтлинков)
     *
     * @param clientId                id клиента
     * @param bannerIds               список id баннеров, в которых заменять href
     * @param bannerIdsHrefExceptions сет id баннеров, в которых не будет поиска и замены кликовой ссылки
     * @param sitelinksExceptions     маппинг id баннера -> список id сайтлинков, которые нужно исключить
     * @param needReplaceBannerHref   нужно ли заменять ссылку в баннерах
     * @param needReplaceSitelinkHref нужно ли заменять ссылку в сайтлинках
     * @param replaceHrefRule         processor поиска и замены ссылки
     * @return список {@link GdiFindAndReplaceBannerHrefItem}
     */
    public List<GdiFindAndReplaceBannerHrefItem> getFindAndReplaceBannersHrefItems(ClientId clientId,
                                                                                   List<Long> bannerIds,
                                                                                   Set<Long> bannerIdsHrefExceptions,
                                                                                   Map<Long, Set<Long>> sitelinksExceptions,
                                                                                   boolean needReplaceBannerHref,
                                                                                   boolean needReplaceSitelinkHref,
                                                                                   boolean needReplaceTurbolandingParams, ReplaceRule replaceHrefRule,
                                                                                   ReplaceRule turbolandingReplaceRule) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<GdiBannerWithSitelinksForReplace> banners = repository
                .getBannerWithSitelinks(shard, clientId, bannerIds, needReplaceSitelinkHref, false);

        List<GdiFindAndReplaceBannerHrefItem> previewItems = new ArrayList<>();
        banners.forEach(banner -> {
            String newHref = needReplaceBannerHref && !bannerIdsHrefExceptions.contains(banner.getId())
                    ? replaceHrefRule.apply(banner.getHref())
                    : null;
            String newTurbolandingParams =
                    needReplaceTurbolandingParams && !bannerIdsHrefExceptions.contains(banner.getId())
                    && banner.getTurbolandingParams() != null
                    ? turbolandingReplaceRule.apply(banner.getTurbolandingParams())
                    : banner.getTurbolandingParams();
            if (Objects.equals(newHref, banner.getHref())) {
                newHref = null;
            }
            List<GdiFindAndReplaceBannerHrefItemSitelink> sitelinksInfo =
                    getBannerSitelinksInfo(banner, sitelinksExceptions, replaceHrefRule);
            if (newHref != null || !sitelinksInfo.isEmpty()) {
                previewItems.add(new GdiFindAndReplaceBannerHrefItem()
                        .withBannerId(banner.getId())
                        .withBannerType(banner.getBannerType())
                        .withAdGroupType(banner.getAdGroupType())
                        .withImageHash(banner.getImageHash())
                        .withOldHref(banner.getHref())
                        .withNewHref(newHref)
                        .withSitelinks(sitelinksInfo)
                        .withOldTurbolandingParams(banner.getTurbolandingParams())
                        .withNewTurbolandingParams(newTurbolandingParams));
            }
        });

        return previewItems;
    }

    /**
     * Возвращает информацию о сайтлинках баннера, необходимую для замены ссылок.
     *
     * @param bannerInfo          информация о баннере, для которого надо собрать информацию о сайтлинках
     * @param sitelinksExceptions мапа id баннера -> id сайтлинков, которые надо исключить
     * @param replaceHrefRule     правило, по которому заменять href
     * @return пустой список, если сайтлинков у баннера нет или в них ничего не надо менять,
     * иначе - информацию обо всех сайтлинках баннера
     */
    private List<GdiFindAndReplaceBannerHrefItemSitelink> getBannerSitelinksInfo(
            GdiBannerWithSitelinksForReplace bannerInfo, Map<Long, Set<Long>> sitelinksExceptions,
            ReplaceRule replaceHrefRule) {
        if (bannerInfo.getSitelinks().isEmpty()) {
            return emptyList();
        }

        List<GdiFindAndReplaceBannerHrefItemSitelink> gdSitelinks = new ArrayList<>();
        boolean sitelinksChanged = false;
        for (Sitelink sitelink : bannerInfo.getSitelinks()) {
            GdiFindAndReplaceBannerHrefItemSitelink itemSitelink =
                    new GdiFindAndReplaceBannerHrefItemSitelink()
                            .withSitelink(sitelink)
                            .withSitelinkId(sitelink.getId())
                            .withOldHref(sitelink.getHref())
                            .withChanged(false);
            String newSitelinkHref = replaceHrefRule.apply(sitelink.getHref());
            if (Objects.equals(newSitelinkHref, sitelink.getHref())) {
                newSitelinkHref = null;
            }

            Set<Long> bannerSitelinkExceptions = sitelinksExceptions.get(bannerInfo.getId());
            boolean sitelinkExcluded = bannerSitelinkExceptions != null
                    && bannerSitelinkExceptions.contains(sitelink.getId());
            if (!sitelinkExcluded && newSitelinkHref != null) {
                sitelinksChanged = true;
                sitelink.withId(null)
                        .withHref(newSitelinkHref);
                itemSitelink.withChanged(true);
            }
            gdSitelinks.add(itemSitelink);
        }
        if (sitelinksChanged) {
            return gdSitelinks;
        } else {
            return emptyList();
        }
    }

    /**
     * Превью изменений баннера. Получаем только результат валидации
     *
     * @param updateBanners список изменений, которые нужно применить к баннеру
     * @param operatorUid   uid оператора
     * @param clientId      id клиента, над которым производится действие
     * @return Результат валидации изменений
     */
    public GridBannerUpdateInfo updateBannersHrefAndSitelinksPreview(
            List<GdiFindAndReplaceBannerHrefItem> updateBanners, Long operatorUid, ClientId clientId) {
        return createGridBannerHrefWithSitelinksUpdate(updateBanners, operatorUid, clientId).preview();
    }

    /**
     * Записывает изменение в базу
     *
     * @param updateBanners список изменений, которые нужно применить к баннеру
     * @param operatorUid   uid оператора
     * @param clientId      id клиента, над которым производится действие
     * @return Результат валидации изменений
     */
    public GridBannerUpdateInfo updateBannersHrefAndSitelinks(
            List<GdiFindAndReplaceBannerHrefItem> updateBanners, Long operatorUid, ClientId clientId) {
        return createGridBannerHrefWithSitelinksUpdate(updateBanners, operatorUid, clientId).update();
    }

    /**
     * Получить все домены баннеров и соответствующих сайтлинков
     *
     * @param bannerIds - список id баннеров
     * @param clientId  - id клиента
     * @return - список доменов
     */
    public Set<String> getDomains(List<Long> bannerIds, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return repository.getBannerAndSitelinksDomains(shard, clientId, bannerIds);
    }

    private GridBannerHrefWithSitelinksUpdate createGridBannerHrefWithSitelinksUpdate(
            List<GdiFindAndReplaceBannerHrefItem> updateBanners, Long operatorUid, ClientId clientId) {
        return new GridBannerHrefWithSitelinksUpdate(bannersUpdateOperationFactory, sitelinkSetService, updateBanners,
                operatorUid, clientId);
    }
}
