package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerWithSitelinksForReplace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerTextItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerTextItemSitelink;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceRequest;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType;
import ru.yandex.direct.grid.core.entity.banner.repository.GridFindAndReplaceBannerRepository;
import ru.yandex.direct.grid.core.entity.banner.service.internal.GridBannerWithSitelinksTextUpdate;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;

import static java.util.Collections.emptyList;
import static java.util.Map.entry;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.BODY;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.DISPLAY_HREF;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.HREF;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.SITELINK_DESCRIPTION;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.SITELINK_HREF;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.SITELINK_TITLE;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.TITLE;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.TITLE_EXTENSION;
import static ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceTargetType.TURBOLANDING_PARAMS;

/**
 * Сервис для получения данных поиска и замены заголовка, доп.заголовка, текста
 * и сайтлинков на баннерах
 */
@Service
@ParametersAreNonnullByDefault
public class GridFindAndReplaceBannerTextService {
    // менять вместе с ru.yandex.direct.grid.core.entity.banner.service.converter.
    // GridFindAndReplaceBannerTextConverter.gdiBannerTextToModelChanges
    private static final Map<BannersBannerType, Set<GdiFindAndReplaceTargetType>> AVAILABLE_FIELDS_BY_BANNER_TYPE =
            Map.ofEntries(
                    entry(BannersBannerType.text, Set.of(
                            TITLE,
                            TITLE_EXTENSION,
                            BODY,
                            HREF,
                            DISPLAY_HREF,
                            TURBOLANDING_PARAMS)),
                    entry(BannersBannerType.content_promotion, Set.of()),
                    entry(BannersBannerType.cpc_video, Set.of(
                            TITLE,
                            BODY,
                            HREF,
                            TURBOLANDING_PARAMS
                    )),
                    entry(BannersBannerType.cpm_banner, Set.of(
                            TITLE,
                            TITLE_EXTENSION,
                            BODY,
                            HREF,
                            TURBOLANDING_PARAMS
                    )),
                    entry(BannersBannerType.cpm_audio, Set.of(
                            HREF,
                            TURBOLANDING_PARAMS
                    )),
                    entry(BannersBannerType.cpm_outdoor, Set.of(HREF)),
                    entry(BannersBannerType.cpm_indoor, Set.of(HREF)),
                    entry(BannersBannerType.performance, Set.of()),
                    entry(BannersBannerType.cpm_geo_pin, Set.of()),
                    entry(BannersBannerType.mobile_content, Set.of(
                            TITLE,
                            BODY,
                            HREF
                    )),
                    entry(BannersBannerType.dynamic, Set.of(
                            BODY,
                            HREF,
                            DISPLAY_HREF,
                            TURBOLANDING_PARAMS
                    )),
                    entry(BannersBannerType.mcbanner, Set.of(HREF)),
                    entry(BannersBannerType.internal, Set.of()),
                    entry(BannersBannerType.image_ad, Set.of(
                            HREF,
                            TURBOLANDING_PARAMS
                    ))
            );

    private static final Map<AdGroupType, Set<GdiFindAndReplaceTargetType>> AVAILABLE_FIELDS_BY_AD_GROUP_TYPE = Map.of(
            AdGroupType.MOBILE_CONTENT, Set.of(TURBOLANDING_PARAMS, HREF)
    );

    private static final Set<GdiFindAndReplaceTargetType> ALL_BANNER_FIELDS =
            Set.of(GdiFindAndReplaceTargetType.values());

    private final ShardHelper shardHelper;
    private final GridFindAndReplaceBannerRepository repository;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final SitelinkSetService sitelinkSetService;

    @Autowired
    public GridFindAndReplaceBannerTextService(ShardHelper shardHelper,
                                               GridFindAndReplaceBannerRepository repository,
                                               BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                               SitelinkSetService sitelinkSetService) {
        this.shardHelper = shardHelper;
        this.repository = repository;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.sitelinkSetService = sitelinkSetService;
    }

    public List<GdiFindAndReplaceBannerTextItem> getBanners(ClientId clientId, List<Long> bannerIds,
                                                            GdiFindAndReplaceRequest replaceRequest) {
        int shard = shardHelper.getShardByClientId(clientId);
        var banners = repository.getBannerWithSitelinks(shard, clientId, bannerIds,
                replaceRequest.getUpdateSitelink(), replaceRequest.getUpdateDisplayHref());

        List<GdiFindAndReplaceBannerTextItem> previewItems = new ArrayList<>();
        banners.forEach(banner -> addItem(banner, replaceRequest, previewItems));

        return previewItems;
    }

    private void addItem(GdiBannerWithSitelinksForReplace banner,
                         GdiFindAndReplaceRequest replaceRequest,
                         List<GdiFindAndReplaceBannerTextItem> previewItems) {


        Set<GdiFindAndReplaceTargetType> availableFieldsByBannerType =
                AVAILABLE_FIELDS_BY_BANNER_TYPE.getOrDefault(banner.getBannerType(), ALL_BANNER_FIELDS);
        Set<GdiFindAndReplaceTargetType> availableFieldsByAdGroupType =
                AVAILABLE_FIELDS_BY_AD_GROUP_TYPE.getOrDefault(banner.getAdGroupType(), ALL_BANNER_FIELDS);
        Set<GdiFindAndReplaceTargetType> availableFields = new HashSet<>(availableFieldsByBannerType);
        availableFields.retainAll(availableFieldsByAdGroupType);
        Set<GdiFindAndReplaceTargetType> targetTypes = replaceRequest.getTargetTypes();

        var needUpdateTitle = needUpdateField(targetTypes, availableFields, TITLE);
        var needUpdateTitleExtension = needUpdateField(targetTypes, availableFields, TITLE_EXTENSION);
        var needUpdateBody = needUpdateField(targetTypes, availableFields, BODY);
        var needUpdateHref = needUpdateField(targetTypes, availableFields, HREF);
        var needUpdateTurbolandingParams = needUpdateField(targetTypes, availableFields, TURBOLANDING_PARAMS);
        var needUpdateDisplayHref = needUpdateField(targetTypes, availableFields, DISPLAY_HREF);

        var newTitle = needUpdateTitle ? replace(replaceRequest, banner.getTitle()) : null;
        var newTitleExtension = needUpdateTitleExtension ? replace(replaceRequest, banner.getTitleExtension()) : null;
        var newBody = needUpdateBody ? replace(replaceRequest, banner.getBody()) : null;
        var newHref = needUpdateHref ? replaceHref(replaceRequest, banner.getHref()) : null;
        var newTurbolandingParams = needUpdateTurbolandingParams
                ? replaceTurbolandingParams(replaceRequest, banner.getTurbolandingParams())
                : banner.getTurbolandingParams();
        var newDisplayHref = needUpdateDisplayHref ? replace(replaceRequest, banner.getDisplayHref()) : null;
        List<GdiFindAndReplaceBannerTextItemSitelink> sitelinksInfo = getBannerSitelinksInfo(banner, replaceRequest);

        previewItems.add(new GdiFindAndReplaceBannerTextItem()
                .withBannerId(banner.getId())
                .withBannerType(banner.getBannerType())
                .withAdGroupType(banner.getAdGroupType())
                .withOldTitle(needUpdateTitle ? banner.getTitle() : null)
                .withNewTitle(newTitle)
                .withOldTitleExtension(needUpdateTitleExtension ? banner.getTitleExtension() : null)
                .withNewTitleExtension(newTitleExtension)
                .withOldBody(needUpdateBody ? banner.getBody() : null)
                .withNewBody(newBody)
                .withOldHref(needUpdateHref ? banner.getHref() : null)
                .withNewHref(newHref)
                .withOldTurbolandingParams(banner.getTurbolandingParams())
                .withNewTurbolandingParams(newTurbolandingParams)
                .withOldDisplayHref(needUpdateDisplayHref ? banner.getDisplayHref() : null)
                .withNewDisplayHref(newDisplayHref)
                .withSitelinks(sitelinksInfo)
                .withUpdateTitle(needUpdateTitle)
                .withUpdateTitleExtension(needUpdateTitleExtension)
                .withUpdateBody(needUpdateBody)
                .withUpdateHref(needUpdateHref)
                .withUpdateTurbolandingParams(needUpdateTurbolandingParams)
                .withUpdateDisplayHref(needUpdateDisplayHref));
    }

    private static boolean needUpdateField(Set<GdiFindAndReplaceTargetType> targetTypes,
                                           Set<GdiFindAndReplaceTargetType> availableFields,
                                           GdiFindAndReplaceTargetType fieldToChange) {
        return targetTypes.contains(fieldToChange) && availableFields.contains(fieldToChange);
    }

    @Nullable
    private String replace(GdiFindAndReplaceRequest replaceRequest, @Nullable String oldValue) {
        return replaceRequest.getReplaceRule().apply(oldValue);
    }

    @Nullable
    private String replaceHref(GdiFindAndReplaceRequest replaceRequest, @Nullable String oldValue) {
        return replaceRequest.getLinkReplaceRule().apply(oldValue);
    }

    @Nullable
    private String replaceTurbolandingParams(GdiFindAndReplaceRequest replaceRequest, @Nullable String oldValue) {
        return replaceRequest.getTurbolandingParamsReplaceRule().apply(oldValue);
    }

    public GridBannerUpdateInfo updateBannersAndSitelinksTextPreview(
            List<GdiFindAndReplaceBannerTextItem> banners, Long operatorUid, ClientId clientId) {
        return new GridBannerWithSitelinksTextUpdate(bannersUpdateOperationFactory, sitelinkSetService,
                operatorUid, clientId,
                banners).preview();
    }

    public GridBannerUpdateInfo updateBannerAndSitelinksText(List<GdiFindAndReplaceBannerTextItem> banners,
                                                             Long operatorUid, ClientId clientId) {
        return new GridBannerWithSitelinksTextUpdate(bannersUpdateOperationFactory, sitelinkSetService,
                operatorUid, clientId,
                banners).update();
    }

    /**
     * Возвращает информацию о сайтлинках баннера, необходимую для текста.
     */
    private List<GdiFindAndReplaceBannerTextItemSitelink> getBannerSitelinksInfo(
            GdiBannerWithSitelinksForReplace bannerInfo, GdiFindAndReplaceRequest replaceRequest) {
        if (bannerInfo.getSitelinks().isEmpty()) {
            return emptyList();
        }

        var updateTitleIndicies = replaceRequest.getSitelinkOrderNumsToUpdateTitle();
        var updateDescriptionIndicies = replaceRequest.getSitelinkOrderNumsToUpdateDescription();
        var updateHrefIndicies = replaceRequest.getSitelinkOrderNumsToUpdateHref();

        List<GdiFindAndReplaceBannerTextItemSitelink> gdSitelinks = new ArrayList<>();
        for (Sitelink sitelink : bannerInfo.getSitelinks()) {
            var itemSitelink = defaultGdiSitelinkItem(sitelink);

            updateSitelinkTitleIfNeeded(updateTitleIndicies, sitelink, replaceRequest);
            updateSitelinkDescriptionIfNeeded(updateDescriptionIndicies, sitelink, replaceRequest);
            updateSitelinkHrefIfNeeded(updateHrefIndicies, sitelink, replaceRequest);

            gdSitelinks.add(itemSitelink);
        }
        return gdSitelinks;
    }

    private void updateSitelinkTitleIfNeeded(
            Set<Long> updateTitleIndicies,
            Sitelink sitelink,
            GdiFindAndReplaceRequest replaceRequest) {
        if (updateTitleIndicies.contains(sitelink.getOrderNum())
                && replaceRequest.getTargetTypes().contains(SITELINK_TITLE)) {
            var newSitelinkTitle = replace(replaceRequest, sitelink.getTitle());
            var titleChanged = !sitelink.getTitle().equals(newSitelinkTitle);
            if (titleChanged) {
                sitelink.withId(null)
                        .withTitle(newSitelinkTitle);
            }
        }
    }

    private void updateSitelinkDescriptionIfNeeded(
            Set<Long> updateDescriptionIndicies,
            Sitelink sitelink,
            GdiFindAndReplaceRequest replaceRequest) {
        if (updateDescriptionIndicies.contains(sitelink.getOrderNum())
                && replaceRequest.getTargetTypes().contains(SITELINK_DESCRIPTION)) {
            var newSitelinkDescription = replace(replaceRequest, sitelink.getDescription());
            var descriptionChanged = sitelink.getDescription() != null
                    && !sitelink.getDescription().equals(newSitelinkDescription);
            if (descriptionChanged) {
                sitelink.withId(null)
                        .withDescription(newSitelinkDescription);
            }
        }
    }

    private void updateSitelinkHrefIfNeeded(
            Set<Long> updateHrefIndicies,
            Sitelink sitelink,
            GdiFindAndReplaceRequest replaceRequest) {
        if (updateHrefIndicies.contains(sitelink.getOrderNum())
                && replaceRequest.getTargetTypes().contains(SITELINK_HREF)) {
            var newSitelinkHref = replaceHref(replaceRequest, sitelink.getHref());
            var hrefChanged = !sitelink.getHref().equals(newSitelinkHref);
            if (hrefChanged) {
                sitelink.withId(null)
                        .withHref(newSitelinkHref);
            }
        }
    }

    private static GdiFindAndReplaceBannerTextItemSitelink defaultGdiSitelinkItem(Sitelink sitelink) {
        return new GdiFindAndReplaceBannerTextItemSitelink()
                .withSitelink(sitelink)
                .withSitelinkId(sitelink.getId())
                .withOldTitle(sitelink.getTitle())
                .withOldDescription(sitelink.getDescription())
                .withOldHref(sitelink.getHref());
    }
}
