package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerWithSitelinksForReplace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsItem;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsReplaceInstruction;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsReplaceInstructionAction;
import ru.yandex.direct.grid.core.entity.banner.repository.GridFindAndReplaceBannerRepository;
import ru.yandex.direct.grid.core.entity.banner.service.internal.GridBannerWithCalloutsUpdate;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;

import static java.util.Collections.emptyList;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.BANNER_TYPE_TO_CLASS;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Сервис для поиска и замены уточнений в баннерах
 */
@Service
@ParametersAreNonnullByDefault
public class GridFindAndReplaceCalloutsService {

    private final ShardHelper shardHelper;
    private final CalloutRepository calloutsRepository;
    private final GridFindAndReplaceBannerRepository repository;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final FeatureService featureService;

    @Autowired
    public GridFindAndReplaceCalloutsService(ShardHelper shardHelper,
                                             @Qualifier("calloutRepository") CalloutRepository calloutsRepository,
                                             GridFindAndReplaceBannerRepository repository,
                                             BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                             FeatureService featureService) {
        this.shardHelper = shardHelper;
        this.calloutsRepository = calloutsRepository;
        this.repository = repository;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.featureService  = featureService;
    }

    public List<GdiFindAndReplaceBannerCalloutsItem> getBanners(
            ClientId clientId, List<Long> bannerIds,
            GdiFindAndReplaceBannerCalloutsReplaceInstruction replaceInstruction) {
        int shard = shardHelper.getShardByClientId(clientId);

        List<GdiBannerWithSitelinksForReplace> allBanners = repository.getBannerWithSitelinks(shard,
                clientId, bannerIds, false, false);

        // как-то не эффективно достаём все баннеры, а потом используем только NewBannerWithCallouts
        // эффективнее было бы сразу передать нужный BannersBannerType в запрос
        var isRmpCalloutsEnabled = featureService.isEnabledForClientId(clientId, FeatureName.RMP_CALLOUTS_ENABLED);
        List<GdiBannerWithSitelinksForReplace> bannersWithCallouts = filterList(allBanners,
                b -> BannersBannerType.mobile_content.equals(b.getBannerType()) ? isRmpCalloutsEnabled
                        : BannerWithCallouts.class.isAssignableFrom(BANNER_TYPE_TO_CLASS.get(b.getBannerType())));

        Map<Long, GdiBannerWithSitelinksForReplace> bannerById = listToMap(bannersWithCallouts,
                GdiBannerWithSitelinksForReplace::getId);

        Map<Long, List<Long>> calloutIdsByBannerIds =
                calloutsRepository.getExistingCalloutIdsByBannerIds(shard, bannerIds);

        List<GdiFindAndReplaceBannerCalloutsItem> changedBanners = new ArrayList<>();

        EntryStream.of(bannerById).forKeyValue((bannerId, banner) -> {
            List<Long> oldCalloutIds = calloutIdsByBannerIds.getOrDefault(bannerId, emptyList());
            List<Long> newCalloutIds = replaceCalloutIds(oldCalloutIds, replaceInstruction);
            if (newCalloutIds != null) {
                changedBanners.add(new GdiFindAndReplaceBannerCalloutsItem()
                        .withBannerId(bannerId)
                        .withBannerType(banner.getBannerType())
                        .withAdGroupType(banner.getAdGroupType())
                        .withOldCallouts(oldCalloutIds)
                        .withNewCallouts(newCalloutIds)
                );
            }
        });

        return changedBanners;
    }

    @Nullable
    static List<Long> replaceCalloutIds(List<Long> existedCalloutIds,
                                        GdiFindAndReplaceBannerCalloutsReplaceInstruction replaceInstruction) {
        GdiFindAndReplaceBannerCalloutsReplaceInstructionAction action = replaceInstruction.getAction();
        List<Long> searchCalloutIds = replaceInstruction.getSearchCalloutIds();
        List<Long> replaceCalloutIds = replaceInstruction.getReplaceCalloutIds();

        switch (action) {
            case ADD:
                return changeCalloutsWithAdd(existedCalloutIds, replaceCalloutIds);
            case REMOVE_ALL:
                return changeCalloutsWithRemoveAll(existedCalloutIds);
            case REMOVE_BY_IDS:
                return changeCalloutsWithRemoveByIds(existedCalloutIds, searchCalloutIds);
            case REPLACE_ALL:
                return changeCalloutsWithReplaceAll(existedCalloutIds, replaceCalloutIds);
            case REPLACE_BY_IDS:
                return changeCalloutsWithReplaceByIds(existedCalloutIds, replaceCalloutIds, searchCalloutIds);
            default:
                throw new UnsupportedOperationException("Action " + action.toString() + " is not supported");
        }
    }

    @Nullable
    private static List<Long> changeCalloutsWithAdd(List<Long> existedCalloutIds,
                                                    List<Long> replaceCalloutIds) {

        Set<Long> existedCalloutIdsSet = new HashSet<>(existedCalloutIds);

        List<Long> calloutIdsToAdd = replaceCalloutIds.stream()
                .filter(not(existedCalloutIdsSet::contains))
                .collect(Collectors.toList());

        if (calloutIdsToAdd.isEmpty()) {
            return null;
        }

        ArrayList<Long> resultedCalloutIds = new ArrayList<>(existedCalloutIds);
        resultedCalloutIds.addAll(calloutIdsToAdd);

        return resultedCalloutIds;
    }

    @Nullable
    private static List<Long> changeCalloutsWithRemoveAll(List<Long> existedCalloutIds) {
        return existedCalloutIds.isEmpty() ? null : emptyList();
    }

    @Nullable
    private static List<Long> changeCalloutsWithRemoveByIds(List<Long> existedCalloutIds,
                                                            List<Long> searchCalloutIds) {
        Set<Long> calloutIdsToDelete = new HashSet<>(searchCalloutIds);

        List<Long> resultedCalloutIds = existedCalloutIds.stream()
                .filter(not(calloutIdsToDelete::contains))
                .collect(Collectors.toList());

        return resultedCalloutIds.size() != existedCalloutIds.size() ? resultedCalloutIds : null;
    }

    @Nullable
    private static List<Long> changeCalloutsWithReplaceAll(List<Long> existedCalloutIds,
                                                           List<Long> replaceCalloutIds) {
        return existedCalloutIds.equals(replaceCalloutIds) ? null : replaceCalloutIds;
    }

    @Nullable
    private static List<Long> changeCalloutsWithReplaceByIds(List<Long> existedCalloutIds,
                                                             List<Long> replaceCalloutIds,
                                                             List<Long> searchCalloutIds) {

        // 1 check
        Set<Long> calloutIdsToReplace = new HashSet<>(searchCalloutIds);

        boolean calloutsChanged = existedCalloutIds.stream()
                .anyMatch(calloutIdsToReplace::contains);

        if (!calloutsChanged) {
            return null;
        }

        // 2 remove
        List<Long> resultedCalloutIds = existedCalloutIds.stream()
                .filter(not(calloutIdsToReplace::contains))
                .collect(Collectors.toList());

        // 3 add
        resultedCalloutIds.addAll(replaceCalloutIds);

        return resultedCalloutIds;
    }

    public GridBannerUpdateInfo update(Long operatorUid, ClientId clientId,
                                       List<GdiFindAndReplaceBannerCalloutsItem> banners) {
        return new GridBannerWithCalloutsUpdate(bannersUpdateOperationFactory, operatorUid, clientId,
                banners).update();
    }

    public GridBannerUpdateInfo preview(Long operatorUid, ClientId clientId,
                                        List<GdiFindAndReplaceBannerCalloutsItem> banners) {
        return new GridBannerWithCalloutsUpdate(bannersUpdateOperationFactory, operatorUid, clientId,
                banners).preview();
    }

}
