package ru.yandex.direct.grid.core.entity.banner.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository;
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageAvatarsHost;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerImageNamespace;
import ru.yandex.direct.grid.core.entity.banner.model.GdiInternalBannerExtraInfo;
import ru.yandex.direct.grid.core.entity.banner.model.GdiInternalBannerImageResource;
import ru.yandex.direct.grid.core.entity.banner.model.GdiSimpleImage;

import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class GridInternalBannerExtraInfoService {
    private final ImageDataRepository imageDataRepository;
    private final TemplateResourceService templateResourceService;
    private final BannerTypedRepository bannerTypedRepository;

    public GridInternalBannerExtraInfoService(
            ImageDataRepository imageDataRepository,
            TemplateResourceService templateResourceService,
            BannerTypedRepository bannerTypedRepository) {
        this.imageDataRepository = imageDataRepository;
        this.templateResourceService = templateResourceService;
        this.bannerTypedRepository = bannerTypedRepository;
    }

    /**
     * Собирает дополнительную информацию о баннерах внутренней рекламы. Метод не возвращает в результате
     * записей для других типов баннеров.
     *
     * @param shard     шард клиента
     * @param clientId  идентификатор клиента
     * @param bannerIds набор ID баннеров, для которых нужно получить информацию о внутренней рекламе
     * @return отображение ID баннера -> доп. информация о баннере внутренней рекламы
     */
    Map<Long, GdiInternalBannerExtraInfo> getInternalBannerExtraInfoByBannerId(
            int shard, ClientId clientId, Collection<Long> bannerIds) {

        Map<Long, BannerWithInternalInfo> infoByBannerId =
                bannerTypedRepository.getIdToModelSafely(shard, bannerIds, BannerWithInternalInfo.class);

        if (infoByBannerId.isEmpty()) {
            return emptyMap();
        }

        // информацию о переменных нужно дополнить данными картинок, чтобы выдать потом фронту
        Set<Long> imageResourceIds = findImageResourceIds(infoByBannerId);
        Map<String, Image> hashToImage = calcReferencedImageInfoMap(shard, clientId, infoByBannerId, imageResourceIds);
        return EntryStream.of(infoByBannerId)
                .mapValues(extraInfo -> convertExtraInfo(extraInfo, imageResourceIds, hashToImage))
                .toImmutableMap();
    }

    private Set<Long> findImageResourceIds(Map<Long, BannerWithInternalInfo> infoByBannerId) {
        Set<Long> referencedTemplateIds = listToSet(infoByBannerId.values(), BannerWithInternalInfo::getTemplateId);
        return templateResourceService.getImageResourceIdsByTemplateIds(referencedTemplateIds);
    }

    private Map<String, Image> calcReferencedImageInfoMap(int shard, ClientId clientId,
                                                          Map<Long, BannerWithInternalInfo> infoByBannerId,
                                                          Set<Long> imageResourceIds) {
        Set<String> referencedImageHashes = findImageHashes(infoByBannerId, imageResourceIds);
        List<Image> images = imageDataRepository.getImages(
                shard, clientId, new ImageFilterContainer().withImageHashes(referencedImageHashes));
        return StreamEx.of(images)
                .mapToEntry(Image::getImageHash, identity())
                .toMap();
    }

    private Set<String> findImageHashes(Map<Long, BannerWithInternalInfo> infoByBannerId,
                                        Set<Long> imageResourceIds) {
        return StreamEx.of(infoByBannerId.values())
                .flatCollection(BannerWithInternalInfo::getTemplateVariables)
                .mapToEntry(TemplateVariable::getTemplateResourceId, TemplateVariable::getInternalValue)
                .filterKeys(imageResourceIds::contains)
                .values()
                .nonNull() // только заполненные хэши
                .toSet();
    }

    private List<GdiInternalBannerImageResource> calcImagesOfBannerResources(BannerWithInternalInfo extraInfo,
                                                                             Set<Long> imageResourceIds,
                                                                             Map<String, Image> hashToImage) {
        return StreamEx.of(extraInfo.getTemplateVariables())
                .mapToEntry(TemplateVariable::getTemplateResourceId, TemplateVariable::getInternalValue)
                .filterKeys(imageResourceIds::contains) // только картинки
                .nonNullValues() // только заполенные хэши
                .mapValues(hashToImage::get)
                .nonNullValues() // на всякий случай уберём картинки, которые не нашлись
                .mapValues(this::convertImage)
                .mapKeyValue((resourceId, image) -> new GdiInternalBannerImageResource().withTemplateResourceId(resourceId).withImage(image))
                .toList();
    }

    private GdiInternalBannerExtraInfo convertExtraInfo(BannerWithInternalInfo extraInfo, Set<Long> imageResourceIds,
                                                        Map<String, Image> hashToImage) {
        return new GdiInternalBannerExtraInfo()
                .withDescription(extraInfo.getDescription())
                .withTemplateId(extraInfo.getTemplateId())
                .withTemplateVariables(extraInfo.getTemplateVariables())
                .withModerationInfo(extraInfo.getModerationInfo())
                .withImages(calcImagesOfBannerResources(extraInfo, imageResourceIds, hashToImage));
    }

    private GdiSimpleImage convertImage(Image image) {
        return new GdiSimpleImage()
                .withName(image.getName())
                .withNamespace(toGdiBannerImageNamespace(image.getNamespace()))
                .withImageHash(image.getImageHash())
                .withAvatarsHost(toGdiBannerImageAvatarsHost(image.getAvatarsHost()))
                .withWidth(image.getSize().getWidth().longValue())
                .withHeight(image.getSize().getHeight().longValue())
                .withImageType(image.getImageType())
                .withMdsMeta(image.getMdsMeta())
                .withMdsGroupId(image.getMdsGroupId().longValue());
    }

    private GdiBannerImageAvatarsHost toGdiBannerImageAvatarsHost(AvatarHost avatarsHost) {
        return GdiBannerImageAvatarsHost.fromSource(AvatarHost.toSource(avatarsHost));
    }

    private GdiBannerImageNamespace toGdiBannerImageNamespace(BannerImageFormatNamespace namespace) {
        return GdiBannerImageNamespace.fromSource(BannerImageFormatNamespace.toSource(namespace));
    }
}
