package ru.yandex.direct.grid.core.entity.banner.service.internal;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperation;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetAddOperation;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerItem;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;

public abstract class AbstractBannerWithSitelinksUpdate<T extends GdiFindAndReplaceBannerItem> {
    protected static final String SITELINKS_SET_FIELD_NAME = "sitelinksSet";

    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final SitelinkSetService sitelinkSetService;

    private final long operatorUid;
    private final ClientId clientId;

    private final List<T> updateBanners;

    // Список баннер - сайтлинк, если сайтлинка не существует, то null
    private final List<Pair<T, SitelinkSet>> bannerToSitelinkSetList;

    /**
     * Ключами являются индексы из входного списка {@link #updateBanners}, а значениями -
     * индексы из списка добавляемых наборов сайтлинков.
     * Нужно для построения общего результата валидации
     */
    private Map<Integer, Integer> addSitelinkSetIndexMap = new HashMap<>();

    /**
     * Результат выполнения операции обновления ссылки баннеров
     */
    @Nullable
    private MassResult<Long> bannerResult;
    /**
     * Результат выполнения добавления наборов сайтлинков
     */
    @Nullable
    private MassResult<Long> sitelinkSetResult;

    /**
     * Результат валидации по всем операциям
     */
    private ValidationResult<List<BannerWithSystemFields>, Defect> validationResult;

    public AbstractBannerWithSitelinksUpdate(BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                             SitelinkSetService sitelinkSetService,
                                             List<T> updateBanners, Long operatorUid,
                                             ClientId clientId) {
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.sitelinkSetService = sitelinkSetService;

        this.updateBanners = updateBanners;
        this.operatorUid = operatorUid;
        this.clientId = clientId;

        checkState(updateBanners.stream().noneMatch(t -> t.getBannerId() == null),
                "bannerId in updateBanners must be defined");
        bannerToSitelinkSetList = convertToCoreSitelinkSets(clientId, updateBanners);
    }

    protected abstract List<Pair<T, SitelinkSet>> convertToCoreSitelinkSets(ClientId clientId, List<T> updateBanners);

    protected abstract Map<Long, String> convertToTurbolandingsParams(List<T> updateBanners);

    protected abstract ModelChanges<BannerWithSystemFields> toModelChanges(
            T banner, SitelinkSet sitelinkSet, String turbolandingParams);

    public GridBannerUpdateInfo preview() {
        return processBanners(op -> op.prepare().orElseGet(op::cancel));
    }

    public GridBannerUpdateInfo update() {
        return processBanners(Operation::prepareAndApply);
    }

    private GridBannerUpdateInfo processBanners(Function<Operation<Long>, MassResult<Long>> executor) {
        sitelinkSetResult = ifNotNull(createSitelinkSetsAddOperation(), executor);
        bannerResult = ifNotNull(createBannersUpdateOperation(), executor);

        prepareValidationResult();
        return new GridBannerUpdateInfo(bannerResult, validationResult);
    }

    @Nullable
    private SitelinkSetAddOperation createSitelinkSetsAddOperation() {
        List<SitelinkSet> sitelinkSetsToAdd =
                extractSubList(addSitelinkSetIndexMap, bannerToSitelinkSetList, pair -> pair.getRight() != null,
                        Pair::getRight);

        return sitelinkSetsToAdd.isEmpty() ? null
                : sitelinkSetService.createAddOperation(clientId, sitelinkSetsToAdd, Applicability.PARTIAL);
    }

    @Nullable
    private BannersUpdateOperation<BannerWithSystemFields> createBannersUpdateOperation() {
        Map<Long, String> turbolandingParams = convertToTurbolandingsParams(updateBanners);
        List<ModelChanges<BannerWithSystemFields>> bannerChangesToUpdate = bannerToSitelinkSetList.stream()
                .map(b -> toModelChanges(b.getLeft(), b.getRight(), turbolandingParams.get(b.getLeft().getBannerId())))
                .collect(toList());

        return bannerChangesToUpdate.isEmpty() ? null : bannersUpdateOperationFactory
                .createPartialUpdateOperation(bannerChangesToUpdate, operatorUid, clientId);
    }

    /**
     * Заполняет результат валидации с соответствием индексов входного списка
     */
    private void prepareValidationResult() {

        // Для результата валидации тип и значение баннера не важно
        // Валидация заполняется из соответствующих результатов операций по индексам
        List<BannerWithSystemFields> coreBanners = mapList(updateBanners, b -> new TextBanner().withId(
                b.getBannerId()));
        Map<Integer, BannerWithSystemFields> coreBannerByIndex = EntryStream.of(coreBanners).toMap();

        validationResult = new ValidationResult<>(coreBanners);

        if (bannerResult != null) {
            transferIssuesFromValidationToValidationWithNewValue(bannerResult.getValidationResult(), validationResult);
        }

        if (sitelinkSetResult != null) {
            ValidationResult<?, Defect> sitelinkSetValidationResult = sitelinkSetResult.getValidationResult();
            addSitelinkSetIndexMap.forEach((destIndex, sourceIndex) -> {
                ValidationResult<?, Defect> sourceVr =
                        sitelinkSetValidationResult.getSubResults().get(index(sourceIndex));
                if (sourceVr != null) {
                    ValidationResult<?, Defect> destVr = validationResult.getSubResults().get(index(destIndex));
                    if (destVr == null) {
                        destVr = validationResult
                                .getOrCreateSubValidationResult(index(destIndex), coreBannerByIndex.get(destIndex));
                    }
                    destVr =
                            destVr.getOrCreateSubValidationResult(field(SITELINKS_SET_FIELD_NAME), sourceVr.getValue());
                    transferIssuesFromValidationToValidationWithNewValue(sourceVr, destVr);
                }
            });
        }
    }
}
