package ru.yandex.direct.grid.core.entity.banner.service.internal;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperation;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.banner.model.GdiFindAndReplaceBannerCalloutsItem;
import ru.yandex.direct.grid.core.entity.banner.service.converter.GridFindAndReplaceBannerCalloutsConverter;
import ru.yandex.direct.grid.core.entity.banner.service.internal.container.GridBannerUpdateInfo;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;


public class GridBannerWithCalloutsUpdate {
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;

    private final long operatorUid;
    private final ClientId clientId;

    private final List<GdiFindAndReplaceBannerCalloutsItem> updateBanners;

    public GridBannerWithCalloutsUpdate(BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                        long operatorUid, ClientId clientId,
                                        List<GdiFindAndReplaceBannerCalloutsItem> updateBanners) {
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;

        this.updateBanners = updateBanners;
        this.operatorUid = operatorUid;
        this.clientId = clientId;

        checkState(updateBanners.stream().noneMatch(t -> t.getBannerId() == null),
                "bannerId in updateBanners must be defined");
    }

    public GridBannerUpdateInfo preview() {
        return processBanners(op -> op.prepare().orElseGet(op::cancel));
    }

    public GridBannerUpdateInfo update() {
        return processBanners(Operation::prepareAndApply);
    }

    private GridBannerUpdateInfo processBanners(Function<Operation<Long>, MassResult<Long>> executor) {
        MassResult<Long> bannerResult = ifNotNull(createBannersUpdateOperation(), executor);
        return prepareGridBannerUpdateInfo(bannerResult);
    }

    @Nullable
    private BannersUpdateOperation createBannersUpdateOperation() {
        List<ModelChanges<BannerWithCallouts>> bannerChangesToUpdate = mapList(updateBanners,
                GridFindAndReplaceBannerCalloutsConverter::gdiBannerTextToModelChanges);

        return bannerChangesToUpdate.isEmpty() ? null : bannersUpdateOperationFactory
                .createPartialUpdateOperation(bannerChangesToUpdate, operatorUid, clientId, BannerWithCallouts.class);
    }

    private GridBannerUpdateInfo prepareGridBannerUpdateInfo(@Nullable MassResult<Long> bannerResult) {
        List<BannerWithSystemFields> coreBanners = mapList(updateBanners,
                b -> new TextBanner().withId(b.getBannerId()));
        var validationResult = new ValidationResult<List<BannerWithSystemFields>, Defect>(coreBanners);

        if (bannerResult != null) {
            transferIssuesFromValidationToValidationWithNewValue(bannerResult.getValidationResult(), validationResult);
        }

        return new GridBannerUpdateInfo(bannerResult, validationResult);
    }
}
