package ru.yandex.direct.grid.core.entity.campaign.repository;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.converter.EshowsSettingsConverter;
import ru.yandex.direct.core.entity.campaign.converter.ImpressionStandardTimeConverter;
import ru.yandex.direct.core.entity.campaign.converter.MobileContentConverter;
import ru.yandex.direct.core.entity.campaign.converter.MobileContentInfoConverter;
import ru.yandex.direct.core.entity.campaign.converter.PriceFlightConverter;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.BudgetDisplayFormat;
import ru.yandex.direct.core.entity.campaign.model.CampAimType;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWarnPlaceInterval;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.PlacementType;
import ru.yandex.direct.core.entity.campaign.model.RfCloseByClickType;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.repository.type.InternalCampaignTypeSupport;
import ru.yandex.direct.core.entity.minuskeywordspack.MinusKeywordsPackUtils;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatus;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStopReason;
import ru.yandex.direct.grid.model.campaign.GdSurveyStatus;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignDelayedOperation;
import ru.yandex.direct.grid.model.campaign.GdiCampaignMediaplanStatus;
import ru.yandex.direct.grid.model.campaign.GdiCampaignMetatype;
import ru.yandex.direct.grid.model.campaign.GdiCampaignOptsStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStatusBsSynced;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiContentLanguage;
import ru.yandex.direct.grid.model.campaign.GdiDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.GdiInternalCampaignRestrictionType;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.common.db.PpcPropertyNames.OPTIMIZE_CAMP_BANNERS_FLAGS;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_CAMP_AGGR_STATUS_FOR_ACTIVE_BANNERS_FLAG;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToInteger;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToIntegerWithZeroNullification;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromYesNoEnumFieldToBoolean;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;
import static ru.yandex.direct.dbschema.ppc.Tables.AGGR_STATUSES_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_PHONES;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_PROMOACTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_ADDITIONAL_DATA;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_DIALOGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPERATIONS_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_PROMOCODES;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MEDIAPLAN_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MEDIAPLAN_STATS;
import static ru.yandex.direct.dbschema.ppc.Tables.METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APPS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APP_TRACKERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.SUBCAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.USER_CAMPAIGNS_FAVORITE;
import static ru.yandex.direct.dbschema.ppc.Tables.WIDGET_PARTNER_CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.localDateTimeAddMonth;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromSet;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
@ParametersAreNonnullByDefault
public class GridCampaignRepository {
    private static final Set<CampaignsType> CAMPAIGN_TYPES = StreamEx.of(UNDER_WALLET)
            .append(CampaignType.WALLET)

            .append(CampaignType.INTERNAL_AUTOBUDGET)
            .append(CampaignType.INTERNAL_DISTRIB)
            .append(CampaignType.INTERNAL_FREE)

            .append(CampaignType.CONTENT_PROMOTION)

            .map(CampaignType::toSource)
            .toSet();

    private static final Field<Object> HAS_BANNERS_FIELD = DSL.selectOne()
            .from(BANNERS)
            .where(BANNERS.CID.equal(CAMPAIGNS.CID))
            .limit(1)
            .asField("has_banners");

    private static final String HAS_NOT_ACRHIVED_BANNERS_FIELD = "has_not_archived_banners";
    private static final Field<Object> HAS_NOT_ACRHIVED_BANNERS_FIELD_OLD = DSL.selectOne()
            .from(BANNERS)
            .where(BANNERS.CID.equal(CAMPAIGNS.CID).and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)))
            .limit(1)
            .asField(HAS_NOT_ACRHIVED_BANNERS_FIELD);
    private static final Field<Object> HAS_NOT_ACRHIVED_BANNERS_FIELD_OPTIMIZED = DSL.val((Object) null)
            .as(HAS_NOT_ACRHIVED_BANNERS_FIELD);

    private static final Field<Object> HAS_ACTIVE_BANNERS_FIELD = DSL.selectOne()
            .from(BANNERS
                    .join(PHRASES).on(BANNERS.PID.equal(PHRASES.PID)))
            .where(
                    BANNERS.CID.equal(CAMPAIGNS.CID),
                    BANNERS.STATUS_ARCH.equal(BannersStatusarch.No),
                    BANNERS.STATUS_SHOW.equal(BannersStatusshow.Yes),
                    DSL.or(
                            BANNERS.STATUS_ACTIVE.equal(BannersStatusactive.Yes),
                            DSL.and(
                                    PHRASES.STATUS_POST_MODERATE.equal(PhrasesStatuspostmoderate.Yes),
                                    BANNERS.STATUS_POST_MODERATE.equal(BannersStatuspostmoderate.Yes),
                                    DSL.condition(DSL.when(
                                            BANNERS.BANNER_TYPE.equal(BannersBannerType.text),
                                            DSL.field(DSL.ifnull(BANNERS.HREF, "").notEqual("")
                                                    .or(BANNERS.PHONEFLAG.equal(BannersPhoneflag.Yes))))
                                            .otherwise(Boolean.TRUE))
                            )
                    ))
            .limit(1)
            .asField("has_active_banners");

    private static final Field<Object> HAS_ACTIVE_BANNERS_FIELD_FROM_AGGR_STATUS = DSL.field(
            // можно уменьшить количество сырого sql так
            //     private static final Field<Boolean> HAS_ACTIVE_BANNERS_FIELD_FROM_AGGR_STATUS = DSL.field(
            //             DSL.or(
            //                     DSL.field("aggr_statuses_campaigns.aggr_data->\"$.cnts.s.RUN_OK\"").gt(0),
            //                     DSL.field("aggr_statuses_campaigns.aggr_data->\"$.cnts.s.RUN_WARN\"").gt(0)
            //             )
            //     ).as("has_active_banners");
            // но сейчас мешает отличающийся тип HAS_ACTIVE_BANNERS_FIELD, лучше делать после убирания проперти
            "aggr_statuses_campaigns.aggr_data->\"$.cnts.s.RUN_OK\" > 0 OR aggr_statuses_campaigns.aggr_data->\"$.cnts.s.RUN_WARN\" > 0"
    ).as("has_active_banners");

    // медиаплан считается новым если его создали не более трех месяцев назад
    private static final int NEW_MEDIAPLAN_INTERVAL_IN_MONTH = 3;

    private static final String HAS_NEW_MEDIAPLAN_FIELD = "has_new_mediaplan";
    private static final Field<Object> HAS_NEW_MEDIAPLAN_FIELD_OLD = DSL.selectOne()
            .from(MEDIAPLAN_STATS)
            .where(MEDIAPLAN_STATS.CID.equal(CAMPAIGNS.CID)
                    .and(MEDIAPLAN_STATS.CREATE_TIME.greaterOrEqual(
                            localDateTimeAddMonth(DSL.currentLocalDateTime(), -NEW_MEDIAPLAN_INTERVAL_IN_MONTH))))
            .limit(1)
            .asField(HAS_NEW_MEDIAPLAN_FIELD);
    private static final Field<Object> HAS_NEW_MEDIAPLAN_FIELD_OPTIMIZED = DSL.val((Object) null)
            .as(HAS_NEW_MEDIAPLAN_FIELD);

    private static final String HAS_MEDIAPLAN_BANNERS_FIELD = "has_mediaplan_banners";
    private static final Field<Object> HAS_MEDIAPLAN_BANNERS_FIELD_OLD = DSL.selectOne()
            .from(MEDIAPLAN_BANNERS)
            .where(MEDIAPLAN_BANNERS.CID.equal(CAMPAIGNS.CID))
            .limit(1)
            .asField(HAS_MEDIAPLAN_BANNERS_FIELD);
    private static final Field<Object> HAS_MEDIAPLAN_BANNERS_FIELD_OPTIMIZED = DSL.val((Object) null)
            .as(HAS_MEDIAPLAN_BANNERS_FIELD);

    private static final Field<Object> HAS_HIERARCHICAL_MULTIPLIERS = DSL.selectOne()
            .from(HIERARCHICAL_MULTIPLIERS)
            .where(
                    HIERARCHICAL_MULTIPLIERS.CID.equal(CAMPAIGNS.CID)
                            .and(HIERARCHICAL_MULTIPLIERS.PID.isNull())
                            .and(HIERARCHICAL_MULTIPLIERS.IS_ENABLED.ne(0L)))
            .limit(1)
            .asField("has_hierarchical_multipliers");
    private static final Field<Object> HAS_METRIKA_COUNTERS_ECOMMERCE = DSL.selectOne()
            .from(METRIKA_COUNTERS)
            .where(
                    METRIKA_COUNTERS.CID.equal(CAMPAIGNS.CID)
                            .and(METRIKA_COUNTERS.IS_DELETED.ne(1L))
                            .and(METRIKA_COUNTERS.HAS_ECOMMERCE.eq(1L)))
            .limit(1)
            .asField("has_metrika_counters_ecommerce");

    private static final List<CampaignsType> INTERNAL_CAMPAIGN_TYPES =
            mapList(CampaignTypeKinds.INTERNAL, CampaignType::toSource);

    private final DslContextProvider contextProvider;
    private final JooqReaderWithSupplier<GdiCampaign> campaignJooqReader;
    private final JooqReaderWithSupplier<GdiBaseCampaign> baseCampaignJooqReader;
    private final PpcProperty<Boolean> optimizeCampBannersFlagsProperty;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    @SuppressWarnings({"methodlength", "java:S3252"})
    public GridCampaignRepository(
            DslContextProvider contextProvider,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.contextProvider = contextProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.optimizeCampBannersFlagsProperty = ppcPropertiesSupport.get(
                OPTIMIZE_CAMP_BANNERS_FLAGS, Duration.ofSeconds(60)
        );

        baseCampaignJooqReader = JooqReaderWithSupplierBuilder.builder(GdiBaseCampaign::new)
                .readProperty(GdiBaseCampaign.ID, fromField(CAMPAIGNS.CID))
                .readProperty(GdiBaseCampaign.ORDER_ID, fromField(CAMPAIGNS.ORDER_ID))
                .readProperty(GdiBaseCampaign.WALLET_ID, fromField(CAMPAIGNS.WALLET_CID))
                .readProperty(GdiBaseCampaign.CLIENT_ID, fromField(CAMPAIGNS.CLIENT_ID))
                .readProperty(GdiBaseCampaign.USER_ID, fromField(CAMPAIGNS.UID))
                .readProperty(GdiBaseCampaign.MANAGER_USER_ID, fromField(CAMPAIGNS.MANAGER_UID))
                .readProperty(GdiBaseCampaign.AGENCY_USER_ID, fromField(CAMPAIGNS.AGENCY_UID))
                .readProperty(GdiBaseCampaign.AGENCY_ID, fromField(CAMPAIGNS.AGENCY_ID))
                .readProperty(GdiBaseCampaign.ARCHIVED, fromYesNoEnumFieldToBoolean(CAMPAIGNS.ARCHIVED))
                .readProperty(GdiBaseCampaign.TYPE, fromField(CAMPAIGNS.TYPE).by(CampaignType::fromSource))

                .readProperty(GdiBaseCampaign.START_DATE, fromField(CAMPAIGNS.START_TIME))
                .readProperty(GdiBaseCampaign.FINISH_DATE, fromField(CAMPAIGNS.FINISH_TIME))

                .readProperty(GdiBaseCampaign.SUM, fromField(CAMPAIGNS.SUM))
                .readProperty(GdiBaseCampaign.SUM_SPENT, fromField(CAMPAIGNS.SUM_SPENT))
                .readProperty(GdiBaseCampaign.SUM_LAST, fromField(CAMPAIGNS.SUM_LAST))
                .readProperty(GdiBaseCampaign.SUM_TO_PAY, fromField(CAMPAIGNS.SUM_TO_PAY))
                .readProperty(GdiBaseCampaign.CURRENCY_CODE, fromField(CAMPAIGNS.CURRENCY)
                        .by(c -> CurrencyCode.valueOf(c.name().toUpperCase())))
                .readProperty(GdiBaseCampaign.CURRENCY_CONVERTED, fromYesNoEnumFieldToBoolean(CAMPAIGNS.CURRENCY_CONVERTED))
                .readProperty(GdiBaseCampaign.SHOWS, fromField(CAMPAIGNS.SHOWS)
                        .by(shows -> nvl(shows, 0L)))
                .readProperty(GdiBaseCampaign.LAST_SHOW_TIME, fromField(CAMPAIGNS.LAST_SHOW_TIME))
                .readProperty(GdiBaseCampaign.CREATE_TIME, fromField(CAMP_OPTIONS.CREATE_TIME))
                .readProperty(GdiBaseCampaign.NO_PAY, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_NO_PAY))
                .readProperty(GdiBaseCampaign.SHOWING, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_SHOW))
                .readProperty(GdiBaseCampaign.STATUS_MODERATE, fromField(CAMPAIGNS.STATUS_MODERATE)
                        .by(CampaignStatusModerate::fromSource))
                .readProperty(GdiBaseCampaign.STATUS_POST_MODERATE, fromField(CAMP_OPTIONS.STATUS_POST_MODERATE)
                        .by(CampaignStatusPostmoderate::fromSource))
                .readProperty(GdiBaseCampaign.DAY_BUDGET, fromField(CAMPAIGNS.DAY_BUDGET))
                .readProperty(GdiBaseCampaign.DAY_BUDGET_STOP_TIME, fromField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME))
                .readProperty(GdiBaseCampaign.DAY_BUDGET_SHOW_MODE, fromField(CAMPAIGNS.DAY_BUDGET_SHOW_MODE)
                        .by(GdiDayBudgetShowMode::fromSource))
                .readProperty(GdiBaseCampaign.PLATFORM, fromField(CAMPAIGNS.PLATFORM).by(CampaignsPlatform::fromSource))
                .readProperty(GdiBaseCampaign.STRATEGY_NAME, fromField(CAMPAIGNS.STRATEGY_NAME)
                        .by(GdiCampaignStrategyName::fromSource))
                .readProperty(GdiBaseCampaign.OPTS_STRATEGY_NAME, fromField(CAMP_OPTIONS.STRATEGY)
                        .by(GdiCampaignOptsStrategyName::fromSource))
                .readProperty(GdiBaseCampaign.STRATEGY_DATA, fromField(CAMPAIGNS.STRATEGY_DATA).by(String.class::cast))
                .readProperty(GdiBaseCampaign.IS_BRAND_LIFT_HIDDEN,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isBrandLiftHiddenFromDb))
                .readProperty(GdiBaseCampaign.AB_SEGMENT_RETARGETING_CONDITION_ID,
                        fromField(CAMPAIGNS.AB_SEGMENT_RET_COND_ID))
                .readProperty(GdiBaseCampaign.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID,
                        fromField(CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID))

                .readProperty(GdiBaseCampaign.METRIKA_COUNTERS,
                        fromField(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS).by(CampaignMappings::metrikaCountersFromDbToList))
                .readProperty(GdiBaseCampaign.STOP_TIME, fromField(CAMP_OPTIONS.STOP_TIME))
                .readProperty(GdiBaseCampaign.FLIGHT_STATUS_APPROVE, fromField(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE)
                        .by(PriceFlightConverter::statusApproveFromDbFormat))
                .readProperty(GdiBaseCampaign.FLIGHT_STATUS_CORRECT, fromField(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT)
                        .by(PriceFlightConverter::statusCorrectFromDbFormat))

                .readProperty(GdiBaseCampaign.SOURCE, fromField(CAMPAIGNS.SOURCE).by(GdiCampaignSource::fromSource))

                .readProperty(GdiBaseCampaign.MOBILE_CONTENT_INFO,
                        fromFields(Set.of(MOBILE_CONTENT.OS_TYPE,
                                MOBILE_CONTENT.STORE_CONTENT_ID,
                                MOBILE_CONTENT.STORE_COUNTRY,
                                MOBILE_APPS.STORE_HREF,
                                MOBILE_CONTENT.MIN_OS_VERSION,
                                MOBILE_APPS.MIN_OS_VERSION,
                                MOBILE_CONTENT.ICON_HASH,
                                MOBILE_CONTENT.NAME,
                                MOBILE_APP_TRACKERS.TRACKER_ID,
                                MOBILE_APP_TRACKERS.TRACKING_SYSTEM,
                                MOBILE_APP_TRACKERS.URL,
                                MOBILE_APP_TRACKERS.IMPRESSION_URL))
                                .by(MobileContentInfoConverter::fromDbFormat))
                .readProperty(GdiBaseCampaign.PROMOCODE_RESTRICTED_DOMAIN, fromField(CAMP_PROMOCODES.RESTRICTED_DOMAIN))

                .readProperty(GdiBaseCampaign.LAST_PAY_DATE, fromField(CAMP_OPTIONS.LAST_PAY_TIME))
                .readProperty(GdiBaseCampaign.IS_UNIVERSAL,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isUniversalFromDb))

                .readProperty(GdiBaseCampaign.MEDIAPLAN_STATUS, fromField(CAMP_OPTIONS.MEDIAPLAN_STATUS)
                        .by(GdiCampaignMediaplanStatus::fromSource))

                .readProperty(GdiBaseCampaign.METATYPE, fromField(CAMPAIGNS.METATYPE).by(GdiCampaignMetatype::fromSource))
                .build();

        campaignJooqReader = JooqReaderWithSupplierBuilder.builder(baseCampaignJooqReader, GdiCampaign::new)
                .readProperty(GdiCampaign.NAME, fromField(CAMPAIGNS.NAME))
                .readProperty(GdiCampaign.DESCRIPTION, fromField(CAMP_OPTIONS.CAMP_DESCRIPTION))
                .readProperty(GdiCampaign.EMPTY, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_EMPTY))
                .readProperty(GdiCampaign.GEO, fromField(CAMPAIGNS.GEO))
                .readProperty(GdiCampaign.STRATEGY_ID, fromField(CAMPAIGNS.STRATEGY_ID))

                .readProperty(GdiCampaign.HAS_TITLE_SUBSTITUTION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasTitleSubstituteFromDb))
                .readProperty(GdiCampaign.HAS_EXTENDED_GEO_TARGETING,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasExtendedGeoTargetingFromDb))
                .readProperty(GdiCampaign.USE_CURRENT_REGION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::useCurrentRegionFromDb))
                .readProperty(GdiCampaign.USE_REGULAR_REGION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::useRegularRegionFromDb))
                .readProperty(GdiCampaign.HAS_ADD_METRIKA_TAG_TO_URL,
                        fromLongFieldToBoolean(CAMP_OPTIONS.STATUS_CLICK_TRACK))

                .readProperty(GdiCampaign.HAS_ENABLE_CPC_HOLD,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasEnableCpcHoldFromDb))
                .readProperty(GdiCampaign.ENABLE_COMPANY_INFO,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::enableCompanyInfoFromDb))
                .readProperty(GdiCampaign.IS_ALONE_TRAFARET_ALLOWED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isAloneTrafaretAllowedFromDb))
                .readProperty(GdiCampaign.HAS_TURBO_SMARTS,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasTurboSmarts))
                .readProperty(GdiCampaign.IS_S2S_TRACKING_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isS2sTrackingEnabled))
                .readProperty(GdiCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isRecommendationsManagementEnabled))
                .readProperty(GdiCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isPriceRecommendationsManagementEnabled))
                .readProperty(GdiCampaign.TURBO_APPS_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasTurboAppFromDb))
                .readProperty(GdiCampaign.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isSimplifiedStrategyViewEnabled))
                .readProperty(GdiCampaign.IS_MEANINGFUL_GOALS_VALUES_FROM_METRIKA_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isMeaningfulGoalsValuesFromMetrika))
                .readProperty(GdiCampaign.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isRequireFiltrationByDontShowDomains))
                .readProperty(GdiCampaign.HAS_ADD_OPENSTAT_TAG_TO_URL,
                        fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_OPEN_STAT))
                .readProperty(GdiCampaign.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isOrderPhraseLengthPrecedenceEnabled))
                .readProperty(GdiCampaign.IS_NEW_IOS_VERSION_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isNewIosVersionEnabled))
                .readProperty(GdiCampaign.IS_SKAD_NETWORK_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isSkadNetworkEnabled))
                .readProperty(GdiCampaign.IS_ALLOWED_ON_ADULT_CONTENT,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isAllowedOnAdultContentFromDb))
                .readProperty(GdiCampaign.IS_WW_MANAGED_ORDER,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isWwManagedOrder))
                .readProperty(GdiCampaign.TIME_TARGET, fromField(CAMPAIGNS.TIME_TARGET)
                        .by(CampaignMappings::timeTargetFromDb))
                .readProperty(GdiCampaign.TIMEZONE_ID, fromField(CAMPAIGNS.TIMEZONE_ID))
                .readProperty(GdiCampaign.ACTIVE, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_ACTIVE))
                .readProperty(GdiCampaign.STATUS_BS_SYNCED, fromField(CAMPAIGNS.STATUS_BS_SYNCED)
                        .by(GdiCampaignStatusBsSynced::fromSource))
                .readProperty(GdiCampaign.CLICKS, fromField(CAMPAIGNS.CLICKS)
                        .by(clicks -> nvl(clicks, 0L)))
                .readProperty(GdiCampaign.DAY_BUDGET_DAILY_CHANGE_COUNT,
                        fromField(CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT).by(Long::intValue))
                .readProperty(GdiCampaign.CONTEXT_LIMIT, fromLongFieldToInteger(CAMPAIGNS.CONTEXT_LIMIT))
                .readProperty(GdiCampaign.MINUS_KEYWORDS, fromField(CAMP_OPTIONS.MINUS_WORDS)
                        .by(MinusKeywordsPackUtils::minusKeywordsFromJson))
                .readProperty(GdiCampaign.BANNER_HREF_PARAMS, fromField(CAMP_OPTIONS.HREF_PARAMS))

                .readProperty(GdiCampaign.DISABLED_DOMAINS, fromField(CAMPAIGNS.DONT_SHOW))
                .readProperty(GdiCampaign.DISABLED_SSP, fromField(CAMPAIGNS.DISABLED_SSP).by(String.class::cast))
                .readProperty(GdiCampaign.DISABLED_IPS, fromField(CAMPAIGNS.DISABLED_IPS))
                .readProperty(GdiCampaign.DISABLED_VIDEO_PLACEMENTS, fromField(CAMPAIGNS.DISABLED_VIDEO_PLACEMENTS)
                        .by(CampaignMappings::disabledVideoPlacementsFromJson))

                .readProperty(GdiCampaign.PLACEMENT_TYPES,
                        fromSet(CAMP_OPTIONS.PLACEMENT_TYPES, PlacementType::fromTypedValue))

                .readProperty(GdiCampaign.DELAYED_OPERATION, fromField(CAMP_OPERATIONS_QUEUE.OPERATION)
                        .by(GdiCampaignDelayedOperation::fromSource))
                .readProperty(GdiCampaign.HAS_SITE_MONITORING,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.STATUS_METRICA_CONTROL))
                .readProperty(GdiCampaign.EXCLUDE_PAUSED_COMPETING_ADS,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.FAIR_AUCTION))
                .readProperty(GdiCampaign.HAS_BROAD_MATCH, fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.BROAD_MATCH_FLAG))
                .readProperty(GdiCampaign.BROAD_MATCH_LIMIT, fromLongFieldToInteger(CAMP_OPTIONS.BROAD_MATCH_LIMIT))
                .readProperty(GdiCampaign.BROAD_MATCH_GOAL_ID, fromField(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID))

                .readProperty(GdiCampaign.SMS_TIME,
                        fromField(CAMP_OPTIONS.SMS_TIME).by(CampaignMappings::smsTimeFromDb))
                .readProperty(GdiCampaign.SMS_FLAGS, fromSet(CAMP_OPTIONS.SMS_FLAGS, SmsFlag.class))
                .readProperty(GdiCampaign.EMAIL, fromField(CAMP_OPTIONS.EMAIL))
                .readProperty(GdiCampaign.ENABLE_CHECK_POSITION_EVENT,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.SEND_WARN))
                .readProperty(GdiCampaign.CHECK_POSITION_INTERVAL, fromField(CAMP_OPTIONS.WARN_PLACE_INTERVAL)
                        .by(CampaignWarnPlaceInterval::fromSource))
                .readProperty(GdiCampaign.WARNING_BALANCE, fromLongFieldToInteger(CAMP_OPTIONS.MONEY_WARNING_VALUE))
                .readProperty(GdiCampaign.ENABLE_SEND_ACCOUNT_NEWS,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.SEND_ACC_NEWS))
                .readProperty(GdiCampaign.ENABLE_OFFLINE_STAT_NOTICE,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.OFFLINE_STAT_NOTICE))
                .readProperty(GdiCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT, fromField(CAMP_OPTIONS.EMAIL_NOTIFICATIONS)
                        .by(CampaignConverter::hasPausedByDayBudgetFromDb))

                .readProperty(GdiCampaign.FAVORITE, fromField(USER_CAMPAIGNS_FAVORITE.CID).by(Objects::nonNull))

                .readProperty(GdiCampaign.CONTENT_LANGUAGE,
                        fromField(CAMP_OPTIONS.CONTENT_LANG).by(GdiContentLanguage::fromTypedValue))
                .readProperty(GdiCampaign.ATTRIBUTION_MODEL,
                        fromField(CAMPAIGNS.ATTRIBUTION_MODEL).by(CampaignAttributionModel::fromSource))
                .readProperty(GdiCampaign.ALLOWED_PAGE_IDS,
                        fromField(CAMP_OPTIONS.ALLOWED_PAGE_IDS).by(CampaignMappings::pageIdsFromDbNullWhenEmpty))
                .readProperty(GdiCampaign.DISALLOWED_PAGE_IDS,
                        fromField(CAMP_OPTIONS.DISALLOWED_PAGE_IDS).by(CampaignMappings::pageIdsFromDbNullWhenEmpty))
                .readProperty(GdiCampaign.ALLOWED_DOMAINS,
                        fromField(CAMP_OPTIONS.ALLOWED_DOMAINS).by(CampaignMappings::stringListFromDbJsonFormat))
                .readProperty(GdiCampaign.ALLOWED_SSP,
                        fromField(CAMP_OPTIONS.ALLOWED_SSP).by(CampaignMappings::stringListFromDbJsonFormat))
                .readProperty(GdiCampaign.ALLOWED_FRONTPAGE_TYPES,
                        fromField(CAMPAIGNS_CPM_YNDX_FRONTPAGE.ALLOWED_FRONTPAGE_TYPES).by(CampaignMappings::allowedYndxFrontpageTypesFromDb))

                .readProperty(GdiCampaign.CLIENT_DIALOG_ID, fromField(CAMP_DIALOGS.CLIENT_DIALOG_ID))

                .readProperty(GdiCampaign.MEANINGFUL_GOALS, fromField(CAMP_OPTIONS.MEANINGFUL_GOALS)
                        .by(CampaignConverter::meaningfulGoalsFromDb))
                .readProperty(GdiCampaign.BRAND_SURVEY_ID, fromField(CAMP_OPTIONS.BRAND_SURVEY_ID))
                .readProperty(GdiCampaign.CONTACT_INFO, fromField(CAMP_OPTIONS.CONTACTINFO)
                        .by(CampaignConverter::vcardFromDb))
                .readProperty(GdiCampaign.IMPRESSION_RATE_COUNT,
                        fromLongFieldToIntegerWithZeroNullification(CAMPAIGNS.RF))
                .readProperty(GdiCampaign.IMPRESSION_RATE_INTERVAL_DAYS,
                        fromLongFieldToIntegerWithZeroNullification(CAMPAIGNS.RF_RESET))
                .readProperty(GdiCampaign.DEFAULT_PERMALINK_ID,
                        fromField(CAMPAIGN_PERMALINKS.PERMALINK_ID))
                .readProperty(GdiCampaign.DEFAULT_CHAIN_ID,
                        fromField(CAMPAIGN_PERMALINKS.CHAIN_ID).by(RepositoryUtils::zeroToNull))
                .readProperty(GdiCampaign.DEFAULT_TRACKING_PHONE_ID, fromField(CAMPAIGN_PHONES.CLIENT_PHONE_ID))
                .readProperty(GdiCampaign.DEVICE_TYPE_TARGETING,
                        fromField(CAMPAIGNS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING)
                                .by(MobileContentConverter::deviceTypeTargetingFromDb))
                .readProperty(GdiCampaign.NETWORK_TARGETING, fromField(CAMPAIGNS_MOBILE_CONTENT.NETWORK_TARGETING)
                        .by(MobileContentConverter::networkTargetingsFromDb))
                .readProperty(GdiCampaign.IS_INSTALLED_APP, fromField(CAMPAIGNS_MOBILE_CONTENT.IS_INSTALLED_APP)
                        .by(RepositoryUtils::booleanFromLong))
                .readProperty(GdiCampaign.MOBILE_APP_ID, fromField(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .readProperty(GdiCampaign.ALTERNATIVE_APP_STORES,
                        fromField(CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES)
                                .by(MobileContentConverter::altAppStoresFromDb))
                .readProperty(GdiCampaign.PRICE_PACKAGE_ID, fromField(CAMPAIGNS_CPM_PRICE.PACKAGE_ID))
                .readProperty(GdiCampaign.FLIGHT_TARGETINGS_SNAPSHOT, fromField(CAMPAIGNS_CPM_PRICE.TARGETINGS_SNAPSHOT)
                        .by(PriceFlightConverter::targetingsSnapshotFromDbFormat))
                .readProperty(GdiCampaign.FLIGHT_ORDER_VOLUME, fromField(CAMPAIGNS_CPM_PRICE.ORDER_VOLUME))
                .readProperty(GdiCampaign.FLIGHT_REASON_INCORRECT, fromField(CAMPAIGNS_CPM_PRICE.REASON_INCORRECT)
                        .by(PriceFlightConverter::reasonIncorrectFromDbFormat))
                .readProperty(GdiCampaign.IMPRESSION_STANDARD_TIME, fromField(CAMP_OPTIONS.IMPRESSION_STANDARD_TIME)
                        .by(ImpressionStandardTimeConverter::fromDbFormat))
                .readProperty(GdiCampaign.ESHOWS_SETTINGS,
                        fromFields(CAMP_OPTIONS.ESHOWS_BANNER_RATE,
                                CAMP_OPTIONS.ESHOWS_VIDEO_RATE,
                                CAMP_OPTIONS.ESHOWS_VIDEO_TYPE)
                                .by(EshowsSettingsConverter::fromDbFormat))
                .readProperty(GdiCampaign.INTERNAL_AD_PLACE_ID, fromField(CAMPAIGNS_INTERNAL.PLACE_ID))
                .readProperty(GdiCampaign.INTERNAL_AD_IS_MOBILE, fromField(CAMPAIGNS_INTERNAL.IS_MOBILE)
                        .by(RepositoryUtils::booleanFromLong))
                .readProperty(GdiCampaign.INTERNAL_AD_PAGE_ID, fromField(CAMPAIGNS_INTERNAL.PAGE_IDS)
                        .by(InternalCampaignTypeSupport::pageIdsFromDb))
                .readProperty(GdiCampaign.INTERNAL_AD_RESTRICTION_TYPE, fromField(CAMPAIGNS_INTERNAL.RESTRICTION_TYPE)
                        .by(GdiInternalCampaignRestrictionType::fromSource))
                .readProperty(GdiCampaign.INTERNAL_AD_RESTRICTION_VALUE,
                        fromField(CAMPAIGNS_INTERNAL.RESTRICTION_VALUE))
                .readProperty(GdiCampaign.INTERNAL_AD_ROTATION_GOAL_ID, fromField(CAMPAIGNS_INTERNAL.ROTATION_GOAL_ID))
                .readProperty(GdiCampaign.INTERNAL_AD_RF_CLOSE_BY_CLICK, fromField(CAMPAIGNS_INTERNAL.RF_CLOSE_BY_CLICK)
                        .by(RfCloseByClickType::fromSource))
                .readProperty(GdiCampaign.INTERNAL_AD_MAX_CLICKS_COUNT, fromLongFieldToInteger(CAMPAIGNS_INTERNAL.MAX_CLICKS_COUNT))
                .readProperty(GdiCampaign.INTERNAL_AD_MAX_CLICKS_PERIOD, fromLongFieldToInteger(CAMPAIGNS_INTERNAL.MAX_CLICKS_PERIOD))
                .readProperty(GdiCampaign.INTERNAL_AD_MAX_STOPS_COUNT, fromLongFieldToInteger(CAMPAIGNS_INTERNAL.MAX_STOPS_COUNT))
                .readProperty(GdiCampaign.INTERNAL_AD_MAX_STOPS_PERIOD, fromLongFieldToInteger(CAMPAIGNS_INTERNAL.MAX_STOPS_PERIOD))

                .readProperty(GdiCampaign.HREF, fromField(CAMP_ADDITIONAL_DATA.HREF))
                .readProperty(GdiCampaign.COMPANY_NAME, fromField(CAMP_ADDITIONAL_DATA.COMPANY_NAME))
                .readProperty(GdiCampaign.BUSINESS_CATEGORY, fromField(CAMP_ADDITIONAL_DATA.BUSINESS_CATEGORY))
                .readProperty(GdiCampaign.CAMP_AIM_TYPE, fromField(CAMP_ADDITIONAL_DATA.CAMP_AIM_TYPE)
                        .by(CampAimType::fromSource))
                .readProperty(GdiCampaign.BUDGET_DISPLAY_FORMAT, fromField(CAMP_ADDITIONAL_DATA.BUDGET_DISPLAY_FORMAT)
                        .by(BudgetDisplayFormat::fromSource))

                .readProperty(GdiCampaign.CALLTRACKING_SETTINGS_ID,
                        fromField(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .readProperty(GdiCampaign.PROMO_EXTENSION_ID, fromField(CAMPAIGN_PROMOACTIONS.PROMOACTION_ID))

                .readProperty(GdiCampaign.MASTER_CID, fromField(SUBCAMPAIGNS.MASTER_CID))
                .readProperty(GdiCampaign.WIDGET_PARTNER_ID, fromField(WIDGET_PARTNER_CAMPAIGNS.WIDGET_PARTNER_ID))
                .build();
    }

    /**
     * Получить список всех кампаний клиента. Получаются только те кампании, которые доступны в интерфейсе директа, а
     * также кошельки.
     *
     * @param uidClientIdShard контейнер с данными пользователя(шард, uid, clientId)
     */
    public List<GdiCampaign> getAllCampaigns(UidClientIdShard uidClientIdShard) {
        return getCampaignsAndWallets(uidClientIdShard, DSL.trueCondition());
    }

    /**
     * Получить список всех кампаний клиента. Получаются только те кампании, которые доступны в интерфейсе директа, а
     * также кошельки.
     *
     * @param uidClientIdShard     контейнер с данными пользователя(шард, uid, clientId)
     * @param nonWalletCampaignIds набор id, по которому получаем кампании.
     *                             Если null - достаем все кампании клиента
     */
    public List<GdiCampaign> getCampaignsAndAllWallets(UidClientIdShard uidClientIdShard,
                                                       Collection<Long> nonWalletCampaignIds) {
        Condition condition = DSL.trueCondition();
        if (!CollectionUtils.isEmpty(nonWalletCampaignIds)) {
            condition = condition.and(CAMPAIGNS.CID.in(nonWalletCampaignIds));
        }
        condition = condition.or(CAMPAIGNS.TYPE.eq(CampaignsType.wallet));
        return getCampaignsAndWallets(uidClientIdShard, condition);
    }

    private List<GdiCampaign> getCampaignsAndWallets(UidClientIdShard uidClientIdShard,
                                                     Condition extraFetchCondition) {
        Set<Field<Object>> fields = new HashSet<>(Set.of(HAS_BANNERS_FIELD,
                HAS_HIERARCHICAL_MULTIPLIERS,
                HAS_METRIKA_COUNTERS_ECOMMERCE
        ));
        if (optimizeCampBannersFlagsProperty.getOrDefault(false)) {
            fields.add(HAS_NOT_ACRHIVED_BANNERS_FIELD_OPTIMIZED);
            fields.add(HAS_NEW_MEDIAPLAN_FIELD_OPTIMIZED);
            fields.add(HAS_MEDIAPLAN_BANNERS_FIELD_OPTIMIZED);
        } else {
            fields.add(HAS_NOT_ACRHIVED_BANNERS_FIELD_OLD);
            fields.add(HAS_NEW_MEDIAPLAN_FIELD_OLD);
            fields.add(HAS_MEDIAPLAN_BANNERS_FIELD_OLD);
        }
        if (!"1".equals(ppcPropertiesSupport.get(USE_CAMP_AGGR_STATUS_FOR_ACTIVE_BANNERS_FLAG.getName()))) {
            fields.add(HAS_ACTIVE_BANNERS_FIELD);
        } else {
            fields.add(HAS_ACTIVE_BANNERS_FIELD_FROM_AGGR_STATUS);
        }
        return contextProvider.ppc(uidClientIdShard.getShard())
                .select(campaignJooqReader.getFieldsToRead())
                .select(fields)
                .from(gdiBaseCampaignJoinTablesStatement()
                        .leftJoin(CAMP_DIALOGS).on(CAMP_DIALOGS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMP_OPERATIONS_QUEUE).on(CAMP_OPERATIONS_QUEUE.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMPAIGNS_CPM_YNDX_FRONTPAGE).on(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(USER_CAMPAIGNS_FAVORITE).on(USER_CAMPAIGNS_FAVORITE.CID.eq(CAMPAIGNS.CID)
                                .and(USER_CAMPAIGNS_FAVORITE.UID.eq(uidClientIdShard.getUid())))
                        .leftJoin(CAMPAIGNS_INTERNAL).on(CAMPAIGNS.TYPE.in(INTERNAL_CAMPAIGN_TYPES)
                                .and(CAMPAIGNS_INTERNAL.CID.eq(CAMPAIGNS.CID)))
                        .leftJoin(CAMPAIGN_PERMALINKS).on(CAMPAIGN_PERMALINKS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMPAIGN_PHONES).on(CAMPAIGN_PHONES.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMP_ADDITIONAL_DATA).on(CAMP_ADDITIONAL_DATA.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMP_CALLTRACKING_SETTINGS).on(CAMP_CALLTRACKING_SETTINGS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMPAIGN_PROMOACTIONS).on(CAMPAIGN_PROMOACTIONS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(AGGR_STATUSES_CAMPAIGNS).on(AGGR_STATUSES_CAMPAIGNS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(SUBCAMPAIGNS).on(SUBCAMPAIGNS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(WIDGET_PARTNER_CAMPAIGNS).on(WIDGET_PARTNER_CAMPAIGNS.CID.eq(CAMPAIGNS.CID))
                )
                .where(fetchCampaignsDefaultCondition(uidClientIdShard).and(extraFetchCondition))
                .fetch(this::fromDb);
    }

    private GdiCampaign fromDb(Record record) {
        GdiCampaign campaign = campaignJooqReader.fromDb(record);

        fillBaseFieldsFromRecord(campaign, record);
        campaign.withHasActiveBanners(record.get(HAS_ACTIVE_BANNERS_FIELD) != null)
                .withHasNotArchiveBanners(record.get(HAS_NOT_ACRHIVED_BANNERS_FIELD) != null)
                .withHasBidModifiers(record.get(HAS_HIERARCHICAL_MULTIPLIERS) != null);
        return campaign;
    }

    public List<GdiBaseCampaign> getAllBaseCampaigns(UidClientIdShard uidClientIdShard) {
        Set<Field<Object>> fields = new HashSet<>(Set.of(HAS_BANNERS_FIELD, HAS_METRIKA_COUNTERS_ECOMMERCE));
        if (optimizeCampBannersFlagsProperty.getOrDefault(false)) {
            fields.add(HAS_NEW_MEDIAPLAN_FIELD_OPTIMIZED);
            fields.add(HAS_MEDIAPLAN_BANNERS_FIELD_OPTIMIZED);
        } else {
            fields.add(HAS_NEW_MEDIAPLAN_FIELD_OLD);
            fields.add(HAS_MEDIAPLAN_BANNERS_FIELD_OLD);
        }
        return contextProvider.ppc(uidClientIdShard.getShard())
                .select(baseCampaignJooqReader.getFieldsToRead())
                .select(fields)
                .from(gdiBaseCampaignJoinTablesStatement())
                .where(fetchCampaignsDefaultCondition(uidClientIdShard))
                .fetch(this::baseFromDb);
    }

    Table<Record> gdiBaseCampaignJoinTablesStatement() {
        return CAMPAIGNS
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CAMP_METRIKA_COUNTERS).on(CAMP_METRIKA_COUNTERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CAMP_PROMOCODES).on(CAMP_PROMOCODES.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CAMPAIGNS_MOBILE_CONTENT).on(CAMPAIGNS_MOBILE_CONTENT.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS_CPM_PRICE.CID.eq(CAMPAIGNS.CID))
                .leftJoin(MOBILE_APPS).on(MOBILE_APPS.MOBILE_APP_ID.eq(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .leftJoin(MOBILE_CONTENT).on(MOBILE_CONTENT.MOBILE_CONTENT_ID
                        .eq(MOBILE_APPS.MOBILE_CONTENT_ID.cast(Long.class)))
                .leftJoin(MOBILE_APP_TRACKERS).on(MOBILE_APP_TRACKERS.MOBILE_APP_ID
                        .eq(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID));
    }

    private Condition fetchCampaignsDefaultCondition(UidClientIdShard uidClientIdShard) {
        return CAMPAIGNS.CLIENT_ID.eq(uidClientIdShard.getClientId().asLong())
                .and(CAMPAIGNS.TYPE.in(CAMPAIGN_TYPES))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));
    }

    private GdiBaseCampaign baseFromDb(Record record) {
        GdiBaseCampaign campaign = baseCampaignJooqReader.fromDb(record);
        fillBaseFieldsFromRecord(campaign, record);
        return campaign;
    }

    private void fillBaseFieldsFromRecord(GdiBaseCampaign campaign, Record record) {
        campaign.withHasBanners(record.get(HAS_BANNERS_FIELD) != null)
                .withHasEcommerce(record.get(HAS_METRIKA_COUNTERS_ECOMMERCE) != null)
                .withHasNewMediaplan(record.get(HAS_NEW_MEDIAPLAN_FIELD) != null)
                .withHasMediaplanBanners(record.get(HAS_MEDIAPLAN_BANNERS_FIELD) != null);
    }

    public static GdBrandSurveyStatus convertBrandSurveyStatus(BrandSurveyStatus brandSurveyStatus) {
        var answer = new GdBrandSurveyStatus()
                .withSurveyStatusDaily(GdSurveyStatus.valueOf(brandSurveyStatus.getSurveyStatusDaily().name()))
                .withReasonIds(brandSurveyStatus.getReasonIds())
                .withBrandSurveyStopReasonsDaily(mapSet(
                        brandSurveyStatus.getBrandSurveyStopReasonsDaily(),
                        brandSurveyStopReason -> GdBrandSurveyStopReason.valueOf(brandSurveyStopReason.name())))
                .withSumSpentByTotalPeriod(brandSurveyStatus.getSumSpentByTotalPeriod())
                .withSumSpentByDay(nvl(brandSurveyStatus.getSumSpentByDay(), BigDecimal.ZERO));

        var basicUplift = brandSurveyStatus.getBasicUplift();
        if (basicUplift != null) {
            answer
                    .withAdRecall(toOutputView(basicUplift.getAdRecall()))
                    .withBrandAwareness(toOutputView(basicUplift.getBrandAwareness()))
                    .withPurchaseIntent(toOutputView(basicUplift.getPurchaseIntent()))
                    .withProductConsideration(toOutputView(basicUplift.getProductConsideration()))
                    .withBrandFavorability(toOutputView(basicUplift.getBrandFavorability()))
                    .withAdMessageRecall(toOutputView(basicUplift.getAdMessageRecall()));
        }

        return answer;
    }

    private static Integer toOutputView(@Nullable Double metric) {

        if (metric == null || Math.round(metric) <= 0) {
            return null;
        }
        return Math.toIntExact(Math.round(metric));
    }
}
