package ru.yandex.direct.grid.core.entity.campaign.repository;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectHavingStep;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalConversion;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.GridStatUtils;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectGridStatData;
import ru.yandex.direct.grid.core.util.stats.goalstat.DirectGoalGridStatData;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.YtStatisticSenecaSasSupport;
import ru.yandex.direct.grid.core.util.yt.YtTestTablesSupport;
import ru.yandex.direct.grid.schema.yt.tables.DirectgridgoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.DirectgridstatBs;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.ytcomponents.service.DirectGridStatDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_ONLY_SENECA_SAS_FOR_GRID_STATTISTIC;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_STAT_CLUSTER_CHOOSER_FOR_GRID_STATTISTIC;
import static ru.yandex.direct.grid.core.util.stats.GridStatNew.getPeriodField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.ytwrapper.YtUtils.periodCondition;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.extractBsMoney;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.longNodeLocalDate;

/**
 * Репозиторий для получения данных кампаний из YT (сейчас это только статистика).
 */
@Repository
@ParametersAreNonnullByDefault
public class GridCampaignYtRepository {
    private static final Logger logger = LoggerFactory.getLogger(GridCampaignYtRepository.class);
    private static final Duration PPC_PROPERTY_CACHE_TTL = Duration.ofMinutes(1);

    private final YtDynamicSupport ytSupport;
    private final PpcProperty<Boolean> useOnlySenecaSasProp;
    private final PpcProperty<Boolean> useStatClusterChooserProp;
    private final YtStatisticSenecaSasSupport ytStatisticSenecaSasSupport;
    private final YtTestTablesSupport ytTestTablesSupport;
    private final DirectGridStatDynContextProvider gridStatDynContextProvider;
    private final CampaignService campaignService;
    private final GridStatNew<DirectgridstatBs, DirectgridgoalsstatBs> gridStat
            = new GridStatNew<>(DirectGridStatData.INSTANCE);
    private final boolean nonProduction;

    @Autowired
    public GridCampaignYtRepository(YtDynamicSupport gridYtSupport,
                                    YtStatisticSenecaSasSupport ytStatisticSenecaSasSupport,
                                    DirectGridStatDynContextProvider gridStatDynContextProvider,
                                    YtTestTablesSupport ytTestTablesSupport,
                                    CampaignService campaignService,
                                    PpcPropertiesSupport ppcPropertiesSupport,
                                    EnvironmentTypeProvider environmentTypeProvider) {
        this.ytSupport = gridYtSupport;
        this.ytStatisticSenecaSasSupport = ytStatisticSenecaSasSupport;
        this.gridStatDynContextProvider = gridStatDynContextProvider;
        this.campaignService = campaignService;
        this.ytTestTablesSupport = ytTestTablesSupport;
        this.useOnlySenecaSasProp = ppcPropertiesSupport.get(
                USE_ONLY_SENECA_SAS_FOR_GRID_STATTISTIC,
                PPC_PROPERTY_CACHE_TTL
        );
        this.useStatClusterChooserProp = ppcPropertiesSupport.get(
                USE_STAT_CLUSTER_CHOOSER_FOR_GRID_STATTISTIC,
                PPC_PROPERTY_CACHE_TTL
        );
        var envType = environmentTypeProvider.get();
        this.nonProduction = envType.isBeta() || envType.isTesting();
    }

    /**
     * Получить суммарные открутки по кампаниям по дням из промежутка.
     * Данные о потраченных деньгах хранятся уже за вычетом НДС, поэтому и возвращаемые значения будут такими же
     *
     * @param campaignIds список идентификаторов кампаний
     * @param startDate   день, начиная с которого нужно получить траты
     * @param endDate     день, заканчивая которым нужно получить траты (включительно)
     * @return хеш, в котором дни соответствуют откруткам по этим кампаниям за этот день
     */
    public Map<LocalDate, BigDecimal> getCampaignCostsByDay(Collection<Long> campaignIds, LocalDate startDate,
                                                            LocalDate endDate) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, Long> masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        Select<?> query = gridStat.constructCostSelect(
                List.of(gridStat.getTableData().effectiveCampaignId(masterIdBySubId),
                        gridStat.getTableData().updateTime().as("UpdateTime")),
                gridStat.getTableData().campaignId().in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId)),
                startDate, endDate);

        long startTime = System.currentTimeMillis();
        Map<LocalDate, BigDecimal> result = selectRows(query).getYTreeRows().stream()
                .collect(Collectors.groupingBy(
                        r -> longNodeLocalDate(r, gridStat.getTableData().updateTime().getName()),
                        Collectors.reducing(BigDecimal.ZERO,
                                r -> nvl(extractBsMoney(r, gridStat.getCost().getName()), BigDecimal.ZERO),
                                BigDecimal::add)
                ));
        logger.info("Get campaign costs by day duration: {} ms", System.currentTimeMillis() - startTime);
        return result;
    }

    /**
     * Получить статистику по кампаниям за переданный период.
     * Данные о потраченных деньгах хранятся уже за вычетом НДС, поэтому и возвращаемые значения будут такими же.
     * Доход считатается по целям {@code goalIds}
     *
     * @param campaignIds список идентификаторов кампаний
     * @param startDate   начало периода
     * @param endDate     конец периода (включительно)
     * @param goalIds     список целей
     */
    public Map<Long, GdiCampaignStats> getCampaignStats(Collection<Long> campaignIds, LocalDate startDate,
                                                        LocalDate endDate, Set<Long> goalIds) {
        return getCampaignStats(campaignIds, startDate, endDate, goalIds, null);
    }

    /**
     * Получить статистику по кампаниям за переданный период.
     * Данные о потраченных деньгах хранятся уже за вычетом НДС, поэтому и возвращаемые значения будут такими же.
     * Доход считатается по целям {@code goalIdsForRevenue} если не null, иначе по {@code goalIds}
     *
     * @param campaignIds       список идентификаторов кампаний
     * @param startDate         начало периода
     * @param endDate           конец периода (включительно)
     * @param goalIds           список целей
     * @param goalIdsForRevenue список целей, по которым считается доход
     */
    public Map<Long, GdiCampaignStats> getCampaignStats(
            Collection<Long> campaignIds, LocalDate startDate, LocalDate endDate,
            Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, Long> masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        Select<?> query = gridStat.constructStatSelect(
                Collections.singletonList(gridStat.getTableData().effectiveCampaignId(masterIdBySubId)),
                gridStat.getTableData().campaignId().in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId)),
                startDate, endDate, goalIds, goalIdsForRevenue);
        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                        n -> new GdiCampaignStats()
                                .withStat(GridStatNew.addZeros(gridStat.extractStatsEntryWithGoalsRevenue(
                                        n, goalIds, goalIdsForRevenue)))
                                .withGoalStats(GridStatNew.addZeros(gridStat.extractGoalStatEntries(n, goalIds))))
                .toMap();
    }

    /**
     * Получить статистику по кампаниям за переданный период с группировкой по дням, неделям или месяцам
     * Данные о потраченных деньгах хранятся уже за вычетом НДС, поэтому и возвращаемые значения будут такими же.
     * Доход считатается по целям {@code goalIdsForRevenue} если не null, иначе по {@code goalIds}
     */
    public Map<Long, Map<LocalDate, GdiCampaignStats>> getCampaignStatsGroupByDate(
            Collection<Long> campaignIds,
            LocalDate startDate,
            LocalDate endDate,
            Set<Long> goalIds,
            @Nullable Set<Long> goalIdsForRevenue,
            ReportOptionGroupByDate groupByDate
    ) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        Field<Long> updateTimeField = getPeriodField(gridStat.getTableData().updateTime(), nvl(groupByDate,
                ReportOptionGroupByDate.NONE)).as("UpdateTime");

        Map<Long, Long> masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        Select<?> query = gridStat.constructStatSelect(
                List.of(gridStat.getTableData().effectiveCampaignId(masterIdBySubId), updateTimeField),
                gridStat.getTableData().campaignId().in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId)),
                startDate, endDate, goalIds, goalIdsForRevenue);

        Map<Long, List<YTreeMapNode>> rawStatByCampaignId = StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(n -> n.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                        identity())
                .grouping();
        return EntryStream.of(rawStatByCampaignId)
                .mapValues(campaignRawStat ->
                        convertCampaignStat(campaignRawStat, goalIds, goalIdsForRevenue, updateTimeField))
                .toMap();
    }

    private Map<LocalDate, GdiCampaignStats> convertCampaignStat(List<YTreeMapNode> campaignRawStat,
                                                                 Set<Long> goalIds,
                                                                 @Nullable Set<Long> goalIdsForRevenue,
                                                                 Field<Long> updateTimeField) {
        return StreamEx.of(campaignRawStat)
                .mapToEntry(rawCampaignDailyStat -> longNodeLocalDate(rawCampaignDailyStat, updateTimeField.getName()),
                        identity())
                .mapValues(rawCampaignDailyStat -> new GdiCampaignStats()
                        .withStat(GridStatNew.addZeros(gridStat.extractStatsEntryWithGoalsRevenue(
                                rawCampaignDailyStat, goalIds, goalIdsForRevenue)))
                        .withGoalStats(GridStatNew.addZeros(gridStat.extractGoalStatEntries(rawCampaignDailyStat,
                                goalIds))))
                .toMap();
    }

    /**
     * Получить информацию по конверсиям для кампаний за заданный период времени для всех целей с кампаний.
     */
    public Map<Long, GdiEntityConversion> getCampaignsConversionCount(Collection<Long> campaignIds,
                                                                      LocalDate startDate,
                                                                      LocalDate endDate) {
        return getCampaignsConversionCount(campaignIds, startDate, endDate, null, false, false);
    }


    /**
     * Получить информацию по конверсиям для кампаний за заданный период времени для доступных целей с кампаний.
     * Данный метод не актуален, хотим убедиться, что скрывать статистику по конверсиям для недоступных целей не нужно
     */
    @Deprecated
    public Map<Long, GdiEntityConversion> getCampaignsConversionCountWithStatsFiltration(
            Collection<Long> campaignIds,
            LocalDate startDate,
            LocalDate endDate,
            Set<Long> availableGoalIds) {
        return getCampaignsConversionCount(campaignIds, startDate, endDate, availableGoalIds, false, true);
    }

    /**
     * Получить информацию по конверсиям для кампаний за заданный период времени.
     * Конверсии считаются по целям, которые ассоциированны с кампанией, для этого используем поле CampaignGoalType:
     * <ul>
     *  <li>1 - цель используется в блоке стратегии</li>
     *  <li>2 - цель используется в блоке КЦ</li>
     * </ul>
     * Логика вычисления числа конверсии:
     * если за день есть данные по целям из стратегии, то берем их, иначе - данные по целям из КЦ
     *
     * @param campaignIds                          список id кампаний
     * @param startDate                            начало периода
     * @param endDate                              конец периода
     * @param availableGoalIds                     доступные клиенту цели
     * @param getRevenueOnlyByAvailableGoals       если true, то доход считать
     *                                             только с доступных целей {@code availableGoalIds}
     * @param getCampaignStatsOnlyByAvailableGoals если true, то всю информацию по конверсиям брать
     *                                             только с доступных целей {@code availableGoalIds}
     * @return мапа, где ключ - id кампании, а значение - данные с кол-вом конверсий и доходом
     */
    public Map<Long, GdiEntityConversion> getCampaignsConversionCount(Collection<Long> campaignIds,
                                                                      LocalDate startDate,
                                                                      LocalDate endDate,
                                                                      @Nullable Set<Long> availableGoalIds,
                                                                      boolean getRevenueOnlyByAvailableGoals,
                                                                      boolean getCampaignStatsOnlyByAvailableGoals) {
        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        var query = gridStat.constructConversionsSelect(
                campaignIds, masterIdBySubId, startDate, endDate, availableGoalIds, null,
                getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals, false);

        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                        gridStat::extractEntityConversionWithRevenue)
                .nonNullValues()
                .toMap(gridStat::mergeEntityConversion);
    }

    /**
     * Получить конверсии по кампании с разбивкой по дням.
     * Для подсчета конверсий будут использоваться только те цели,
     * которые заданны как ключевые или цель стратегии в кампании.
     *
     * @param campaignIds
     * @param startDate
     * @param endDate
     * @return
     */
    public Map<Long, Map<LocalDate, GdiEntityConversion>> getCampaignsConversions(
            Collection<Long> campaignIds,
            LocalDate startDate,
            LocalDate endDate,
            @Nullable Set<Long> availableGoalIds,
            ReportOptionGroupByDate groupByDate,
            boolean getRevenueOnlyByAvailableGoals,
            boolean getCampaignStatsOnlyByAvailableGoals,
            boolean isUacWithOnlyInstallEnabled) {
        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        var query = gridStat.constructConversionsSelect(
                campaignIds, masterIdBySubId, startDate, endDate, availableGoalIds, groupByDate,
                getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals, isUacWithOnlyInstallEnabled);

        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(
                        node -> node.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                        gridStat::extractEntityConversionWithRevenue
                ).nonNullValues()
                .mapValues(conversion -> Map.of(conversion.getDate(), conversion))
                .toMap(this::mergeCampaignDailyConversion);
    }

    private Map<LocalDate, GdiEntityConversion> mergeCampaignDailyConversion(Map<LocalDate, GdiEntityConversion> first,
                                                                             Map<LocalDate, GdiEntityConversion> second) {
        var union = new HashMap<>(first);
        union.putAll(second);
        return union;
    }

    /**
     * Частичное получение данных о статистики целей кампаний. Не учитывает данные о статистике со всей кампании
     */
    public Map<Long, List<GdiGoalConversion>> getCampaignGoalsConversionsCount(Collection<Long> campaignIds,
                                                                               LocalDate startDate,
                                                                               LocalDate endDate,
                                                                               Collection<Long> goalIds) {
        DirectGoalGridStatData statTable = new DirectGoalGridStatData();

        Map<Long, Long> masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        Field<Long> campaignId = statTable.effectiveCampaignId(masterIdBySubId);
        Field<Long> goalId = statTable.goalId().as(statTable.goalId().getName());
        Field<BigDecimal> goalsNum = YtDSL.ytIfNull(DSL.sum(statTable.goalsNum()), BigDecimal.ZERO)
                .as(statTable.goalsNum().getName());
        Field<BigDecimal> income = YtDSL.ytIfNull(DSL.sum(statTable.priceCur().divide(DECIMAL_MULT)), BigDecimal.ZERO)
                .as(statTable.priceCur().getName());

        var query = YtDSL.ytContext()
                .select(campaignId, goalId, goalsNum, income)
                .from(statTable.table())
                .where(statTable.campaignId()
                        .in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId))
                        .and(statTable.goalId().in(goalIds))
                        .and(periodCondition(statTable.updateTime(), startDate, endDate)))
                .groupBy(List.of(campaignId, goalId));

        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(statTable.effectiveCampaignId().getName()).longValue(),
                        gridStat::extractGoalConversionWithRevenue)
                .nonNullValues()
                .grouping();
    }

    /**
     * Частичное получение данных о статистики по определенной цели и кампании. Не учитывает данные о статистике со
     * всей кампании.
     */
    public Map<Long, List<GdiGoalConversion>> getCampaignGoalsConversionsCount(
            Map<Long, LocalDateRange> localDateRangeByCampaignId,
            Map<Long, Set<Long>> goalIdsByCampaignId) {
        if (localDateRangeByCampaignId.isEmpty()) {
            return emptyMap();
        }
        DirectGoalGridStatData statTable = new DirectGoalGridStatData();

        Map<Long, Long> masterIdBySubId =
                campaignService.getSubCampaignIdsWithMasterIds(localDateRangeByCampaignId.keySet());
        Field<Long> campaignId = statTable.effectiveCampaignId(masterIdBySubId);
        Field<Long> goalId = statTable.goalId().as(statTable.goalId().getName());
        Field<BigDecimal> goalsNum = YtDSL.ytIfNull(DSL.sum(statTable.goalsNum()), BigDecimal.ZERO)
                .as(statTable.goalsNum().getName());
        Field<BigDecimal> revenue = YtDSL.ytIfNull(DSL.sum(statTable.priceCur().divide(DECIMAL_MULT)), BigDecimal.ZERO)
                .as(statTable.priceCur().getName());

        var selectStep = YtDSL.ytContext()
                .select(campaignId, goalId, goalsNum, revenue)
                .from(statTable.table());

        Condition whereCondition = EntryStream.of(localDateRangeByCampaignId)
                .append(EntryStream.of(masterIdBySubId).mapValues(localDateRangeByCampaignId::get))
                .mapKeyValue((cid, range) -> statTable.campaignId().eq(cid)
                        .and(periodCondition(statTable.updateTime(), range.getFromInclusive(),
                                range.getToInclusive()))
                        .and(goalIdsByCampaignId.get(cid) != null ?
                                statTable.goalId().in(goalIdsByCampaignId.get(cid)) : DSL.noCondition()))
                .foldLeft(DSL.noCondition(), Condition::or);

        var query = selectStep
                .where(whereCondition)
                .groupBy(List.of(campaignId, goalId));

        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(statTable.effectiveCampaignId().getName()).longValue(),
                        gridStat::extractGoalConversionWithRevenue)
                .nonNullValues()
                .grouping();
    }

    public Map<Long, GdiEntityStats> getCampaignEntityStats(Collection<Long> campaignIds, LocalDate startDate,
                                                            LocalDate endDate) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, LocalDateRange> localDateRangeByCampaignId = listToMap(campaignIds, identity(),
                campaignId -> new LocalDateRange()
                        .withFromInclusive(startDate)
                        .withToInclusive(endDate));

        return getCampaignEntityStats(localDateRangeByCampaignId);
    }

    public Map<Long, GdiEntityStats> getCampaignEntityStats(Map<Long, LocalDateRange> localDateRangeByCampaignId) {
        if (localDateRangeByCampaignId.isEmpty()) {
            return emptyMap();
        }
        Map<Long, Long> masterIdBySubId =
                campaignService.getSubCampaignIdsWithMasterIds(localDateRangeByCampaignId.keySet());
        List<Field<BigDecimal>> selectFields = gridStat.getStatSelectFields();
        SelectJoinStep<Record> step = YtDSL.ytContext()
                .select(gridStat.getTableData().effectiveCampaignId(masterIdBySubId))
                .select(selectFields)
                .from(gridStat.getTableData().table());

        Condition whereCondition = getWhereConditionForStatSelect(localDateRangeByCampaignId, masterIdBySubId);

        SelectHavingStep<Record> query = step.where(whereCondition)
                .groupBy(Collections.singletonList(gridStat.getTableData().effectiveCampaignId()));

        return listToMap(selectRows(query).getYTreeRows(),
                r -> r.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                gridStat::extractStatsEntry);
    }

    private Condition getWhereConditionForStatSelect(Map<Long, LocalDateRange> localDateRangeByCampaignId,
                                                     Map<Long, Long> masterIdBySubId) {
        Comparator<LocalDateRange> localDateRangeComparator = getLocalDateRangeComparator();

        return EntryStream.of(localDateRangeByCampaignId)
                .append(EntryStream.of(masterIdBySubId).mapValues(localDateRangeByCampaignId::get))
                .invert()
                .sorted(Map.Entry.comparingByKey(localDateRangeComparator))
                .collapseKeys()
                .mapKeyValue((range, campaignIds) ->
                        periodCondition(gridStat.getTableData().updateTime(), range.getFromInclusive(),
                                range.getToInclusive())
                                .and(gridStat.getTableData().campaignId()
                                        .in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId))))
                .foldLeft(DSL.noCondition(), Condition::or);
    }

    private static Comparator<LocalDateRange> getLocalDateRangeComparator() {
        return Comparator.comparing(LocalDateRange::getFromInclusive)
                .thenComparing(LocalDateRange::getToInclusive);
    }

    public Map<Long, GdiCampaignStats> getCampaignStatsWithFilteringGoals(Collection<Long> campaignIds,
                                                                          LocalDate startDate,
                                                                          LocalDate endDate, Set<Long> goalIds,
                                                                          Set<Long> availableGoalIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds);
        var query = gridStat.constructStatSelect(
                Collections.singletonList(gridStat.getTableData().effectiveCampaignId(masterIdBySubId)),
                gridStat.getTableData().campaignId().in(GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId)),
                startDate, endDate, availableGoalIds, null);

        return StreamEx.of(selectRows(query).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(gridStat.getTableData().effectiveCampaignId().getName()).longValue(),
                        n -> {
                            List<GdiGoalStats> fullGoalStats = gridStat.extractGoalStatEntries(n, availableGoalIds);
                            List<GdiGoalStats> filteredGoalStats = filterList(fullGoalStats,
                                    x -> goalIds.contains(x.getGoalId()));
                            GdiEntityStats stat = GridStatNew.addZeros(gridStat.extractStatsEntry(n));
                            BigDecimal goals = BigDecimal.valueOf(aggregateGoalStat(fullGoalStats));
                            stat.setGoals(goals);
                            return new GdiCampaignStats()
                                    .withStat(stat)
                                    .withGoalStats(GridStatNew.addZeros(filteredGoalStats));
                        })
                .toMap();
    }

    private static Long aggregateGoalStat(List<GdiGoalStats> goalStats) {
        return StreamEx.of(goalStats)
                .map(GdiGoalStats::getGoals)
                .foldLeft(0L, (current, next) -> current += next);
    }

    // todo https://st.yandex-team.ru/DIRECT-138089

    /**
     * Если по проперте включено использование clusterChooser'а на основе таблиц статистики, то используем его
     * Иначе, если по проперте включено чтение статистики только из кластера seneca-sas, то читаем из него
     * Иначе используем clusterChooser на основе mysql-sync
     */
    private UnversionedRowset selectRows(Select query) {
        if (nonProduction && SecurityContextHolder.getContext().getAuthentication() != null
                && FeatureHelper.feature(FeatureName.USE_TEST_YT_STAT).enabled()) {
            return ytTestTablesSupport.selectStatistics(query);
        }
        if (useStatClusterChooserProp.getOrDefault(false)) {
            return gridStatDynContextProvider.getContext().executeSelect(query);
        }
        if (useOnlySenecaSasProp.getOrDefault(false)) {
            return ytStatisticSenecaSasSupport.selectStatistics(query);
        } else {
            return ytSupport.selectRows(query);
        }
    }
}
