package ru.yandex.direct.grid.core.entity.campaign.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.utils.BlockedUserUtil;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.client.GdiClientInfo;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiCampaignActionsHolder;
import ru.yandex.direct.grid.model.campaign.GdiWalletAction;
import ru.yandex.direct.grid.model.campaign.GdiWalletActionsHolder;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.RbacCampPerms;
import ru.yandex.direct.rbac.model.SubclientGrants;
import ru.yandex.direct.validation.Predicates;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithPricePackagePermissionUtils.canResetFlightStatusApprove;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAnyTeamLeader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isLimitedSupport;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isMediaPlanner;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isPlacer;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperReader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSupport;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для подсчета доступа оператора к кампаниям в основном на основании информации об операторе и его правах
 */
@Service
@ParametersAreNonnullByDefault
public class GridCampaignAccessService {
    private static final Logger logger = LoggerFactory.getLogger(GridCampaignAccessService.class);
    /**
     * Хеш, в котором каждому из расчитываемых здесь действий оператора ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили действие в набор доступных
     */
    private static final Map<GdiCampaignAction, Predicate<CampActionsNode>> ACTION_PREDICATES =
            ImmutableMap.<GdiCampaignAction, Predicate<CampActionsNode>>builder()
                    .put(GdiCampaignAction.STOP_CAMP, c -> canEditBase(c) && !isMediaPlanner(c.operator))
                    .put(GdiCampaignAction.SHOW_CAMP_STAT, c -> canEditBase(c) || allowLookup(c))
                    .put(GdiCampaignAction.SHOW_CAMP_SETTINGS,
                            c -> (canEditBase(c) && isMediaPlanner(c.operator)) || allowSeeSettings(c))
                    .put(GdiCampaignAction.EDIT_CAMP, GridCampaignAccessService::canEditCampaign)
                    .put(GdiCampaignAction.EDIT_WEEKLY_BUDGET, GridCampaignAccessService::canEditCampaign)
                    .put(GdiCampaignAction.DISABLE_WEEKLY_BUDGET, GridCampaignAccessService::canEditCampaign)
                    .put(GdiCampaignAction.RESUME_CAMP, c -> canEditBase(c) && !isMediaPlanner(c.operator))
                    .put(GdiCampaignAction.COPY_CAMP_CLIENT, GridCampaignAccessService::canCopyCampClient)
                    .put(GdiCampaignAction.LOOKUP, c -> canEditBase(c) || allowLookup(c))
                    .put(GdiCampaignAction.PAY, GridCampaignAccessService::allowPay)
                    .put(GdiCampaignAction.ARCHIVE_CAMP, GridCampaignAccessService::allowArchiveOrUnarchive)
                    .put(GdiCampaignAction.UNARCHIVE_CAMP, GridCampaignAccessService::allowArchiveOrUnarchive)
                    .put(GdiCampaignAction.DELETE_CAMP, GridCampaignAccessService::canDeleteCamp)
                    .put(GdiCampaignAction.ACCEPT_SERVICING, c -> c.isWaitServicing)
                    .put(GdiCampaignAction.OFFER_SERVICING, GridCampaignAccessService::allowOfferServicing)
                    .put(GdiCampaignAction.SERVICED, c -> c.hasManager)
                    .put(GdiCampaignAction.AGENCY_SERVICED, c -> c.hasAgency)
                    .put(GdiCampaignAction.ALLOW_TRANSFER_MONEY_SUBCLIENT,
                            c -> c.hasAgency && c.perms.canTransferMoney() && !c.clientIsBlockedForEdit)
                    .put(GdiCampaignAction.OTHER_MANAGER_SERVICED, c -> c.hasManager && !c.hasAgency)
                    .put(GdiCampaignAction.REMODERATE_CAMP, c ->
                            GridCampaignAccessServiceUtils.getCanRemoderateCampaign(c.operator) && !c.clientIsBlockedForEdit)
                    .put(GdiCampaignAction.SHOW_BS_LINK, Predicates.ignore())
                    .put(GdiCampaignAction.EXPORT_IN_EXCEL, c -> c.perms.canExportInExcel())
                    .put(GdiCampaignAction.CAN_BE_AUTODELETED, Predicates.ignore())
                    .put(GdiCampaignAction.RESET_FLIGHT_STATUS_APPROVE,
                            c -> !c.clientIsBlockedForEdit && canResetFlightStatusApprove(c.operator))
                    .put(GdiCampaignAction.VIEW_OFFLINE_REPORTS, GridCampaignAccessService::canViewOfflineReports)
                    .put(GdiCampaignAction.MANAGE_VCARDS, c -> (!isMediaPlanner(c.operator) && c.perms.canWrite())
                            || hasOneOfRoles(c.operator, SUPERREADER, LIMITED_SUPPORT))
                    .put(GdiCampaignAction.BAN_PAY, c -> canBanOrUnbanPay(c.operator))
                    .put(GdiCampaignAction.UNBAN_PAY, c -> canBanOrUnbanPay(c.operator)
                            && !c.unbindAgencies.contains(ClientId.fromLong(c.campaign.getAgencyId())))
                    .put(GdiCampaignAction.SERVICING_STOPPED, c ->
                            c.unbindAgencies.contains(ClientId.fromLong(c.campaign.getAgencyId())))
                    .put(GdiCampaignAction.MANAGE_PROMO_EXTENSION, c -> canEditBase(c) && !isMediaPlanner(c.operator))
                    .build();

    private static final Map<GdiWalletAction, Predicate<WalletActionsNode>> WALLET_ACTION_PREDICATES =
            ImmutableMap.<GdiWalletAction, Predicate<WalletActionsNode>>builder()
                    .put(GdiWalletAction.EDIT, GridCampaignAccessService::canEditWallet)
                    .put(GdiWalletAction.PAY, GridCampaignAccessService::allowPayWallet)
                    .build();

    @ParametersAreNonnullByDefault
    private static class CampActionsNode {
        private final User operator;
        private final GdiClientInfo clientInfo;
        private final GdiBaseCampaign campaign;
        private final RbacCampPerms perms;
        private final RbacCampPerms clientPerms;

        private final boolean isWaitServicing;
        private final boolean hasManager;
        private final boolean hasAgency;
        private final Set<ClientId> unbindAgencies;
        private final boolean clientIsBlockedForEdit;

        private CampActionsNode(User operator, GdiClientInfo clientInfo,
                                GdiBaseCampaign campaign, User subjectUser, RbacCampPerms perms,
                                RbacCampPerms clientPerms, boolean isWaitServicing, Set<ClientId> unbindAgencies) {
            this.operator = operator;
            this.clientInfo = clientInfo;
            this.campaign = campaign;
            this.perms = perms;
            this.clientPerms = clientPerms;
            this.isWaitServicing = isWaitServicing;
            this.unbindAgencies = unbindAgencies;

            hasManager = isValidId(campaign.getManagerUserId());
            hasAgency = isValidId(campaign.getAgencyUserId());
            clientIsBlockedForEdit = BlockedUserUtil.checkClientIsBlockedForEdit(subjectUser, operator);
        }

        private GdiCampaignActionsHolder toActionsHolder() {
            Set<GdiCampaignAction> campaignActions = getCampaignActions();
            return new GdiCampaignActionsHolder()
                    .withCanEdit(campaignActions.contains(GdiCampaignAction.EDIT_CAMP))
                    .withHasAgency(hasAgency)
                    .withHasManager(hasManager)
                    .withActions(campaignActions);
        }

        private Set<GdiCampaignAction> getCampaignActions() {
            return EntryStream.of(ACTION_PREDICATES)
                    .filterValues(p -> p.test(this))
                    .keys()
                    .toSet();
        }
    }

    private static class WalletActionsNode {
        private final GdiBaseCampaign wallet;
        private final User operator;
        private final SubclientGrants subclientGrants;
        private final RbacCampPerms perms;
        private final Boolean agencyRepAndNoPayIsSet;

        private final boolean hasAgency;
        private final boolean clientIsBlockedForEdit;

        private WalletActionsNode(User operator, GdiBaseCampaign wallet, User subjectUser, RbacCampPerms perms,
                                  SubclientGrants subclientGrants, boolean isAgencyWallet,
                                  Boolean isAgencyRepAndNoPayIsSet) {
            this.operator = operator;
            this.wallet = wallet;
            this.perms = perms;
            this.subclientGrants = subclientGrants;

            this.hasAgency = isAgencyWallet;
            this.agencyRepAndNoPayIsSet = isAgencyRepAndNoPayIsSet;
            this.clientIsBlockedForEdit = BlockedUserUtil.checkClientIsBlockedForEdit(subjectUser, operator);
        }

        private GdiWalletActionsHolder toWalletActionsHolder() {
            return new GdiWalletActionsHolder()
                    .withActions(getWalletActions())
                    .withSubclientCanEdit(
                            ifNotNull(subclientGrants, SubclientGrants::canCreateCampaign)) //aka isSuperSubClient
                    .withSubclientAllowTransferMoney(ifNotNull(subclientGrants, SubclientGrants::canToTransferMoney));
        }

        private Set<GdiWalletAction> getWalletActions() {
            return EntryStream.of(WALLET_ACTION_PREDICATES)
                    .filterValues(p -> p.test(this))
                    .keys()
                    .toSet();
        }
    }

    private final RbacService rbacService;
    private final UserService userService;
    private final AgencyClientRelationService agencyClientRelationService;

    @Autowired
    public GridCampaignAccessService(RbacService rbacService, UserService userService,
                                     AgencyClientRelationService agencyClientRelationService) {
        this.rbacService = rbacService;
        this.userService = userService;
        this.agencyClientRelationService = agencyClientRelationService;
    }

    private static boolean canEditBase(CampActionsNode c) {
        if (c.clientIsBlockedForEdit) {
            return false;
        }

        return c.perms.canWrite()
                || isSuper(c.operator)
                || (isPlacer(c.operator) && c.hasManager)
                || (c.clientPerms.canWrite() && isLimitedSupport(c.operator));
    }

    private static boolean canEditCampaign(CampActionsNode c) {
        return canEditBase(c) && !isMediaPlanner(c.operator);
    }

    private static boolean canCopyCampClient(CampActionsNode c) {
        return canEditBase(c) && (isSuper(c.operator) || isClient(c.operator) || isLimitedSupport(c.operator));
    }

    private static boolean canEditWallet(WalletActionsNode w) {
        //data3/desktop.blocks/b-wallet-link/b-wallet-link.bemtree.js
        return !isReadOnlyWalletAccess(w) && w.perms.canWrite();
    }

    private static boolean isReadOnlyWalletAccess(WalletActionsNode w) {
        if (w.clientIsBlockedForEdit) {
            return true;
        }

        return isPlacer(w.operator) || isMediaPlanner(w.operator) || isSuperReader(w.operator);
    }

    private static boolean allowPay(CampActionsNode c) {
        if (c.clientIsBlockedForEdit || isMediaPlanner(c.operator)) {
            return false;
        }
        if (isSuper(c.operator) || c.perms.canTransferMoney()) {
            return true;
        }
        return c.clientPerms.canTransferMoney() && isLimitedSupport(c.operator);
    }

    private static boolean allowLookup(CampActionsNode c) {
        return isNotSuperManagerForServiced(c)
                && (c.perms.canRead() || (c.clientPerms.canRead() && isLimitedSupport(c.operator)));
    }

    private static boolean allowSeeSettings(CampActionsNode c) {
        return isSuperReader(c.operator) ||
                ((c.hasAgency && c.perms.canRead() && c.operator.getRole() == CLIENT || c.campaign.getArchived())
                        && isNotSuperManagerForServiced(c));
    }

    private static boolean allowOfferServicing(CampActionsNode c) {
        return !c.isWaitServicing && !c.hasAgency && !isSupport(c.operator) && !isPlacer(c.operator);
    }

    private static boolean allowPayWallet(WalletActionsNode w) {
        if (w.clientIsBlockedForEdit) {
            return false;
        }

        //логика из старого фронта data3/desktop.blocks/b-wallet-rest/b-wallet-rest.bemtree.js
        boolean isRelevantRole = !isPlacer(w.operator) && !isSuperReader(w.operator) && !isMediaPlanner(w.operator);

        if (w.hasAgency) {
            //проверяем права субклиента (часть логики про вычисление allow_pay из Wallet.pm)
            boolean allowPay = (w.operator.getRole() == CLIENT) ? w.subclientGrants.canCreateCampaign()
                    || w.subclientGrants.canToTransferMoney() : true;

            return isRelevantRole && !w.agencyRepAndNoPayIsSet && allowPay;
        } else {
            return isRelevantRole;
        }
    }

    private static boolean allowArchiveOrUnarchive(CampActionsNode c) {
        if (!c.perms.canRead() || c.clientIsBlockedForEdit) {
            return false;
        }

        if (hasOneOfRoles(c.operator, RbacRole.MEDIA, RbacRole.SUPERREADER)) {
            return false;
        }

        if (isCampaignInSpecialArchive(c)) {
            return false;
        }

        // ??? andy-ilyin@ 2019-07-24 непонятно, почему вообще для всех ролей не сделали canWrite
        if (c.operator.getRole() == RbacRole.INTERNAL_AD_MANAGER) {
            return c.perms.canWrite();
        }

        return true;
    }

    private static boolean canDeleteCamp(CampActionsNode c) {
        if (c.clientIsBlockedForEdit) {
            return false;
        }
        return c.perms.canDrop() || (c.clientPerms.canDrop() && isLimitedSupport(c.operator));
    }

    private static boolean isNotSuperManagerForServiced(CampActionsNode c) {
        // суперменеджер умеет смотреть только самоходные кампании
        return !(isSuperManager(c.operator) && (c.hasManager || c.hasAgency));
    }

    private static boolean isCampaignInSpecialArchive(CampActionsNode c) {
        // TODO(bzzzz): еще тут была такая проверка, но кажется она лишняя
        // # кампания могла быть архивна на момент перехода клиента к мультивалютности
        //        # и осталась в у. е., то разархивировать её не даём
        //        my $client_id = $camp->{ClientID};
        //        my $uid = $camp->{uid};
        //        $client_currencies_cache->{$client_id} ||= get_one_field_sql(PPC(ClientID => $client_id),
        //            ['SELECT work_currency FROM clients', WHERE => {ClientID => $client_id}]) || "YND_FIXED";
        //        if ($client_currencies_cache->{$client_id} && $client_currencies_cache->{$client_id} ne 'YND_FIXED') {
        //            return 1;
        //        }
        return isCurrencyConverted(c.campaign);
    }

    public static boolean isCurrencyConverted(GdiBaseCampaign campaign) {
        return campaign.getArchived() && campaign.getCurrencyCode() == CurrencyCode.YND_FIXED && campaign
                .getCurrencyConverted();
    }

    /**
     * Валюта {@code YND_FIXED} кампании считается архивной,
     * если у клиента другая валюта после перехода к мультивалютности
     * <p>
     * Например, это может быть кампания, заархивированная до перехода к мультивалютности
     */
    public static boolean isCurrencyArchived(CampActionsNode c) {
        return c.clientInfo.getWorkCurrency() != CurrencyCode.YND_FIXED
                && c.campaign.getCurrencyCode() == CurrencyCode.YND_FIXED;
    }

    private static boolean canViewOfflineReports(CampActionsNode c) {
        return hasOneOfRoles(c.operator, SUPER, SUPERREADER, AGENCY, MEDIA, SUPPORT, LIMITED_SUPPORT)
                && !isCurrencyArchived(c);
    }

    private static boolean canBanOrUnbanPay(User user) {
        return isSuper(user) || isSupport(user) || isAnyTeamLeader(user);
    }

    /**
     * Посчитать возможности доступа оператора к переданным кампаниям.
     * Это аналог RBACDirect::rbac_get_campaigns_actions из перлового директа
     *
     * @param operator    описание оператора
     * @param campaigns   список объектов кампаний
     * @param subjectUser описание пользователя
     */
    public <T extends GdiBaseCampaign> Map<Long, GdiCampaignActionsHolder> getCampaignsActions(
            User operator,
            GdiClientInfo clientInfo,
            Collection<T> campaigns,
            User subjectUser) {
        List<Long> campaignIds = mapList(campaigns, GdiBaseCampaign::getId);
        Map<Long, RbacCampPerms> campaignsRights = rbacService.getCampaignsRights(operator.getUid(), campaignIds);
        Map<Long, RbacCampPerms> campaignsClientsRights =
                rbacService.getCampaignsRights(clientInfo.getChiefUserId(), campaignIds);
        Map<Long, Boolean> campaignWaitForServicingUid = rbacService.getCampaignsWaitForServicing(campaignIds);
        Set<ClientId> unbindAgencies = agencyClientRelationService
                .getUnbindedAgencies(ClientId.fromLong(clientInfo.getId()));

        return campaigns.stream()
                .collect(Collectors.toMap(
                        GdiBaseCampaign::getId,
                        c -> new CampActionsNode(operator, clientInfo, c, subjectUser,
                                campaignsRights.getOrDefault(c.getId(), RbacCampPerms.EMPTY),
                                campaignsClientsRights.getOrDefault(c.getId(), RbacCampPerms.EMPTY),
                                campaignWaitForServicingUid.get(c.getId()),
                                unbindAgencies)
                                .toActionsHolder()
                        )
                );
    }

    /**
     * Возващает доступные действия оператора по отношению к кошелькам
     * Создано по мотивам перлового Wallet::get_wallets_by_uids
     * и фронтового data3/desktop.blocks/b-wallet-rest/b-wallet-rest.bemtree.js
     *
     * @param operator оператор
     * @param wallets  общие счета клиента.
     */
    public Map<Long, GdiWalletActionsHolder> getWalletsActions(User operator,
                                                               Collection<GdiBaseCampaign> wallets,
                                                               User subjectUser) {

        if (wallets.isEmpty()) {
            return Collections.emptyMap();
        }

        List<Long> walletIds = mapList(wallets, GdiBaseCampaign::getId);
        if (wallets.size() > 1) {
            logger.warn("Client has more than one wallet. walletIds = {}", walletIds.toArray());
        }
        Map<Long, RbacCampPerms> campaignsRights = rbacService.getCampaignsRights(operator.getUid(), walletIds);


        List<GdiBaseCampaign> personalWallets = filterList(wallets, w -> w.getAgencyId() == 0);
        List<GdiBaseCampaign> agencyWallets = filterList(wallets, this::isAgencyWallet);

        List<WalletActionsNode> walletActionsNodeList = new ArrayList<>();

        if (!personalWallets.isEmpty()) {
            walletActionsNodeList.addAll(
                    getActionsNodeForPersonalWallets(operator, personalWallets, campaignsRights, subjectUser));
        }
        // если среди кошельков есть агентские, то дополнительно проверяем права на них, выданные агентством субклиенту.
        if (!agencyWallets.isEmpty()) {
            walletActionsNodeList.addAll(
                    getActionsNodeForAgencyWallets(operator, agencyWallets, campaignsRights, subjectUser));
        }
        return listToMap(walletActionsNodeList, node -> node.wallet.getId(), WalletActionsNode::toWalletActionsHolder);
    }

    private List<WalletActionsNode> getActionsNodeForPersonalWallets(
            User operator,
            Collection<GdiBaseCampaign> wallets,
            Map<Long, RbacCampPerms> campaignsRights,
            User subjectUser
    ) {
        return mapList(wallets, w -> new WalletActionsNode(operator, w, subjectUser,
                campaignsRights.getOrDefault(w.getId(), RbacCampPerms.EMPTY),
                null, false, isAgencyRepAndNoPayIsSet(operator)));
    }

    private List<WalletActionsNode> getActionsNodeForAgencyWallets(
            User operator,
            Collection<GdiBaseCampaign> wallets,
            Map<Long, RbacCampPerms> campaignsRights,
            User subjectUser
    ) {
        //получаем главных представителей для agencyUid кошелька
        Set<Long> agencyUids = listToSet(wallets, GdiBaseCampaign::getAgencyUserId);
        Map<Long, Long> agencyUidToAgencyChiefUid = listToMap(agencyUids, identity(), rbacService::getChief);
        Set<Long> chiefUids = ImmutableSet.copyOf(agencyUidToAgencyChiefUid.values());

        Map<Long, SubclientGrants> agencyUidToSubclientGrants = agencyUidToAgencyChiefUid.isEmpty()
                ? Collections.emptyMap()
                : listToMap(rbacService.getSubclientGrants(chiefUids, operator.getClientId()),
                SubclientGrants::getAgencyUid);

        return mapList(wallets, w -> new WalletActionsNode(operator, w, subjectUser,
                campaignsRights.getOrDefault(w.getId(), RbacCampPerms.EMPTY),
                agencyUidToSubclientGrants.getOrDefault(agencyUidToAgencyChiefUid.get(w.getAgencyUserId()),
                        SubclientGrants.buildWithPermissions(w.getAgencyUserId(), operator.getClientId(),
                                Collections.emptySet())),
                true, isAgencyRepAndNoPayIsSet(operator)));
    }

    private boolean isAgencyWallet(GdiBaseCampaign wallet) {
        return wallet.getAgencyId() != 0 && wallet.getAgencyUserId() != null;
    }

    private boolean isAgencyRepAndNoPayIsSet(User operator) {
        //Todo(pashkus) хранить свойства из user_agency в User - DIRECT-84072
        //проверяем права ограниченных представителей на выставление счета
        return operator.getRole() == RbacRole.AGENCY && operator.getRepType() == RbacRepType.LIMITED && userService
                .getUserAgencyIsNoPay(operator.getUid());
    }
}
