package ru.yandex.direct.grid.core.entity.campaign.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPromotion;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignsPromotionsRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandSafetyService;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstantsService;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.goal.service.CampaignConversionPriceForGoalsWithCategoryCpaSource;
import ru.yandex.direct.core.entity.goal.service.ConversionPriceForecastService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.repository.GridCampaignRepository;
import ru.yandex.direct.grid.core.entity.campaign.repository.GridCampaignYtRepository;
import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalConversion;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiAggregatorGoal;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignGoalsCostPerAction;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCpaSource;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiGoalCostPerAction;
import ru.yandex.direct.grid.core.entity.model.client.GdiClientInfo;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.GridStatUtils;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiCampaignActionsHolder;
import ru.yandex.direct.grid.model.campaign.GdiCampaignMetatype;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdiWalletActionsHolder;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.libs.timetarget.TimeTargetUtils;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.OLD_CAMPAIGNS_PROMOTIONS_TRESHOLD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SPECIAL_GOAL_IDS;
import static ru.yandex.direct.feature.FeatureName.CATEGORY_CPA_SOURCE_FOR_CONVERSION_PRICE_RECOMMENDATION;
import static ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignServiceUtils.calculateCampaignsGoalStat;
import static ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignServiceUtils.fillCampaignStatWithConversionsByDate;
import static ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignServiceUtils.getSourcesWithRedirectEnabled;
import static ru.yandex.direct.grid.core.util.money.GridMoneyUtils.applyNdsSubtruction;
import static ru.yandex.direct.grid.core.util.stats.GridStatUtils.updateStatWithConversions;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;

/**
 * Сервис для работы с кампаниями клиентов. Предназначен исключительно для вызова из сервисов нового интерфейса
 */
@Service
@ParametersAreNonnullByDefault
public class GridCampaignService {
    private final FeatureService featureService;
    private final AdGroupService adGroupService;
    private final CampaignService campaignService;
    private final ClientNdsService clientNdsService;
    private final CampaignRepository campaignRepository;
    private final GridCampaignRepository gridCampaignRepository;
    private final CampaignConstantsService campaignConstantsService;
    private final GridCampaignYtRepository gridCampaignYtRepository;
    private final GridCampaignAccessService gridCampaignAccessService;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final CampaignWithBrandSafetyService campaignWithBrandSafetyService;
    private final MetrikaClient metrikaClient;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignsPromotionsRepository campaignsPromotionsRepository;
    private final ConversionPriceForecastService conversionPriceForecastService;

    @Autowired
    public GridCampaignService(FeatureService featureService,
                               AdGroupService adGroupService,
                               CampaignService campaignService,
                               ClientNdsService clientNdsService,
                               CampaignRepository campaignRepository,
                               GridCampaignRepository gridCampaignRepository,
                               CampaignConstantsService campaignConstantsService,
                               GridCampaignYtRepository gridCampaignYtRepository,
                               GridCampaignAccessService gridCampaignAccessService,
                               AggregatedStatusesViewService aggregatedStatusesViewService,
                               CampaignWithBrandSafetyService campaignWithBrandSafetyService,
                               MetrikaClient metrikaClient,
                               CampMetrikaCountersService campMetrikaCountersService,
                               AdGroupRepository adGroupRepository,
                               CampaignsPromotionsRepository campaignsPromotionsRepository,
                               ConversionPriceForecastService conversionPriceForecastService) {
        this.featureService = featureService;
        this.adGroupService = adGroupService;
        this.campaignService = campaignService;
        this.clientNdsService = clientNdsService;
        this.campaignRepository = campaignRepository;
        this.gridCampaignRepository = gridCampaignRepository;
        this.gridCampaignYtRepository = gridCampaignYtRepository;
        this.campaignConstantsService = campaignConstantsService;
        this.gridCampaignAccessService = gridCampaignAccessService;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.campaignWithBrandSafetyService = campaignWithBrandSafetyService;
        this.metrikaClient = metrikaClient;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.adGroupRepository = adGroupRepository;
        this.campaignsPromotionsRepository = campaignsPromotionsRepository;
        this.conversionPriceForecastService = conversionPriceForecastService;
    }

    public static final Logger logger = LoggerFactory.getLogger(GridCampaignService.class);

    /**
     * Получить список всех кампаний клиента. Получаются только те кампании, которые доступны в интерфейсе директа, а
     * так же кошельки.
     * Из всех сумм в результирующей модели будет вычтен НДС, если это применимо к валюте, в которой находится клиент
     *
     * @param clientInfo внутреннее представление клиента
     * @param operator   данные оператора
     */
    public List<GdiCampaign> getAllCampaigns(int shard, User subjectUser, GdiClientInfo clientInfo,
                                             User operator) {
        var uidClientIdShard = UidClientIdShard.of(subjectUser.getUid(), subjectUser.getClientId(), shard);
        List<GdiCampaign> campaigns = gridCampaignRepository.getAllCampaigns(uidClientIdShard);
        return enrichCampaignsAndWallets(shard, campaigns, subjectUser, clientInfo, operator);
    }

    public List<GdiBaseCampaign> getAllBaseCampaigns(int shard, User subjectUser,
                                                     GdiClientInfo clientInfo, User operator) {
        var uidClientIdShard = UidClientIdShard.of(subjectUser.getUid(), subjectUser.getClientId(), shard);
        List<GdiBaseCampaign> campaigns = gridCampaignRepository.getAllBaseCampaigns(uidClientIdShard);
        fillCommonFields(shard, subjectUser, clientInfo, operator, campaigns);
        return campaigns;
    }

    /**
     * Получить список кампаний клиента по переданному набору id.
     * Получаются только те кампании, которые доступны в интерфейсе директа, а так же кошельки.
     * Из всех сумм в результирующей модели будет вычтен НДС, если это применимо к валюте, в которой находится клиент
     */
    public List<GdiCampaign> getCampaignsAndAllWallets(int shard, Collection<Long> nonWalletCampaignIds,
                                                       User subjectUser, GdiClientInfo clientInfo,
                                                       User operator) {
        var uidClientIdShard = UidClientIdShard.of(subjectUser.getUid(), subjectUser.getClientId(), shard);
        List<GdiCampaign> campaigns =
                gridCampaignRepository.getCampaignsAndAllWallets(uidClientIdShard, nonWalletCampaignIds);
        return enrichCampaignsAndWallets(shard, campaigns, subjectUser, clientInfo, operator);
    }

    private List<GdiCampaign> enrichCampaignsAndWallets(int shard, List<GdiCampaign> campaigns,
                                                        User subjectUser, GdiClientInfo clientInfo, User operator) {
        Map<Long, AggregatedStatusCampaignData> campaignStatusesByIds =
                aggregatedStatusesViewService.getCampaignStatusesByIds(shard, listToSet(campaigns, GdiCampaign::getId));

        Set<Long> campaignIds = campaigns.stream().map(GdiCampaign::getId).collect(Collectors.toSet());
        Map<Long, List<Long>> brandSafetyCategoriesByCampaignId = campaignWithBrandSafetyService
                .getCategories(campaignIds);

        Map<Long, List<CampaignsPromotion>> campaignsPromotionsByCampaignId =
                campaignsPromotionsRepository.getCampaignsPromotionsByCid(shard, campaignIds);

        fillCommonFields(shard, subjectUser, clientInfo, operator, campaigns);
        campaigns.forEach(c -> {
            c.setAggregatedStatus(campaignStatusesByIds.get(c.getId()));
            c.setAttributionModel(nvl(c.getAttributionModel(), campaignConstantsService.getDefaultAttributionModel()));
            if (c.getBroadMatchLimit() == 0) {
                c.setBroadMatchLimit(CampaignConstants.BROAD_MATCH_LIMIT_DEFAULT);
            }
            if (c.getTimezoneId() == 0L) {
                c.setTimezoneId(TimeTargetUtils.DEFAULT_TIMEZONE);
            }
            c.setBrandSafetyCategories(brandSafetyCategoriesByCampaignId.getOrDefault(c.getId(),
                    Collections.emptyList()));
            c.setCampaignsPromotions(filterList(
                    campaignsPromotionsByCampaignId.getOrDefault(c.getId(), emptyList()),
                    cp -> !cp.getFinish().isBefore(LocalDate.now().minusDays(OLD_CAMPAIGNS_PROMOTIONS_TRESHOLD))
            ));
        });

        return campaigns;
    }

    private <T extends GdiBaseCampaign> void fillCommonFields(int shard,
                                                              User subjectUser,
                                                              GdiClientInfo clientInfo,
                                                              User operator,
                                                              List<T> campaigns) {
        List<CampaignForBlockedMoneyCheck> campaignsForCheck = mapList(campaigns, this::toCheckCamp);
        Map<Long, Boolean> campaignIdToBlocked =
                campaignService.moneyOnCampaignsIsBlocked(campaignsForCheck, false, false, true);

        ClientNds clientNds = clientNdsService
                .getEffectiveClientNds(subjectUser.getClientId(),
                        ClientId.fromNullableLong(clientInfo.getAgencyClientId()), clientInfo.getNonResident());

        Map<Long, GdiCampaignActionsHolder> idToActions =
                gridCampaignAccessService.getCampaignsActions(operator, clientInfo, campaigns, subjectUser);
        Map<Long, CampaignType> idToCampaignType = listToMap(campaigns, GdiBaseCampaign::getId,
                GdiBaseCampaign::getType);
        Map<Long, GdiCampaignSource> idToCampaignSource = listToMap(campaigns, GdiBaseCampaign::getId,
                GdiBaseCampaign::getSource);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(subjectUser.getClientId());
        filterCampaignActions(shard, idToActions, idToCampaignType, idToCampaignSource, enabledFeatures);

        //если у клиента есть кошелек, проверяем права на него отдельно
        Map<Long, GdiWalletActionsHolder> walletIdToActions =
                gridCampaignAccessService.getWalletsActions(operator, getWallets(campaigns), subjectUser);

        campaigns.forEach(c -> {
            c.setMoneyBlocked(campaignIdToBlocked.getOrDefault(c.getId(), false));
            c.setSumRest(getCampaignSumRest(c));
            substractNds(c, clientNds);

            c.setActions(idToActions.get(c.getId()));
            c.setWalletActions(walletIdToActions.getOrDefault(c.getId(), null));
            if (AvailableCampaignSources.INSTANCE.isUC(GdiCampaignSource.toSource(c.getSource()))) {
                c.setIsGoods(c.getMetatype() == GdiCampaignMetatype.ECOM);
            }
        });
    }

    public Integer getCampaignsCount(int shard, ClientId clientId) {
        return campaignRepository.getCampaignCountsForCountLimitCheck(shard, clientId.asLong()).getAll();
    }

    private <T extends GdiBaseCampaign> List<GdiBaseCampaign> getWallets(Collection<T> allCampaigns) {
        return filterList(allCampaigns, c -> c.getType() == CampaignType.WALLET);
    }

    private CampaignForBlockedMoneyCheck toCheckCamp(GdiBaseCampaign campaign) {
        return new Campaign()
                .withId(campaign.getId())
                .withUserId(campaign.getUserId())
                .withAgencyUserId(campaign.getAgencyUserId())
                .withManagerUserId(campaign.getManagerUserId())
                .withType(campaign.getType())
                .withWalletId(campaign.getWalletId())
                .withSum(campaign.getSum())
                .withSumToPay(campaign.getSumToPay())
                .withStatusModerate(campaign.getStatusModerate())
                .withStatusPostModerate(campaign.getStatusPostModerate());
    }

    private void substractNds(GdiBaseCampaign campaign, @Nullable ClientNds clientNds) {
        if (clientNds == null || campaign.getCurrencyCode() == null
                || campaign.getCurrencyCode() == CurrencyCode.YND_FIXED) {
            return;
        }

        applyNdsSubtruction(campaign::getSum, campaign::setSum, clientNds);
        applyNdsSubtruction(campaign::getSumLast, campaign::setSumLast, clientNds);
        applyNdsSubtruction(campaign::getSumSpent, campaign::setSumSpent, clientNds);
        applyNdsSubtruction(campaign::getSumToPay, campaign::setSumToPay, clientNds);
        applyNdsSubtruction(campaign::getSumRest, campaign::setSumRest, clientNds);
    }

    private static BigDecimal getCampaignSumRest(GdiBaseCampaign campaign) {
        if (campaign.getSum().compareTo(campaign.getSumSpent()) < 0) {
            return BigDecimal.ZERO;
        }
        return campaign.getSum().subtract(campaign.getSumSpent());
    }

    /**
     * Получить прогноз оставшихся дней, на которые хватит денег на кошельке для кампаний.
     * Высчитывается по формуле:<p>
     * средняя скорость трат в день = берем потраченный бюджет за последние 7 дней / на число дней, когда были
     * хоть какие-то траты<p>
     * сколько дней осталось = round(баланс / средняя скорость трат)
     *
     * @param walletSum            сумма на кошельке
     * @param campaignsUnderWallet кампании, связанные с кошельком
     */
    public Integer getPaidDaysLeft(BigDecimal walletSum, Collection<GdiBaseCampaign> campaignsUnderWallet) {
        LocalDate now = LocalDate.now();
        Map<LocalDate, BigDecimal> costsByDay =
                gridCampaignYtRepository.getCampaignCostsByDay(mapList(campaignsUnderWallet, GdiBaseCampaign::getId),
                        now.minusDays(6), now);
        return getPaidDaysLeft(walletSum, costsByDay);
    }

    /**
     * Получить прогноз оставшихся дней, на которые хватит денег на кошельке для кампаний.
     *
     * @param walletSum  сумма на кошельке
     * @param costsByDay открутки по кампаниям, связанные с кошельком, в разрезе дней
     */
    public Integer getPaidDaysLeft(BigDecimal walletSum, Map<LocalDate, BigDecimal> costsByDay) {
        int daysWithCost = 0;
        BigDecimal totalCost = BigDecimal.ZERO;
        for (Map.Entry<LocalDate, BigDecimal> entry : costsByDay.entrySet()) {
            BigDecimal cost = entry.getValue();
            if (cost.signum() > 0) {
                daysWithCost++;
                totalCost = totalCost.add(cost);
            }
        }
        if (daysWithCost == 0) {
            return null;
        }
        if (walletSum.signum() < 0) {
            return 0;
        }
        return walletSum.multiply(BigDecimal.valueOf(daysWithCost)).divide(totalCost, RoundingMode.DOWN).intValue();
    }

    public static List<GdiGoalStats> getGoalsWithCostPerAction(GdiCampaignStats stat) {
        return filterList(stat.getGoalStats(), goalStat ->
                greaterThanZero(goalStat.getCostPerAction()));
    }

    /**
     * Получить статистику по кампаниям за переданный период.
     *
     * @param campaigns        список кампаний
     * @param startDate        начало периода
     * @param endDate          конец периода (включительно)
     * @param availableGoalIds список доступных клиенту целей
     * @param enabledFeatures  список фич клиента
     * @return статистика по кампаниям
     */
    public Map<Long, GdiCampaignStats> getCampaignStatsForCampaignGoals(Collection<GdiCampaign> campaigns,
                                                                        LocalDate startDate,
                                                                        LocalDate endDate,
                                                                        @Nullable Set<Long> availableGoalIds,
                                                                        Set<String> enabledFeatures) {
        Set<Long> campaignIds = listToSet(campaigns, GdiCampaign::getId);
        Map<Long, GdiCampaignStats> stats =
                gridCampaignYtRepository.getCampaignStats(campaignIds, startDate, endDate,
                        Set.of());

        boolean directUnavailableGoalsAllowed =
                enabledFeatures.contains(FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED.getName());
        boolean getCampaignStatsOnlyByAvailableGoals =
                enabledFeatures.contains(FeatureName.GRID_CAMPAIGN_GOALS_FILTRATION_FOR_STAT.getName());
        Map<Long, GdiEntityConversion> campaignsConversion;
        if (directUnavailableGoalsAllowed && getCampaignStatsOnlyByAvailableGoals && availableGoalIds != null) {
            // после включения целей без доступа в проде хотим убедиться,
            // что не скрывать статистику - правильное решение, поэтому эта ветка должна стать неактуальной
            Map<GdiCampaign, StrategyData> strategyByCampaigns = getStrategyByCampaigns(campaigns);
            Set<Long> campaignIdsWithUnavailableGoals =
                    getCampaignIdsWithUnavailableGoals(strategyByCampaigns, availableGoalIds);

            Set<Long> payForConversionCampaignIds = getPayForConversionCampaignIds(strategyByCampaigns);
            Set<Long> payForClickCampaignIds = Sets.difference(campaignIds, payForConversionCampaignIds);

            campaignsConversion = new HashMap<>();
            if (!payForClickCampaignIds.isEmpty()) {
                // Для кампаний со стратегией "Оплата за клики"
                // конверсионные метрики показываем только по доступным целям
                campaignsConversion.putAll(gridCampaignYtRepository.getCampaignsConversionCountWithStatsFiltration(
                        payForClickCampaignIds, startDate, endDate, availableGoalIds));
            }
            if (!payForConversionCampaignIds.isEmpty()) {
                // Для кампаний со стратегией "Оплата за конверсии"
                // отображаем всю статистику по конверсионным метрикам (не ограничивая по доступным)
                campaignsConversion.putAll(gridCampaignYtRepository.getCampaignsConversionCount(
                        payForConversionCampaignIds, startDate, endDate));
                // но не отображаем статистику по кликовым метрикам для тех кампаний,
                // которые используют недоступные цели.
                Set<Long> payForConversionCampaignIdsUnavailableGoals =
                        Sets.intersection(payForConversionCampaignIds, campaignIdsWithUnavailableGoals);
                resetStatWithClicks(stats, payForConversionCampaignIdsUnavailableGoals);
            }
            campaignIdsWithUnavailableGoals.forEach(cid ->
                    ifNotNull(stats.get(cid), s -> s.withIsRestrictedByUnavailableGoals(true)));
        } else {
            boolean getRevenueOnlyByAvailableGoals =
                    enabledFeatures.contains(FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS.getName());
            campaignsConversion = gridCampaignYtRepository.getCampaignsConversionCount(campaignIds, startDate, endDate,
                    availableGoalIds, getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals);
        }
        stats.forEach((cid, stat) -> updateStatWithConversions(stat.getStat(), campaignsConversion.get(cid)));

        return collectCampaignStats(campaignIds, Set.of(), stats);
    }

    /**
     * Получить статистику по кампаниям за переданный период.
     *
     * @param campaignIds  список идентификаторов кампаний
     * @param startDate    начало периода
     * @param endDate      конец периода (включительно)
     * @param goalIds      список целей
     * @return статистика по кампаниям
     */
    public Map<Long, GdiCampaignStats> getCampaignStats(Collection<Long> campaignIds, LocalDate startDate,
                                                        LocalDate endDate, Set<Long> goalIds) {
        Map<Long, GdiCampaignStats> stats =
                gridCampaignYtRepository.getCampaignStats(campaignIds, startDate, endDate, goalIds);

        return collectCampaignStats(campaignIds, goalIds, stats);
    }

    /**
     * Получить статистику по кампаниям за переданный период
     *
     * @param campaigns       список кампаний
     * @param startDate       начало периода
     * @param endDate         конец периода (включительно)
     * @param goalIds         список целей
     * @param enabledFeatures список фич клиента
     * @return статистика по кампаниям
     */
    public Map<Long, GdiCampaignStats> getCampaignStats(Collection<GdiCampaign> campaigns,
                                                        LocalDate startDate, LocalDate endDate,
                                                        Set<Long> goalIds,
                                                        @Nullable Set<Long> availableGoalIds,
                                                        Set<String> enabledFeatures) {
        Set<Long> campaignIds = listToSet(campaigns, GdiCampaign::getId);
        boolean directUnavailableGoalsAllowed =
                enabledFeatures.contains(FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED.getName());
        boolean getCampaignStatsOnlyByAvailableGoals =
                enabledFeatures.contains(FeatureName.GRID_CAMPAIGN_GOALS_FILTRATION_FOR_STAT.getName());
        Map<Long, GdiCampaignStats> stats;
        if (directUnavailableGoalsAllowed && getCampaignStatsOnlyByAvailableGoals && availableGoalIds != null) {
            Set<Long> availableRequestedGoalIds = Sets.intersection(goalIds, availableGoalIds);
            if (availableRequestedGoalIds.size() == goalIds.size()) {
                stats = gridCampaignYtRepository.getCampaignStats(campaignIds, startDate, endDate,
                        goalIds);
            } else {
                Map<GdiCampaign, StrategyData> strategyByCampaigns = getStrategyByCampaigns(campaigns);
                Set<Long> campaignIdsWithUnavailableGoals =
                        getCampaignIdsWithUnavailableGoals(strategyByCampaigns, availableGoalIds);

                Set<Long> payForConversionCampaignIds = getPayForConversionCampaignIds(strategyByCampaigns);
                Set<Long> payForClickCampaignIds = Sets.difference(campaignIds, payForConversionCampaignIds);

                stats = new HashMap<>();
                if (!payForClickCampaignIds.isEmpty()) {
                    // Для кампаний со стратегией "Оплата за клики"
                    // конверсионные метрики показываем только по доступным целям
                    stats.putAll(gridCampaignYtRepository.getCampaignStats(
                            payForClickCampaignIds, startDate, endDate,
                            availableRequestedGoalIds));
                }
                if (!payForConversionCampaignIds.isEmpty()) {
                    // Для кампаний со стратегией "Оплата за конверсии"
                    // отображаем всю статистику по конверсионным метрикам (не ограничивая по доступным)
                    stats.putAll(gridCampaignYtRepository.getCampaignStats(
                            payForConversionCampaignIds, startDate, endDate, goalIds));
                    // но не отображаем статистику по кликовым метрикам для тех кампаний,
                    // которые используют недоступные цели.
                    Set<Long> payForConversionCampaignIdsUnavailableGoals =
                            Sets.intersection(payForConversionCampaignIds, campaignIdsWithUnavailableGoals);
                    resetStatWithClicks(stats, payForConversionCampaignIdsUnavailableGoals);
                }
                campaignIdsWithUnavailableGoals.forEach(cid ->
                        ifNotNull(stats.get(cid), s -> s.withIsRestrictedByUnavailableGoals(true)));
            }
        } else {
            boolean getRevenueOnlyByAvailableGoals =
                    enabledFeatures.contains(FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS.getName());
            Set<Long> goalIdsForRevenue = getRevenueOnlyByAvailableGoals ? availableGoalIds : null;
            stats = gridCampaignYtRepository.getCampaignStats(campaignIds, startDate, endDate,
                    goalIds, goalIdsForRevenue);
        }
        return collectCampaignStats(campaignIds, goalIds, stats);
    }

    private Map<GdiCampaign, StrategyData> getStrategyByCampaigns(Collection<GdiCampaign> campaigns) {
        return StreamEx.of(campaigns)
                .mapToEntry(Function.identity(), GdiCampaign::getStrategyData)
                .mapValues(CampaignMappings::strategyDataFromDb)
                .nonNullValues()
                .toMap();
    }

    private Set<Long> getCampaignIdsWithUnavailableGoals(Map<GdiCampaign, StrategyData> strategyByCampaigns,
                                                         Set<Long> availableGoalIds) {
        return EntryStream.of(strategyByCampaigns)
                .mapToValue((campaign, strategyData) -> extractGoalIds(strategyData,
                        campaign.getMeaningfulGoals()))
                .mapKeys(GdiCampaign::getId)
                .removeValues(goals -> Sets.difference(goals, availableGoalIds).isEmpty())
                .keys()
                .toSet();
    }

    private Set<Long> getPayForConversionCampaignIds(Map<GdiCampaign, StrategyData> strategyByCampaigns) {
        return EntryStream.of(strategyByCampaigns)
                .filterValues(strategyData -> Boolean.TRUE.equals(strategyData.getPayForConversion()))
                .mapKeys(GdiCampaign::getId)
                .keys()
                .toSet();
    }

    private Set<Long> extractGoalIds(StrategyData strategyData, @Nullable List<MeaningfulGoal> meaningfulGoals) {
        return StreamEx.of(nvl(meaningfulGoals, List.of()))
                .map(MeaningfulGoal::getGoalId)
                .append(strategyData.getGoalId())
                .nonNull()
                .remove(SPECIAL_GOAL_IDS::contains)
                .toSet();
    }

    /**
     * Занулить кликовые метрики для кампаний, на которых используется хотя бы одна недоступная цель
     */
    private void resetStatWithClicks(Map<Long, GdiCampaignStats> stats,
                                     Set<Long> campaignIdsToResetClicks) {
        campaignIdsToResetClicks.forEach(cid ->
                GridStatUtils.resetStatWithClicks(ifNotNull(stats.get(cid), GdiCampaignStats::getStat)));
    }

    private Map<Long, GdiCampaignStats> collectCampaignStats(Collection<Long> campaignIds, Set<Long> goalIds,
                                                             Map<Long, GdiCampaignStats> stats) {
        return campaignIds.stream()
                .collect(Collectors.toMap(
                        identity(),
                        id -> nvl(stats.get(id), createZeroStats(goalIds))
                ));
    }

    public Map<Long, Map<LocalDate, GdiCampaignStats>> getCampaignStatsGroupByDate(Collection<Long> campaignIds,
                                                                                   LocalDate startDate,
                                                                                   LocalDate endDate,
                                                                                   Set<Long> goalIds,
                                                                                   @Nullable Set<Long> goalIdsForRevenue,
                                                                                   ReportOptionGroupByDate groupByDate,
                                                                                   boolean getRevenueOnlyByAvailableGoals,
                                                                                   boolean getCampaignStatsOnlyByAvailableGoals,
                                                                                   boolean isUacWithOnlyInstallEnabled) {
        Map<Long, Map<LocalDate, GdiCampaignStats>> stats =
                gridCampaignYtRepository.getCampaignStatsGroupByDate(campaignIds, startDate, endDate, goalIds,
                        goalIdsForRevenue, groupByDate);
        var campaignsConversion = gridCampaignYtRepository
                .getCampaignsConversions(campaignIds, startDate, endDate, goalIds, groupByDate,
                        getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals,
                        isUacWithOnlyInstallEnabled);
        stats.forEach(
                (cid, stat) -> fillCampaignStatWithConversionsByDate(
                        stat, campaignsConversion.getOrDefault(cid, Map.of())
                )
        );
        return getCampaignStatsSafely(campaignIds, stats);
    }

    public Map<Long, Map<LocalDate, GdiCampaignStats>> getCampaignStatsForCampaignGoalsByDate(
            Collection<Long> campaignIds,
            LocalDate startDate,
            LocalDate endDate,
            @Nullable Set<Long> availableGoalIds,
            ReportOptionGroupByDate groupByDate,
            boolean getRevenueOnlyByAvailableGoals,
            boolean getCampaignStatsOnlyByAvailableGoals) {
        Map<Long, Map<LocalDate, GdiCampaignStats>> stats =
                gridCampaignYtRepository.getCampaignStatsGroupByDate(campaignIds, startDate, endDate, Set.of(),
                        null, groupByDate);
        var campaignsConversion = gridCampaignYtRepository
                .getCampaignsConversions(campaignIds, startDate, endDate, availableGoalIds, groupByDate,
                        getRevenueOnlyByAvailableGoals, getCampaignStatsOnlyByAvailableGoals, false);
        stats.forEach(
                (cid, stat) -> fillCampaignStatWithConversionsByDate(
                        stat, campaignsConversion.getOrDefault(cid, Map.of())
                )
        );
        return getCampaignStatsSafely(campaignIds, stats);
    }

    private Map<Long, Map<LocalDate, GdiCampaignStats>> getCampaignStatsSafely(
            Collection<Long> campaignIds,
            Map<Long, Map<LocalDate, GdiCampaignStats>> stats) {
        return StreamEx.of(campaignIds).toMap(id -> nvl(stats.get(id), emptyMap()));
    }

    /**
     * Получает статистику по кампаниям. В отличии от {@link #getCampaignStatsForCampaignGoalsByDate}
     * использует два запроса в yt без join-ов. Позволяет обрабатывать быстрее большое число целей.
     */
    public Map<Long, GdiCampaignStats> getCampaignStatsWithOptimization(
            Collection<Long> campaignIds, LocalDate startDate, LocalDate endDate, Set<Long> goalIds) {

        Map<Long, GdiEntityStats> campaignStatByCampaignId =
                gridCampaignYtRepository.getCampaignEntityStats(campaignIds, startDate, endDate);

        Set<Long> campaignIdsWithStat = campaignStatByCampaignId.keySet();
        Map<Long, List<GdiGoalConversion>> campaignGoalsConversionsByCampaignId =
                gridCampaignYtRepository.getCampaignGoalsConversionsCount(campaignIdsWithStat,
                        startDate, endDate, goalIds);

        Map<Long, List<GdiGoalStats>> campaignGoalsStatByCampaignId =
                calculateCampaignsGoalStat(campaignGoalsConversionsByCampaignId, campaignStatByCampaignId);

        return StreamEx.of(campaignIds)
                .mapToEntry(identity(), campaignStatByCampaignId::get)
                .mapToValue((campaignId, campaignStat) -> getCampaignStatsOrDefaultZeroStats(
                        campaignGoalsStatByCampaignId, campaignId, campaignStat, goalIds))
                .toMap();
    }

    /**
     * Для получения статистики по всем кц нужно получать статистику по нескольким целям одной кампании
     *
     * @param localDateRangeByCampaignId  для каждой кампании свой отрезок времени,
     *                                    по-которому хочется получить статистику
     * @param goalIdByCampaignId          id цели по которой хочется получиться статистику для кампании
     * @param aggregatorGoalsByCampaignId для каждой кампании список целей агрегаторов -- цели, конверсии по которым,
     *                                    считаем как сумму конверсий sub-целей
     */
    public Map<Long, GdiCampaignStats> getCampaignGoalStatsWithOptimizationForDifferentDateRanges(
            Map<Long, LocalDateRange> localDateRangeByCampaignId,
            Map<Long, Long> goalIdByCampaignId,
            Map<Long, List<GdiAggregatorGoal>> aggregatorGoalsByCampaignId) {
        Set<Long> campaignIdsWithGoalIdInStrategy = filterToSet(localDateRangeByCampaignId.keySet(),
                goalIdByCampaignId.keySet()::contains);

        if (campaignIdsWithGoalIdInStrategy.isEmpty()) {
            return emptyMap();
        }

        Map<Long, LocalDateRange> localDateRangeByCampaignIdWithGoalIdInStrategy =
                EntryStream.of(localDateRangeByCampaignId)
                        .filterKeys(campaignIdsWithGoalIdInStrategy::contains)
                        .toMap();

        Map<Long, GdiEntityStats> campaignStatByCampaignId =
                gridCampaignYtRepository.getCampaignEntityStats(localDateRangeByCampaignIdWithGoalIdInStrategy);

        Set<Long> campaignIdsWithStat = campaignStatByCampaignId.keySet();

        Map<Long, List<GdiGoalConversion>> aggregatedCampaignGoalsConversionsByCampaignId =
                getCampaignGoalsConversionWithMergingResultForAggregatedGoals(
                        goalIdByCampaignId,
                        aggregatorGoalsByCampaignId,
                        localDateRangeByCampaignIdWithGoalIdInStrategy,
                        campaignIdsWithStat);

        Map<Long, List<GdiGoalStats>> campaignGoalsStatByCampaignId =
                calculateCampaignsGoalStat(aggregatedCampaignGoalsConversionsByCampaignId, campaignStatByCampaignId);

        return StreamEx.of(localDateRangeByCampaignIdWithGoalIdInStrategy.keySet())
                .mapToEntry(identity(), campaignStatByCampaignId::get)
                .mapToValue((campaignId, campaignStat) ->
                        getCampaignStatsOrDefaultZeroStats(campaignGoalsStatByCampaignId, campaignId, campaignStat,
                                Set.of(goalIdByCampaignId.get(campaignId))))
                .toMap();
    }

    /**
     * Получаем данные о конверсиях
     * Для целей агрегирующих другие цели суммируем конверсии
     */
    private Map<Long, List<GdiGoalConversion>> getCampaignGoalsConversionWithMergingResultForAggregatedGoals(
            Map<Long, Long> goalIdByCampaignId,
            Map<Long, List<GdiAggregatorGoal>> aggregatorGoalsByCampaignId,
            Map<Long, LocalDateRange> localDateRangeByCampaignIdWithGoalIdInStrategy,
            Set<Long> campaignIdsWithStat) {
        Map<Long, Set<Long>> flatGoalIdsByCampaignId = EntryStream.of(goalIdByCampaignId)
                .mapToValue((campaignId, goalId) ->
                        extractGoalIdsFromAggregatedGoalId(aggregatorGoalsByCampaignId.get(campaignId), goalId))
                .toMap();

        Map<Long, List<GdiGoalConversion>> campaignGoalsConversionsByCampaignId =
                getCampaignGoalsConversionsCountForCampaignIdWithStat(flatGoalIdsByCampaignId,
                        localDateRangeByCampaignIdWithGoalIdInStrategy, campaignIdsWithStat);

        return EntryStream.of(campaignGoalsConversionsByCampaignId)
                .mapToValue((campaignId, stats) ->
                        sumGoalStatForAggregatedGoals(getAggregatorGoalIdBySubGoalId(aggregatorGoalsByCampaignId.get(campaignId)), stats))
                .toMap();
    }

    public static Map<Long, Long> getAggregatorGoalIdBySubGoalId(@Nullable List<GdiAggregatorGoal> aggregatorGoals) {
        if (isEmpty(aggregatorGoals)) {
            return emptyMap();
        }
        return StreamEx.of(aggregatorGoals)
                .mapToEntry(GdiAggregatorGoal::getId, GdiAggregatorGoal::getSubGoalIds)
                .flatMapValues(Collection::stream)
                .invert()
                .toMap();
    }

    public static List<GdiGoalConversion> sumGoalStatForAggregatedGoals(Map<Long, Long> aggregatorGoalIdByGoalId,
                                                                  List<GdiGoalConversion> stats) {
        List<GdiGoalConversion> goalConversions = filterList(stats,
                stat -> !aggregatorGoalIdByGoalId.containsKey(stat.getGoalId()));

        return StreamEx.of(stats)
                .mapToEntry(GdiGoalConversion::getGoalId, Function.identity())
                .filterKeys(aggregatorGoalIdByGoalId::containsKey)
                .mapKeys(aggregatorGoalIdByGoalId::get)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapKeyValue(GridCampaignService::sumAggregatedGoalConversions)
                .append(goalConversions)
                .toList();
    }

    private Set<Long> extractGoalIdsFromAggregatedGoalId(@Nullable List<GdiAggregatorGoal> aggregatorGoals,
                                                         Long goalId) {
        Map<Long, GdiAggregatorGoal> aggregatorGoalById = listToMap(aggregatorGoals, GdiAggregatorGoal::getId);
        return aggregatorGoalById != null && aggregatorGoalById.containsKey(goalId) ?
                Set.copyOf(aggregatorGoalById.get(goalId).getSubGoalIds()) :
                Set.of(goalId);
    }

    public static GdiCampaignStats getCampaignStatsOrDefaultZeroStats(
            Map<Long, List<GdiGoalStats>> campaignGoalsStatByCampaignId,
            Long campaignId,
            @Nullable GdiEntityStats campaignStat,
            Set<Long> goalsIds) {
        return campaignStat == null ?
                createZeroStats(goalsIds) :
                new GdiCampaignStats()
                        .withStat(campaignStat)
                        .withGoalStats(addZeroStatForAbsentGoals(campaignGoalsStatByCampaignId.get(campaignId),
                                goalsIds));
    }

    private static List<GdiGoalStats> addZeroStatForAbsentGoals(@Nullable List<GdiGoalStats> goalStats,
                                                                Set<Long> goalIds) {
        if (goalStats == null) {
            return createZeroGoalStats(goalIds);
        }

        Set<Long> goalIdsWithExistedStat = listToSet(goalStats, GdiGoalStats::getGoalId);
        List<GdiGoalStats> resultGoalStats = createZeroGoalStats(
                filterToSet(goalIds, id -> !goalIdsWithExistedStat.contains(id)));
        resultGoalStats.addAll(goalStats);
        return resultGoalStats;
    }

    public Map<Long, List<GdiGoalConversion>> getCampaignGoalsConversionsCountForCampaignIdWithStat(
            Map<Long, Set<Long>> goalIdsByCampaignId,
            Map<Long, LocalDateRange> localDateRangeByCampaignIdWithGoalIdInStrategy,
            Set<Long> campaignIdsWithStat) {
        Map<Long, LocalDateRange> localDateRangeByCampaignIdWithStat =
                EntryStream.of(localDateRangeByCampaignIdWithGoalIdInStrategy)
                        .filterKeys(campaignIdsWithStat::contains)
                        .toMap();
        Map<Long, Set<Long>> goalIdByCampaignIdWithStat = EntryStream.of(goalIdsByCampaignId)
                .filterKeys(campaignIdsWithStat::contains)
                .toMap();
        return gridCampaignYtRepository.getCampaignGoalsConversionsCount(
                localDateRangeByCampaignIdWithStat, goalIdByCampaignIdWithStat);
    }

    /**
     * Получить статистику по кампаниям за переданный период с фильтрацией по доступным целям
     *
     * @param campaignIds  список идентификаторов кампаний
     * @param startDate    начало периода
     * @param endDate      конец периода (включительно)
     * @param goalIds      список целей
     */
    public Map<Long, GdiCampaignStats> getCampaignStatsWithGoalFiltering(
            Collection<Long> campaignIds,
            LocalDate startDate,
            LocalDate endDate,
            Set<Long> goalIds,
            Set<Long> availableGoalIds) {
        Map<Long, GdiCampaignStats> stats =
                gridCampaignYtRepository.getCampaignStatsWithFilteringGoals(campaignIds, startDate, endDate,
                        goalIds, availableGoalIds);

        return collectCampaignStats(campaignIds, goalIds, stats);
    }

    public static GdiCampaignStats createZeroStats(Set<Long> goalIds) {
        return new GdiCampaignStats()
                .withStat(GridStatNew.addZeros(new GdiEntityStats()))
                .withGoalStats(createZeroGoalStats(goalIds));
    }

    public static List<GdiGoalStats> createZeroGoalStats(Set<Long> goalIds) {
        return GridStatNew.addZeros(mapList(goalIds,
                goalId -> new GdiGoalStats().withGoalId(goalId)));
    }

    public Map<Long, GdiCampaignGoalsCostPerAction> getAverageCostPerActionForGoalsInSpecifiedCampaigns(
            ClientId clientId,
            Map<Long, List<Long>> goalIdsByCampaignId,
            Map<Long, String> urlsByCampaignId,
            LocalDate startDate,
            LocalDate endDate) {
        if (goalIdsByCampaignId.isEmpty()) {
            return emptyMap();
        }
        Set<Long> goalIdsToRequest = flatMapToSet(goalIdsByCampaignId.values(), identity());

        Map<Long, GdiCampaignStats> campaignStats = getCampaignStatsWithOptimization(goalIdsByCampaignId.keySet(),
                startDate, endDate, goalIdsToRequest);

        Set<Long> goalIdsWithPerCampStatistics = StreamEx.of(campaignStats.values())
                .remove(stat -> stat.getGoalStats().isEmpty())
                .map(GridCampaignService::getGoalsWithCostPerAction)
                .flatMap(Collection::stream)
                .map(GdiGoalStats::getGoalId)
                .toSet();

        if (goalIdsWithPerCampStatistics.size() == goalIdsToRequest.size()) {
            return EntryStream.of(campaignStats)
                    .mapValues(stat -> toGdiCampaignGoalCostPerAction(stat.getGoalStats()))
                    .toMap();
        }

        //для целей без достижений внутри кампании считаем cpa по логину
        Set<Long> goalIdsWithoutPerCampStatistics = filterToSet(goalIdsToRequest,
                goalId -> !goalIdsWithPerCampStatistics.contains(goalId));

        Map<Long, BigDecimal> averageCostPerActionForGoalsByLoginStats =
                getAverageCostPerActionForGoalsByLoginStats(clientId, goalIdsWithoutPerCampStatistics,
                        startDate, endDate);

        Set<Long> goalIdsWithPerLoginStatistics = averageCostPerActionForGoalsByLoginStats.keySet();

        boolean allGoalsHaveStatsPerCampOrLogin =
                goalIdsWithPerLoginStatistics.size() + goalIdsWithPerCampStatistics.size() == goalIdsToRequest.size();
        boolean categoryCpaSourceFeatureEnabled = featureService.isEnabledForClientId(clientId,
                CATEGORY_CPA_SOURCE_FOR_CONVERSION_PRICE_RECOMMENDATION);
        if (allGoalsHaveStatsPerCampOrLogin || !categoryCpaSourceFeatureEnabled) {
            var emptyResultForCategoryCpaSource =
                    StreamEx.of(campaignStats.keySet())
                            .toMap(c -> new CampaignConversionPriceForGoalsWithCategoryCpaSource(emptyMap()));
            return mergeResults(goalIdsByCampaignId, campaignStats, averageCostPerActionForGoalsByLoginStats,
                    emptyResultForCategoryCpaSource);
        }

        //для целей без достижений внутри кампании и по логину считаем cpa по категории
        Set<Long> goalIdsWithoutPerCampOrLoginStatistics = filterToSet(goalIdsToRequest,
                goalId -> !goalIdsWithPerCampStatistics.contains(goalId) && !goalIdsWithPerLoginStatistics.contains(goalId));

        Map<Long, CampaignConversionPriceForGoalsWithCategoryCpaSource> averageCostPerActionForGoalsByUrls =
                getAverageCostPerActionForGoalsByWithCategoryCpaSource(
                        clientId,
                        goalIdsWithoutPerCampOrLoginStatistics,
                        urlsByCampaignId);

        return mergeResults(goalIdsByCampaignId, campaignStats, averageCostPerActionForGoalsByLoginStats,
                averageCostPerActionForGoalsByUrls);
    }

    public Map<Long, GdiCampaignGoalsCostPerAction> mergeResults(Map<Long, List<Long>> goalIdsByCampaignId,
                                                                 Map<Long, GdiCampaignStats> campaignStatsMap,
                                                                 Map<Long, BigDecimal> goalsAveregeCostPerActionForClientCampaignsByGoalId,
                                                                 Map<Long,
                                                                         CampaignConversionPriceForGoalsWithCategoryCpaSource> avCostPerActionForGoalsByCategory) {

        return EntryStream.of(goalIdsByCampaignId)
                .mapToValue((campaignId, goalIds) -> calculateGdiCampaignGoalsCostPerAction(
                        goalIds,
                        goalsAveregeCostPerActionForClientCampaignsByGoalId,
                        campaignStatsMap.get(campaignId),
                        avCostPerActionForGoalsByCategory.get(campaignId)))
                .toMap();
    }

    public static GdiCampaignGoalsCostPerAction calculateGdiCampaignGoalsCostPerAction(List<Long> goalIds,
                                                                                       Map<Long, BigDecimal> averageCostPerActionForGoalsWithLoginCpaSource,
                                                                                       @Nullable GdiCampaignStats campaignGoalsStat,
                                                                                       CampaignConversionPriceForGoalsWithCategoryCpaSource campaignConversionPriceForGoalsWithCategoryCpaSource) {

        //статистика может быть пустой, или goalStat в ней может быть пустой, или там могут быть costPerAction=0
        // во всех указанных случаях используем статистику на логин. Если статистика на логин пуста -
        // смотрим статистические значения по категории.
        Map<Long, BigDecimal> campaignGoalCostPerActionByGoalId = campaignGoalsStat != null
                ? listToMap(campaignGoalsStat.getGoalStats(), GdiGoalStats::getGoalId, GdiGoalStats::getCostPerAction)
                : emptyMap();

        Map<Long, BigDecimal> averageCostPerActionForGoalsWithCategoryCpaSource =
                campaignConversionPriceForGoalsWithCategoryCpaSource.getPriceByGoalId();

        List<GdiGoalCostPerAction> goalCostPerAction = StreamEx.of(goalIds)
                .sorted()
                .filter(goalId -> hasGoalCost(averageCostPerActionForGoalsWithLoginCpaSource,
                        campaignGoalCostPerActionByGoalId, averageCostPerActionForGoalsWithCategoryCpaSource, goalId))
                .map(goalId -> calculateGdiGoalCostPerAction(averageCostPerActionForGoalsWithLoginCpaSource,
                        campaignGoalCostPerActionByGoalId, averageCostPerActionForGoalsWithCategoryCpaSource, goalId))
                .toList();
        return new GdiCampaignGoalsCostPerAction()
                .withGoalsCostPerAction(goalCostPerAction);
    }

    private static GdiGoalCostPerAction calculateGdiGoalCostPerAction(Map<Long, BigDecimal> averageCostPerActionForGoalsWithLoginCpaSource,
                                                                      Map<Long, BigDecimal> campaignGoalCostPerActionByGoalId,
                                                                      Map<Long, BigDecimal> averageCostPerActionForGoalsWithCategoryCpaSource,
                                                                      Long goalId) {
        if (greaterThanZero(campaignGoalCostPerActionByGoalId.get(goalId))) {
            return toGdiCampaignGoalCostPerAction(goalId,
                    campaignGoalCostPerActionByGoalId.get(goalId));
        } else if (averageCostPerActionForGoalsWithLoginCpaSource.containsKey(goalId)) {
            return toGdiCampaignGoalCostPerActionForLogin(goalId,
                    averageCostPerActionForGoalsWithLoginCpaSource);
        } else {
            return toGdiCampaignGoalCostPerActionForCategory(goalId,
                    averageCostPerActionForGoalsWithCategoryCpaSource.get(goalId));
        }
    }

    private static boolean hasGoalCost(Map<Long, BigDecimal> averageCostPerActionForGoalsWithLoginCpaSource,
                                       Map<Long, BigDecimal> campaignGoalCostPerActionByGoalId,
                                       Map<Long, BigDecimal> averageCostPerActionForGoalsWithCategoryCpaSource,
                                       Long goalId) {
        return greaterThanZero(campaignGoalCostPerActionByGoalId.get(goalId)) ||
                averageCostPerActionForGoalsWithLoginCpaSource.containsKey(goalId) ||
                averageCostPerActionForGoalsWithCategoryCpaSource.containsKey(goalId);
    }

    public static GdiCampaignGoalsCostPerAction toGdiCampaignGoalCostPerAction(List<GdiGoalStats> goalStats) {
        var goalCostPerActionList = StreamEx.of(goalStats)
                .sortedBy(GdiGoalStats::getGoalId)
                .map(GridCampaignService::toGdiCampaignGoalCostPerAction)
                .toList();

        return new GdiCampaignGoalsCostPerAction()
                .withGoalsCostPerAction(goalCostPerActionList);
    }

    public static GdiGoalCostPerAction toGdiCampaignGoalCostPerAction(Long goalId, BigDecimal campCostPerAction) {
        return new GdiGoalCostPerAction()
                .withGoalId(goalId)
                .withSource(GdiCpaSource.CAMPAIGN)
                .withCostPerAction(campCostPerAction);
    }

    public static GdiGoalCostPerAction toGdiCampaignGoalCostPerActionForLogin(Long goalId,
                                                                              BigDecimal averegeCostPerActionForClientCampaigns) {
        return new GdiGoalCostPerAction()
                .withGoalId(goalId)
                .withSource(GdiCpaSource.LOGIN)
                .withCostPerAction(averegeCostPerActionForClientCampaigns);
    }

    public static GdiGoalCostPerAction toGdiCampaignGoalCostPerActionForCategory(Long goalId,
                                                                                 BigDecimal avCostPerActionForCategory) {
        return new GdiGoalCostPerAction()
                .withGoalId(goalId)
                .withSource(GdiCpaSource.CATEGORY)
                .withCostPerAction(avCostPerActionForCategory);
    }

    public static GdiGoalCostPerAction toGdiCampaignGoalCostPerAction(GdiGoalStats goalStat) {
        return new GdiGoalCostPerAction()
                .withGoalId(goalStat.getGoalId())
                .withSource(GdiCpaSource.CAMPAIGN)
                .withCostPerAction(goalStat.getCostPerAction());
    }

    public static GdiGoalCostPerAction toGdiCampaignGoalCostPerActionForLogin(Long goalId,
                                                                              Map<Long, BigDecimal> goalsAveregeCostPerActionForLoginCampaigns) {
        return new GdiGoalCostPerAction()
                .withGoalId(goalId)
                .withSource(GdiCpaSource.LOGIN)
                .withCostPerAction(goalsAveregeCostPerActionForLoginCampaigns.get(goalId));
    }

    /**
     * Считает средний cpa для переданных целей по всем кампаниям клиента за указанный период.
     * Если данных по цели нет, в мапе не будет ключа с id цели.
     */
    public Map<Long, BigDecimal> getAverageCostPerActionForGoalsByLoginStats(ClientId clientId,
                                                                             Set<Long> goalIds,
                                                                             LocalDate startDate,
                                                                             LocalDate endDate) {
        if (goalIds.isEmpty()) {
            return emptyMap();
        }
        Set<Long> campaignIds = campaignService.getClientCampaignIds(clientId);
        Map<Long, GdiCampaignStats> campaignStats = getCampaignStatsWithOptimization(campaignIds, startDate, endDate,
                goalIds);

        Map<Long, Long> actionCountByGoalId = getActionCountByGoalId(campaignStats, true);

        Map<Long, Set<Long>> goalIdsWithGoalCountersByCampaignId =
                getGoalIdsFromCampaignCountersByCampaignId(clientId, campaignIds, goalIds);
        Map<Long, BigDecimal> costByGoalId = getCostByGoalId(campaignStats, goalIdsWithGoalCountersByCampaignId);

        return EntryStream.of(actionCountByGoalId)
                .mapToValue((goalId, actionsCount) -> calculateCpa(costByGoalId.get(goalId),
                        BigDecimal.valueOf(actionsCount)))
                .toMap();
    }

    /*
    Получить кампании с подмножеством целей принадлежацих счетчику, заданному на кампании.
     */
    private Map<Long, Set<Long>> getGoalIdsFromCampaignCountersByCampaignId(ClientId clientId, Set<Long> campaignIds,
                                                                            Set<Long> requestedGoalIds) {
        Map<Long, List<Long>> counterIdsByCampaignId = campMetrikaCountersService.getCounterByCampaignIds(clientId,
                campaignIds);

        Set<Integer> counterIds = flatMapToSet(counterIdsByCampaignId.values(), a -> mapList(a, Long::intValue));

        Map<Integer, List<CounterGoal>> goalsByCounterId =
                metrikaClient.getMassCountersGoalsFromMetrika(counterIds);

        Map<Long, Set<Long>> requestedGoalIdsByCounterId = EntryStream.of(goalsByCounterId)
                .mapKeys(Integer::longValue)
                .flatMapValues(counterGoals -> counterGoals.stream().map(goal -> (long) goal.getId()))
                .filterValues(requestedGoalIds::contains)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys(Collectors.toSet())
                .toMap();

        return EntryStream.of(counterIdsByCampaignId)
                .flatMapValues(Collection::stream)
                .mapValues(requestedGoalIdsByCounterId::get)
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .grouping(Collectors.toSet());
    }

    /**
     * Общее количество конверсий по целям на всех кампаниях клиента
     */
    public Map<Long, Long> getTotalActionsCountForGoalsInAllClientCampaigns(ClientId clientId,
                                                                            Set<Long> goalIds,
                                                                            LocalDate startDate,
                                                                            LocalDate endDate) {
        if (goalIds.isEmpty()) {
            return emptyMap();
        }
        Set<Long> campaignIds = campaignService.getClientCampaignIds(clientId);
        Map<Long, GdiCampaignStats> campaignStats = getCampaignStatsWithOptimization(campaignIds, startDate, endDate,
                goalIds);

        return getActionCountByGoalId(campaignStats, false);
    }

    /**
     * Средняя цена за конверсию для кампаний по разным типам целей
     */
    public Map<Long, CampaignConversionPriceForGoalsWithCategoryCpaSource> getAverageCostPerActionForGoalsByWithCategoryCpaSource(ClientId clientId,
                                                                                                                                  Set<Long> goalIds,
                                                                                                                                  Map<Long, String> urlByCampaignId) {
        if (goalIds.isEmpty()) {
            return listToMap(
                    urlByCampaignId.keySet(),
                    identity(),
                    url -> new CampaignConversionPriceForGoalsWithCategoryCpaSource(emptyMap()));
        }

        return conversionPriceForecastService.getRecommendedConversionPriceByGoalIds(clientId, goalIds,
                urlByCampaignId);
    }

    /**
     * Средняя цена за конверсию для кампании без campaignId
     */
    public CampaignConversionPriceForGoalsWithCategoryCpaSource getAverageCostPerActionForGoalsByWithCategoryCpaSource(ClientId clientId,
                                                                                                                       Set<Long> goalIds,
                                                                                                                       String url) {
        if (goalIds.isEmpty()) {
            return new CampaignConversionPriceForGoalsWithCategoryCpaSource(emptyMap());
        }

        return conversionPriceForecastService.getRecommendedConversionPriceByGoalIdsForDraftCamp(clientId, goalIds,
                url);
    }

    private static BigDecimal calculateCpa(BigDecimal cost, BigDecimal actionsCount) {
        return cost.divide(actionsCount, RoundingMode.HALF_UP);
    }

    private static Map<Long, BigDecimal> getCostByGoalId(Map<Long, GdiCampaignStats> campaignStats, Map<Long,
            Set<Long>> goalIdsFromCampaignCountersByCampaignId) {
        return EntryStream.of(campaignStats)
                .flatMapValues(GridCampaignService::getEntriesOfCostAndGoalStat)
                .filterKeyValue((campaignId, costAndGoalStatsEntry) -> hasConversionsForGoalOrCampaignUseGoalCounter(
                        campaignId, costAndGoalStatsEntry.getValue(), goalIdsFromCampaignCountersByCampaignId))
                .mapToKey((campaignId, costAndGoalStatsEntry) -> costAndGoalStatsEntry.getValue().getGoalId())
                .mapValues(Map.Entry::getKey)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys(BigDecimal::add)
                .toMap();
    }

    private static Stream<Map.Entry<BigDecimal, GdiGoalStats>> getEntriesOfCostAndGoalStat(GdiCampaignStats campaignStats) {
        return StreamEx.of(campaignStats.getGoalStats())
                .filter(goalStat -> goalStat.getGoals() != null)
                .map(goalStats -> Maps.immutableEntry(campaignStats.getStat().getCost(), goalStats));
    }

    private static Map<Long, Long> getActionCountByGoalId(Map<Long, GdiCampaignStats> campaignStats,
                                                          boolean filterZeroStat) {
        return StreamEx.ofValues(campaignStats)
                .flatMap(stat -> stat.getGoalStats().stream())
                .filter(goalStat -> !filterZeroStat || GridCampaignService.hasConversionsForGoal(goalStat))
                .mapToEntry(GdiGoalStats::getGoalId, GdiGoalStats::getGoals)
                .toMap(Long::sum);
    }

    private static boolean hasConversionsForGoalOrCampaignUseGoalCounter(Long campaignId, GdiGoalStats goalStat,
                                                                         Map<Long, Set<Long>> goalIdsFromCampaignCountersByCampaignId) {
        return goalStat.getGoals() != null && (goalStat.getGoals() > 0
                || hasCampaignGoalCounter(campaignId, goalStat.getGoalId(), goalIdsFromCampaignCountersByCampaignId));
    }

    private static boolean hasCampaignGoalCounter(Long campaignId, Long goalId,
                                                  Map<Long, Set<Long>> goalIdsFromCampaignCountersByCampaignId) {
        return goalIdsFromCampaignCountersByCampaignId.containsKey(campaignId)
                && goalIdsFromCampaignCountersByCampaignId.get(campaignId).contains(goalId);
    }

    private static boolean hasConversionsForGoal(GdiGoalStats goalStat) {
        return goalStat.getGoals() != null && goalStat.getGoals() > 0;
    }

    /**
     * Удаляет недоступные действия над кампаниями из {@code GdiCampaignActionsHolder}
     */
    private void filterCampaignActions(int shard, Map<Long, GdiCampaignActionsHolder> idToActions,
                                       Map<Long, CampaignType> idToCampaignType,
                                       Map<Long, GdiCampaignSource> idToCampaignSource,
                                       Set<String> enabledFeatures) {
        Set<Long> cpmBannerCampaignIds = EntryStream.of(idToCampaignType)
                .filterValues(type -> type == CampaignType.CPM_BANNER)
                .keys()
                .toSet();
        var geoproductAvailability = adGroupService.getGeoproductAvailabilityByCampaignId(shard, cpmBannerCampaignIds);
        boolean hasCpmGeoproductEnabled = enabledFeatures.contains(FeatureName.CPM_GEOPRODUCT_ENABLED.getName());

        Set<Long> contentPromotionCampaignIds = EntryStream.of(idToCampaignType)
                .filterValues(type -> type == CampaignType.CONTENT_PROMOTION)
                .keys()
                .toSet();
        Map<Long, ContentPromotionAdgroupType> contentPromotionTypeByCampaignId = contentPromotionCampaignIds.isEmpty()
                ? Map.of()
                : adGroupRepository.getContentPromotionAdGroupTypeByCampaignId(shard, contentPromotionCampaignIds);
        var hasContentPromotionVideoEnabled = enabledFeatures.contains(FeatureName.CONTENT_PROMOTION_VIDEO.getName());
        var hasContentPromotionCollectionEnabled =
                enabledFeatures.contains(FeatureName.CONTENT_PROMOTION_COLLECTION.getName());
        var hasCpmDealsEnabled = enabledFeatures.contains(FeatureName.CPM_DEALS.getName());

        idToActions.forEach((cid, actionsHolder) -> {
            // Запрещаем разархивировать геопродуктовые, CpmDeals и ContentPromotion кампании без фич
            var campaignType = idToCampaignType.get(cid);
            if ((campaignType == CampaignType.CPM_BANNER &&
                    geoproductAvailability.get(cid) == GeoproductAvailability.YES && !hasCpmGeoproductEnabled)
                    || (campaignType == CampaignType.CPM_DEALS && !hasCpmDealsEnabled)
                    || (campaignType == CampaignType.CONTENT_PROMOTION && contentPromotionCannotBeUnarchived(
                    cid, contentPromotionTypeByCampaignId, hasContentPromotionVideoEnabled,
                    hasContentPromotionCollectionEnabled))) {
                var modifiedActions =
                        Sets.difference(actionsHolder.getActions(), Set.of(GdiCampaignAction.UNARCHIVE_CAMP));

                actionsHolder.withActions(modifiedActions);
            }
        });

        var viewRedirectCampaignSources = getSourcesWithRedirectEnabled(enabledFeatures, false);
        var editRedirectCampaignSources = getSourcesWithRedirectEnabled(enabledFeatures, true);
        idToActions.forEach((cid, actionsHolder) -> {
            var campaignSource = idToCampaignSource.get(cid);
            boolean viewRedirect = viewRedirectCampaignSources.contains(campaignSource);
            boolean editRedirect = editRedirectCampaignSources.contains(campaignSource);
            if (viewRedirect || editRedirect) {
                Set<GdiCampaignAction> modifiedActions = new HashSet<>(actionsHolder.getActions());
                if (viewRedirect) {
                    modifiedActions.add(GdiCampaignAction.REDIRECT_VIEW);
                }
                if (editRedirect) {
                    modifiedActions.add(GdiCampaignAction.REDIRECT_EDIT);
                }
                actionsHolder.withActions(modifiedActions);
            }
        });
    }

    private boolean contentPromotionCannotBeUnarchived(Long cid,
                                                       Map<Long, ContentPromotionAdgroupType> cpTypeByCampaignId,
                                                       boolean hasContentPromotionVideoEnabled,
                                                       boolean hasContentPromotionCollectionEnabled) {
        var cpType = cpTypeByCampaignId.get(cid);
        return (cpType == ContentPromotionAdgroupType.VIDEO && !hasContentPromotionVideoEnabled)
                || (cpType == ContentPromotionAdgroupType.COLLECTION && !hasContentPromotionCollectionEnabled);
    }

    static GdiGoalConversion sumAggregatedGoalConversions(Long goalId, List<GdiGoalConversion> goalConversions) {
        var gdiGoalConversion = new GdiGoalConversion()
                .withGoalId(goalId)
                .withRevenue(0L)
                .withGoals(0L);
        StreamEx.of(goalConversions).foldLeft(gdiGoalConversion, (a, b) ->
                a.withRevenue(a.getRevenue() + b.getRevenue())
                        .withGoals(a.getGoals() + b.getGoals()));
        return gdiGoalConversion;
    }

}
