package ru.yandex.direct.grid.core.entity.campaign.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalConversion;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;

import static ru.yandex.direct.grid.core.util.stats.GridStatUtils.updateStatWithConversions;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;

@ParametersAreNonnullByDefault
public class GridCampaignServiceUtils {
    public static final long PERCENT_MULTIPLIER = 100L;

    public static final Map<GdiCampaignSource, String> VISIBLE_FEATURE_BY_CAMPAIGN_SOURCE = Map.of(
            GdiCampaignSource.USLUGI, FeatureName.CAMPAIGN_SOURCE_USLUGI_VISIBLE.getName(),
            GdiCampaignSource.EDA, FeatureName.CAMPAIGN_SOURCE_EDA_VISIBLE.getName()
    );
    public static final Map<GdiCampaignSource, String> EDITABLE_FEATURE_BY_CAMPAIGN_SOURCE = Map.of(
            GdiCampaignSource.USLUGI, FeatureName.CAMPAIGN_SOURCE_USLUGI_EDITABLE.getName(),
            GdiCampaignSource.EDA, FeatureName.CAMPAIGN_SOURCE_EDA_EDITABLE.getName()
    );

    private GridCampaignServiceUtils() {
    }

    /**
     * Получаем статистику целей из данных о конверсиях и статистики по кампаниям
     */
    public static Map<Long, List<GdiGoalStats>> calculateCampaignsGoalStat(
            Map<Long, List<GdiGoalConversion>> campaignGoalsConversionByCampaignId,
            Map<Long, GdiEntityStats> campaignStatByCampaignId) {
        return EntryStream.of(campaignGoalsConversionByCampaignId)
                .mapToValue((campaignId, goalsStat) -> calculateCampaignGoalsStat(campaignStatByCampaignId, campaignId,
                        goalsStat))
                .toMap();
    }

    private static List<GdiGoalStats> calculateCampaignGoalsStat(Map<Long, GdiEntityStats> campaignStatByCampaignId,
                                                                 Long campaignId,
                                                                 List<GdiGoalConversion> goalsConversion) {
        return mapList(goalsConversion, goalStat -> calculateGoalStat(campaignStatByCampaignId.get(campaignId),
                goalStat));
    }

    private static GdiGoalStats calculateGoalStat(GdiEntityStats campaignStat, GdiGoalConversion goalConversion) {
        boolean hasConversions = goalConversion.getGoals() != null && goalConversion.getGoals() > 0;
        GdiGoalStats goalStats = new GdiGoalStats()
                .withGoalId(goalConversion.getGoalId())
                .withGoals(goalConversion.getGoals())
                .withRevenue(goalConversion.getRevenue())
                //из-за асинхронности попадания данных в таблички yt со статистикой, может быть так,
                //что есть конверсии, а кликов нет поэтому заполняем нулями
                .withCostPerAction(BigDecimal.ZERO)
                .withConversionRate(BigDecimal.ZERO);
        if (hasConversions) {
            BigDecimal conversionCount = BigDecimal.valueOf(goalConversion.getGoals());

            BigDecimal costPerAction = campaignStat.getCost().divide(conversionCount, RoundingMode.HALF_UP);
            goalStats.setCostPerAction(costPerAction);
        }

        if (hasConversions && greaterThanZero(campaignStat.getClicks())) {
            BigDecimal conversionRate = BigDecimal.valueOf(goalConversion.getGoals() * PERCENT_MULTIPLIER)
                    .divide(campaignStat.getClicks(), RoundingMode.HALF_UP);
            goalStats.setConversionRate(conversionRate);
        }
        return goalStats;
    }

    static void fillCampaignStatWithConversionsByDate(Map<LocalDate, GdiCampaignStats> statsByDate,
                                                      Map<LocalDate, GdiEntityConversion> conversionsByDate) {
        statsByDate.forEach((day, campaignStats) ->
                updateStatWithConversions(campaignStats.getStat(), conversionsByDate.get(day))
        );
    }

    /**
     * @param enabledFeatures - включенные фичи пользователя
     * @param editable        - если true, то проверяем доступность редиректа на редактирование,
     *                        если false, то проверяем доступность просмотра
     * @return список доступных CampaignSource
     */
    public static Set<GdiCampaignSource> getSourcesWithRedirectEnabled(Set<String> enabledFeatures, boolean editable) {
        return EntryStream.of(editable ? EDITABLE_FEATURE_BY_CAMPAIGN_SOURCE : VISIBLE_FEATURE_BY_CAMPAIGN_SOURCE)
                .filterValues(enabledFeatures::contains)
                .keys()
                .append(GdiCampaignSource.ZEN)
                .toSet();
    }
}
