package ru.yandex.direct.grid.core.entity.deal.service;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.deal.container.DealStat;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.core.entity.placements.model.Placement;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDeal;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDealPlacement;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class GdiDealConverter {

    public static GdiDeal dealToGdiDeal(Deal deal, Map<Long, Placement> placementsMap) {
        return new GdiDeal()
                .withId(deal.getId())
                .withName(Optional.ofNullable(deal.getName()).orElse(deal.getAdfoxName()))
                .withDealType(deal.getDealType())
                .withMinPrice(deal.getAgencyMinPrice().bigDecimalValue())
                .withStatus(deal.getDirectStatus())
                .withCompleteReason(deal.getCompleteReason())
                .withCurrencyCode(String.valueOf(deal.getCurrencyCode()))
                .withContacts(deal.getContacts())
                .withDateStart(deal.getDateStart())
                .withDateEnd(deal.getDateEnd())
                .withDescription(Optional.ofNullable(deal.getDescription()).orElse(deal.getAdfoxDescription()))
                .withPublisherName(deal.getPublisherName())
                .withExpectedImpressionsPerWeek(deal.getExpectedImpressionsPerWeek())
                .withExpectedMoneyPerWeek(deal.getExpectedMoneyPerWeek())
                .withTargetingsText(deal.getTargetingsText())
                .withAgencyFeePercent(
                        deal.getAgencyFeePercent() == null ? BigDecimal.ZERO : deal.getAgencyFeePercent().asPercent())
                .withPlacements(filterList(
                        mapList(deal.getPlacements(), p -> fromDealPlacement(p, placementsMap)),
                        Objects::nonNull
                ));
    }

    public static void addStatToGdiDeal(GdiDeal gdiDeal, DealStat dealStat) {
        gdiDeal.withShows(dealStat.getShows())
                .withClicks(dealStat.getClicks())
                .withSpent(dealStat.getSpent())
                .withCpc(dealStat.getCpc())
                .withCpm(dealStat.getCpm())
                .withCtr(dealStat.getCtr() == null ? null : dealStat.getCtr().asPercent());
    }

    @Nullable
    private static GdiDealPlacement fromDealPlacement(@Nullable DealPlacement dealPlacement,
                                                      Map<Long, Placement> placementsMap) {
        if (dealPlacement == null) {
            return null;
        }

        Placement placement = placementsMap.get(dealPlacement.getPageId());
        if (placement == null) {
            return null;
        }

        Set<String> formats = DealService.extractFormatsFromPlacement(placement, dealPlacement);

        return new GdiDealPlacement()
                .withPageId(dealPlacement.getPageId())
                .withDomain(placement.getDomain())
                .withImpId(dealPlacement.getImpId())
                .withFormats(formats);
    }
}
