package ru.yandex.direct.grid.core.entity.deal.service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.deal.container.DealStat;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.repository.DealRepository;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.core.entity.placements.model.Placement;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDeal;

import static ru.yandex.direct.grid.core.entity.deal.service.GdiDealConverter.addStatToGdiDeal;
import static ru.yandex.direct.grid.core.entity.deal.service.GdiDealConverter.dealToGdiDeal;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для получения данных и статистики сделок
 */
@Service
@ParametersAreNonnullByDefault
public class GridDealService {

    private final DealRepository dealRepository;
    private final DealService dealService;

    @Autowired
    public GridDealService(DealRepository dealRepository,
                           DealService dealService) {
        this.dealRepository = dealRepository;
        this.dealService = dealService;
    }

    /**
     * Получить данные о сделках
     *
     * @param shard     шард
     * @param clientId  ID клиента
     * @param needStats нужна ли статистика
     */
    public List<GdiDeal> getDeals(int shard, ClientId clientId, boolean needStats) {
        List<Deal> deals = dealRepository.getDealsBriefByClientId(shard, clientId);
        List<Long> dealsIds = mapList(deals, Deal::getId);
        Map<Long, Placement> placementsMap = dealService.getPlacementsMapByDeals(deals);
        List<GdiDeal> gdiDeals = mapList(deals, deal -> dealToGdiDeal(deal, placementsMap));

        if (needStats && !dealsIds.isEmpty()) {
            Collection<DealStat> dealStats = dealService.getDealStatistics(dealsIds);
            Map<Long, DealStat> dealsStatById = StreamEx.of(dealStats)
                    .mapToEntry(DealStat::getId)
                    .invert()
                    .toMap();
            StreamEx.of(gdiDeals)
                    .filter(deal -> dealsStatById.containsKey(deal.getId()))
                    .forEach(deal -> addStatToGdiDeal(deal, dealsStatById.get(deal.getId())));
        }

        Map<Long, List<Long>> linkedCampaignsByDealIds = dealService.getLinkedCampaignsByDealIds(dealsIds);
        gdiDeals.forEach(deal -> deal.withNumberOfLinkedCampaigns(
                linkedCampaignsByDealIds.getOrDefault(deal.getId(), Collections.emptyList()).size()));

        return gdiDeals;
    }
}
