package ru.yandex.direct.grid.core.entity.dynamiccondition;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Row3;
import org.jooq.SelectHavingStep;
import org.jooq.types.ULong;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.core.entity.dynamiccondition.model.GdiDynamicCondition;
import ru.yandex.direct.grid.core.entity.dynamiccondition.model.GdiDynamicTargetStats;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTPHRASESTATV2_BS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
@ParametersAreNonnullByDefault
public class GridDynamicTargetYtRepository {
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);
    private static final Directphrasestatv2Bs STAT = DIRECTPHRASESTATV2_BS.as("S");
    private static final Field<Long> CAMPAIGN_ID = aliased(STAT.EXPORT_ID);
    private static final Field<Long> AD_GROUP_ID = aliased(STAT.GROUP_EXPORT_ID);
    private static final Field<ULong> PHRASE_ID = aliased(STAT.PHRASE_EXPORT_ID);
    private static final String PHRASE_ID_FIELD_NAME = PHRASE_ID.getName();

    private final YtDynamicSupport ytSupport;

    public GridDynamicTargetYtRepository(YtDynamicSupport ytSupport) {
        this.ytSupport = ytSupport;
    }

    public Map<Long, GdiDynamicTargetStats> getStatistic(List<GdiDynamicCondition> dynamicTargets,
                                                         LocalDate statDateFrom, LocalDate statDateTo,
                                                         Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        List<Row3<Long, Long, ULong>> idFilter =
                mapList(dynamicTargets,
                        r -> row(r.getCampaignId(), r.getAdGroupId(), ULong.valueOf(r.getDynamicConditionId())));
        Condition idFilterCondition = row(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID).in(idFilter);

        SelectHavingStep<Record> selectStep = gridStat.constructStatSelectWithPhraseGoals(
                List.of(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID),
                idFilterCondition,
                statDateFrom, statDateTo, goalIds, goalIdsForRevenue
        );

        UnversionedRowset unversionedRowset = ytSupport.selectRows(selectStep);
        List<YTreeMapNode> yTreeRows = unversionedRowset.getYTreeRows();
        Map<Long, GdiDynamicTargetStats> result = new HashMap<>(yTreeRows.size());
        for (YTreeMapNode node : yTreeRows) {
            long conditionId = node.getOrThrow(PHRASE_ID_FIELD_NAME).longValue();

            GdiEntityStats gdiEntityStats = gridStat.extractStatsEntry(node);
            gdiEntityStats = GridStatNew.addZeros(gdiEntityStats);
            List<GdiGoalStats> gdiGoalStats = gridStat.extractGoalStatEntries(node, goalIds);
            GdiDynamicTargetStats gdiDynamicTargetStats = new GdiDynamicTargetStats()
                    .withStat(gdiEntityStats)
                    .withGoalStats(gdiGoalStats);

            result.put(conditionId, gdiDynamicTargetStats);
        }
        return result;
    }
}
