package ru.yandex.direct.grid.core.entity.group.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.gson.Gson;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Row2;
import org.jooq.Select;
import org.jooq.SelectSelectStep;
import org.jooq.SelectWindowStep;
import org.jooq.TableField;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.AdGroupFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupFilter;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupOrderBy;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupOrderByField;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupsWithTotals;
import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendationSort;
import ru.yandex.direct.grid.core.entity.showcondition.model.GridShowConditionType;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.stats.goalstat.DirectGoalPhraseStatData;
import ru.yandex.direct.grid.core.util.stats_from_query.WithTotalsUtils;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.grid.schema.yt.tables.PhrasestableDirect;
import ru.yandex.direct.grid.schema.yt.tables.records.PhrasestableDirectRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromStringFieldToEnum;
import static ru.yandex.direct.dbschema.ppc.enums.AdgroupsCpmBannerCriterionType.keyword;
import static ru.yandex.direct.dbschema.ppc.enums.AdgroupsCpmBannerCriterionType.user_profile;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.base;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.cpm_indoor;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.cpm_outdoor;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.cpm_video;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.dynamic;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.mcbanner;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.mobile_content;
import static ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType.performance;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetNumericSubstring;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.not;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCase;
import static ru.yandex.direct.grid.schema.yt.Tables.PHRASESTABLE_DIRECT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для получения данных и статистики групп баннеров из YT
 */
@Repository
@ParametersAreNonnullByDefault
public class GridAdGroupYtRepository {
    private static final PhrasestableDirect PHRASES = PHRASESTABLE_DIRECT.as("P");
    private static final List<TableField<?, ?>> DEFAULT_ORDER = ImmutableList.of(PHRASES.CID, PHRASES.PID);
    private static final Gson GSON = new Gson();

    private static final Map<GdiGroupOrderByField, Field> RECOMMENDATION_SORTING_FIELDS = ImmutableMap.of(
            GdiGroupOrderByField.CLICKS_FORECAST, DSL.field("clicksForecast"),
            GdiGroupOrderByField.COST_FORECAST, DSL.field("costForecast")
    );

    private static final Map<GdiGroupOrderByField, Function<GdiRecommendationSort, BigDecimal>>
            RECOMMENDATION_SORTING_METHODS = ImmutableMap.of(
            GdiGroupOrderByField.CLICKS_FORECAST, GdiRecommendationSort::getClicks,
            GdiGroupOrderByField.COST_FORECAST, GdiRecommendationSort::getCost
    );

    private static final Map<GridShowConditionType, Condition> QUERY_CONDITION_BY_SHOW_CONDITION_TYPES
            = ImmutableMap.<GridShowConditionType, Condition>builder()
            .put(GridShowConditionType.KEYWORD,
                    PHRASES.ADGROUP_TYPE.in(base.getLiteral(), mobile_content.getLiteral(), mcbanner.getLiteral())
                            .or(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.cpm_banner.getLiteral())
                                    .and(PHRASES.CRITERION_TYPE.eq(keyword.getLiteral()))))
            .put(GridShowConditionType.USER_PROFILE,
                    PHRASES.ADGROUP_TYPE.in(cpm_video.getLiteral(), cpm_indoor.getLiteral(), cpm_outdoor.getLiteral())
                            .or(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.cpm_banner.getLiteral())
                                    .and(PHRASES.CRITERION_TYPE.eq(user_profile.getLiteral()))))
            .put(GridShowConditionType.RELEVANCE_MATCH,
                    PHRASES.ADGROUP_TYPE.in(base.getLiteral(), mobile_content.getLiteral()))
            .put(GridShowConditionType.RETARGETING,
                    PHRASES.ADGROUP_TYPE.in(base.getLiteral(), mobile_content.getLiteral()))
            .put(GridShowConditionType.MOBILE_INTERESTS,
                    PHRASES.ADGROUP_TYPE.in(base.getLiteral(), mobile_content.getLiteral()))
            .put(GridShowConditionType.DYNAMIC_TARGETING,
                    PHRASES.ADGROUP_TYPE.in(dynamic.getLiteral()))
            .put(GridShowConditionType.FILTERS,
                    PHRASES.ADGROUP_TYPE.in(dynamic.getLiteral(), performance.getLiteral()))
            .build();

    private static final JooqFilterProcessor<GdiGroupFilter> FILTER_PROCESSOR =
            JooqFilterProcessor.<GdiGroupFilter>builder()
                    .withFilter(GdiGroupFilter::getCampaignIdIn, PHRASES.CID::in)
                    .withFilter(GdiGroupFilter::getGroupIdIn, PHRASES.PID::in)
                    .withFilter(GdiGroupFilter::getGroupIdNotIn, not(PHRASES.PID::in))
                    .withFilter(GdiGroupFilter::getGroupIdContainsAny, inSetNumericSubstring(PHRASES.PID))
                    //Фильтр на рекомендации - тот же фильтр на идентификаторы групп. Но их много (до 20 тысяч)
                    //и yt работает, только когда задается детальное условие на префикс индекса (cid, pid)
                    .withFilter(GdiGroupFilter::getRecommendations, recommendations -> {
                        List<Row2<Long, Long>> rows = mapList(recommendations,
                                r -> row(r.getCid(), r.getPid()));
                        return row(PHRASES.CID, PHRASES.PID).in(rows);
                    })
                    .withFilter(GdiGroupFilter::getTypeIn,
                            types -> PHRASES.ADGROUP_TYPE.in(mapList(types, t -> Objects
                                    .requireNonNull(AdGroupType.toSource(t)).getLiteral())))
                    .withFilter(GdiGroupFilter::getNameContains, substringIgnoreCase(PHRASES.GROUP_NAME))
                    .withFilter(GdiGroupFilter::getNameIn, inSetIgnoreCase(PHRASES.GROUP_NAME))
                    .withFilter(GdiGroupFilter::getNameNotContains, not(substringIgnoreCase(PHRASES.GROUP_NAME)))
                    .withFilter(GdiGroupFilter::getNameNotIn, not(inSetIgnoreCase(PHRASES.GROUP_NAME)))
                    .withFilter(GdiGroupFilter::getShowConditionTypeIn,
                            GridAdGroupYtRepository::createQueryConditionFromShowCondtionTypes)
                    .build();

    private static Condition createQueryConditionFromShowCondtionTypes(Set<GridShowConditionType> showConditionTypes) {
        return StreamEx.of(showConditionTypes)
                .map(QUERY_CONDITION_BY_SHOW_CONDITION_TYPES::get)
                .nonNull()
                .foldLeft(Condition::or)
                .orElse(null);
    }

    private final YtDynamicSupport ytSupport;
    private final YtFieldMapper<GdiGroup, PhrasestableDirectRecord> groupsMapper;
    private final List<OrderField<?>> defaultOrderBy;
    private final Map<GdiGroupOrderByField, Field<?>> sortingMap;
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);

    @Autowired
    public GridAdGroupYtRepository(YtDynamicSupport gridYtSupport) {
        this.ytSupport = gridYtSupport;

        JooqReaderWithSupplier<GdiGroup> internalReader =
                JooqReaderWithSupplierBuilder.builder(GdiGroup::new)
                        .readProperty(GdiGroup.ID, fromField(PHRASES.PID))
                        .readProperty(GdiGroup.CAMPAIGN_ID, fromField(PHRASES.CID))
                        .readPropertyForFirst(GdiGroup.TYPE, fromField(PHRASES.ADGROUP_TYPE)
                                .by(GridAdGroupYtRepository::exportType))
                        .readPropertyForFirst(GdiGroup.NAME, fromField(PHRASES.GROUP_NAME))
                        .readPropertyForFirst(GdiGroup.CRITERION_TYPE, fromStringFieldToEnum(
                                PHRASES.CRITERION_TYPE, CriterionType.class, CriterionType.USER_PROFILE))
                        .readPropertyForFirst(GdiGroup.STATUS_MODERATE,
                                fromStringFieldToEnum(PHRASES.STATUS_MODERATE, StatusModerate.class))
                        .readPropertyForFirst(GdiGroup.STATUS_POST_MODERATE,
                                fromStringFieldToEnum(PHRASES.STATUS_POST_MODERATE, StatusPostModerate.class))
                        .build();

        groupsMapper = new YtFieldMapper<>(internalReader, PHRASES);
        defaultOrderBy = mapList(DEFAULT_ORDER, groupsMapper::fieldAlias);

        sortingMap = new HashMap<>();
        sortingMap.put(GdiGroupOrderByField.CAMPAIGN_ID, groupsMapper.fieldAlias(PHRASES.CID));
        sortingMap.put(GdiGroupOrderByField.ID, groupsMapper.fieldAlias(PHRASES.PID));
        sortingMap.put(GdiGroupOrderByField.NAME, groupsMapper.fieldAlias(PHRASES.GROUP_NAME));
        sortingMap.put(GdiGroupOrderByField.TYPE, groupsMapper.fieldAlias(PHRASES.ADGROUP_TYPE));

        gridStat.addStatOrderingToMap(sortingMap, GdiGroupOrderByField.class);

        for (Map.Entry<GdiGroupOrderByField, Field> e : RECOMMENDATION_SORTING_FIELDS.entrySet()) {
            sortingMap.put(e.getKey(), e.getValue());
        }
    }

    private static AdGroupType exportType(@Nullable String value) {
        if (value == null) {
            return null;
        }
        return AdGroupType.fromSource(PhrasesAdgroupType.valueOf(value));
    }

    /**
     * Получить данные о группах баннеров
     *
     * @param shard                        шард, в котором хранятся группы (исключительно для улучшения результатов
     *                                     запроса)
     * @param filter                       фильтр для выборки данных
     * @param groupOrderByList             список полей с порядком сортировки данных
     * @param statFrom                     начала периода, за который нужно получать статистику
     * @param statTo                       конец периода, за который нужно получать статистику (включительно)
     * @param limitOffset                  верхний предел количества полученных групп и смещение относительно начала
     *                                     выборки
     * @param goalIds                      идентификаторы целей
     * @param goalIdsForRevenue            идентификаторы целей, по которым считается доход
     * @param adGroupFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля запрошены на
     *                                     верхнем уровне
     * @param addWithTotalsToQuery         добавляет в запросе with totals с получением общей сатистики
     */
    public GdiGroupsWithTotals getGroups(int shard, GdiGroupFilter filter,
                                         List<GdiGroupOrderBy> groupOrderByList,
                                         LocalDate statFrom, LocalDate statTo, LimitOffset limitOffset,
                                         Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue,
                                         AdGroupFetchedFieldsResolver adGroupFetchedFieldsResolver,
                                         boolean addWithTotalsToQuery) {
        if (CollectionUtils.isAllEmpty(filter.getCampaignIdIn(), filter.getGroupIdIn(), filter.getGroupIdNotIn(),
                filter.getRecommendations())) {
            return new GdiGroupsWithTotals()
                    .withGdiGroups(Collections.emptyList());
        }

        Directphrasestatv2Bs statTable = gridStat.getTableData().table();
        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(groupsMapper.getFieldsToRead())
                .select(groupsMapper.getFieldsToReadForFirst());

        //Читаем статистику, только если она запрошена или есть фильтры по ней
        boolean hasStatFieldsSort = gridStat.hasStatFieldsSort(groupOrderByList);
        boolean withStatistic = adGroupFetchedFieldsResolver.getStats() || filter.getStats() != null
                || filter.getGoalStats() != null || hasStatFieldsSort;
        if (withStatistic) {

            TableOnConditionStep<Record> joinStep =
                    gridStat.joinStatColumns(PHRASES, row(PHRASES.CID, PHRASES.PID)
                                    .eq(statTable.EXPORT_ID, statTable.GROUP_EXPORT_ID),
                            statFrom, statTo, null);
            joinStep = gridStat.joinGoalStatColumns(joinStep, goalIds);

            selectStep.select(gridStat.getStatSelectFields());
            selectStep.select(gridStat.getGoalStatFields(goalIds, goalIdsForRevenue));
            selectStep.from(joinStep);
        } else {
            selectStep.from(PHRASES);
        }

        Condition baseCondition = FILTER_PROCESSOR.apply(filter);
        // Группируем по всем выбираемым полям, так как они не аггрегируемые
        List<Field<?>> groupBy = new ArrayList<>(groupsMapper.getFieldsToRead());
        // И добавляем ключи для ускорения
        groupBy.add(PHRASES.__HASH__);
        groupBy.add(PHRASES.__SHARD__);

        // Сортировка по полям из рекомендаций применяется только с фильтром на рекомендации
        // Поэтому берем рекомендации из фильтра
        Set<GdiGroupOrderByField> recommendationOrderSet = listToSet(groupOrderByList, GdiGroupOrderBy::getField);
        recommendationOrderSet.retainAll(RECOMMENDATION_SORTING_FIELDS.keySet());
        if (!recommendationOrderSet.isEmpty()) {
            List<GdiRecommendationSort> sortList = filter.getRecommendations().stream()
                    .map(r -> GSON.fromJson(r.getKpi(), GdiRecommendationSort.class).withId(r.getPid()))
                    .collect(Collectors.toList());

            for (GdiGroupOrderByField sortingField : recommendationOrderSet) {
                Field field = RECOMMENDATION_SORTING_FIELDS.get(sortingField);
                Map<Long, BigDecimal> idToSortMap = listToMap(sortList, GdiRecommendationSort::getId,
                        RECOMMENDATION_SORTING_METHODS.get(sortingField));
                selectStep.select(YtDSL.transform(PHRASES.PID, idToSortMap, BigDecimal.class).as(field.getName()));
                groupBy.add(field);
            }
        }

        selectStep.where(baseCondition)
                .groupBy(groupBy);

        SelectWindowStep<Record> windowStep = selectStep;

        Condition havingCondition = gridStat.getStatHavingCondition(filter.getStats());
        if (filter.getGoalStats() != null) {
            havingCondition = gridStat.addGoalStatHavingCondition(havingCondition, filter.getGoalStats());
        }

        if (havingCondition != null) {
            windowStep = selectStep.having(havingCondition);
        }

        List<OrderField<?>> orderFields = gridStat.getOrderFields(sortingMap, groupOrderByList, defaultOrderBy);

        Select query = windowStep
                .orderBy(orderFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        List<GdiGroup> groups = mapList(ytSupport
                        .selectRows(shard, query, addWithTotalsToQuery && withStatistic).getYTreeRows(),
                (n -> groupsMapper.fromNode(n)
                        .withStat(GridStatNew.addZeros(goalIdsForRevenue != null ?
                                gridStat.extractStatsEntryWithGoalsRevenue(n, goalIds, goalIdsForRevenue) :
                                gridStat.extractStatsEntry(n)))
                        .withGoalStats(gridStat.extractGoalStatEntries(n, goalIds))));

        // Показываем полученные totals из запроса только если групп = лимиту (groups.size == limit + total row)
        boolean withTotalStats = WithTotalsUtils
                .withTotalStats(addWithTotalsToQuery, withStatistic, groups, limitOffset);

        GdiGroup groupWithTotals = !withTotalStats ? null : StreamEx.of(groups)
                .findFirst(group -> Objects.isNull(group.getId()))
                .orElse(null);

        return new GdiGroupsWithTotals()
                .withGdiGroups(StreamEx.of(groups).filter(group -> Objects.nonNull(group.getId())).toList())
                .withTotalStats(groupWithTotals != null ? groupWithTotals.getStat() : null);
    }

    public Map<Long, GdiEntityConversion> getGroupsStatisticByUsedGoals(Collection<Long> campaignIds,
                                                                        LocalDate startDate, LocalDate endDate,
                                                                        @Nullable Set<Long> availableGoalIds,
                                                                        boolean getRevenueOnlyByAvailableGoals) {
        var select = gridStat.constructConversionsSelect(campaignIds, null, startDate, endDate,
                availableGoalIds, null, getRevenueOnlyByAvailableGoals, false, false);
        var statTable = (DirectGoalPhraseStatData) gridStat.getTableData().goalTableData();

        return StreamEx.of(ytSupport.selectRows(select).getYTreeRows())
                .mapToEntry(
                        node -> node.getOrThrow(statTable.groupId().getName()).longValue(),
                        gridStat::extractEntityConversionWithRevenue
                ).nonNullValues()
                .toMap(gridStat::mergeEntityConversion);
    }
}
