package ru.yandex.direct.grid.core.entity.group.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStates;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProviderFactory;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.AdGroupAdditionalTargetingService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.bidmodifier.container.MultipliersBounds;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerLogo;
import ru.yandex.direct.grid.core.entity.banner.repository.GridImageRepository;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.AdGroupFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.group.container.GdiAdGroupRegionsInfo;
import ru.yandex.direct.grid.core.entity.group.container.GdiGroupRelevanceMatch;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupFilter;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupOrderBy;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupPrimaryStatus;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupsWithTotals;
import ru.yandex.direct.grid.core.entity.group.model.GdiInternalGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiMinusKeywordsPackInfo;
import ru.yandex.direct.grid.core.entity.group.model.GdiPerformanceGroup;
import ru.yandex.direct.grid.core.entity.group.model.additionaltargeting.GdiAdditionalTargeting;
import ru.yandex.direct.grid.core.entity.group.repository.GridAdGroupYtRepository;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionRuleItem;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupMapper.toGdiRelevanceMatchCategories;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupUtils.getAdGroupImplementations;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupUtils.getGroupStatus;
import static ru.yandex.direct.grid.core.util.stats.GridStatUtils.updateStatWithConversions;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

/**
 * Сервис для получения данных и статистики групп баннеров
 */
@Service
@ParametersAreNonnullByDefault
public class GridAdGroupService {
    private final GridAdGroupYtRepository gridAdGroupYtRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final AdGroupService adGroupService;
    private final ClientGeoService clientGeoService;
    private final GeoTreeFactory geoTreeFactory;
    private final BidModifierService bidModifierService;
    private final GridRecommendationService gridRecommendationService;
    private final GridImageRepository gridImageRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerService bannerService;
    private final AdGroupAdditionalTargetingService adGroupAdditionalTargetingService;
    private final AdGroupGeoTreeProviderFactory geoTreeProviderFactory;
    private final FeatureService featureService;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final MetrikaGoalsService metrikaGoalsService;

    @Autowired
    public GridAdGroupService(GridAdGroupYtRepository gridAdGroupYtRepository,
                              AdGroupRepository adGroupRepository,
                              CampaignRepository campaignRepository,
                              MinusKeywordsPackRepository minusKeywordsPackRepository,
                              AdGroupService adGroupService,
                              BidModifierService bidModifierService,
                              ClientGeoService clientGeoService,
                              GeoTreeFactory geoTreeFactory,
                              GridRecommendationService gridRecommendationService,
                              GridImageRepository gridImageRepository,
                              BannerTypedRepository bannerTypedRepository,
                              BannerService bannerService,
                              AdGroupAdditionalTargetingService adGroupAdditionalTargetingService,
                              AdGroupGeoTreeProviderFactory geoTreeProviderFactory,
                              FeatureService featureService,
                              CryptaSegmentRepository cryptaSegmentRepository,
                              RelevanceMatchRepository relevanceMatchRepository,
                              MetrikaGoalsService metrikaGoalsService) {
        this.gridAdGroupYtRepository = gridAdGroupYtRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.adGroupService = adGroupService;
        this.bidModifierService = bidModifierService;
        this.clientGeoService = clientGeoService;
        this.geoTreeFactory = geoTreeFactory;
        this.gridRecommendationService = gridRecommendationService;
        this.gridImageRepository = gridImageRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerService = bannerService;
        this.adGroupAdditionalTargetingService = adGroupAdditionalTargetingService;
        this.geoTreeProviderFactory = geoTreeProviderFactory;
        this.featureService = featureService;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.metrikaGoalsService = metrikaGoalsService;
    }

    /**
     * Получить данные о группах баннеров
     * <p>
     * При добавлении новых фильтров в коде нужно так же их учесть в методе:
     * {@link ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter#hasAnyCodeFilter}
     *
     * @param shard                        шард, в котором хранятся группы (исключительно для улучшения результатов
     *                                     запроса)
     * @param clientCountryRegionId        идентификатор страны клиента
     * @param clientId                     идентификатор клиента
     * @param operator                     идентификатор оператора
     * @param filter                       настройки фильтрации выбранных групп
     * @param recommendationTypes          типы рекомендаций
     * @param adGroupOrderByList           настройки упорядочивания групп
     * @param statRequirements             начало периода, за который мы получаем статистику по группам
     * @param goalIds                      идентификаторы целей
     * @param adGroupFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля запрошены на
     */
    public GdiGroupsWithTotals getGroups(int shard, long clientCountryRegionId, ClientId clientId, User operator,
                                         GdiGroupFilter filter,
                                         Set<GdiRecommendationType> recommendationTypes,
                                         List<GdiGroupOrderBy> adGroupOrderByList,
                                         GdStatRequirements statRequirements,
                                         boolean withFilterByAggrStatus,
                                         Set<Long> goalIds, AdGroupFetchedFieldsResolver adGroupFetchedFieldsResolver) {

        // подкорректируем фильтр по adGroupIds, если задан фильтр по библиотечным минус-фразам
        updateFilterIfMwFilterSpecified(shard, clientId, filter);


        Map<Long, List<GdiRecommendation>> groupIdToRecommendations = emptyMap();

        // При выборе групп стоит ограничение в GridAdGroupService.MAX_GROUP_ROWS и могут быть выбраны группы
        // без рекомендаций.
        // Поэтому сначала выбираем рекомендации, если они требуются в условии
        // и затем группы с фильтром на cid, pid из рекомендаций
        if (!recommendationTypes.isEmpty()) {
            groupIdToRecommendations
                    = getRecommendations(clientId, operator, recommendationTypes, filter.getCampaignIdIn());

            //Корректируем фильтры для ускорения, все что возможно переносим в рекомендации
            List<GdiRecommendation> filterRecommendations = flatMap(groupIdToRecommendations.values(), identity())
                    .stream()
                    .filter(r -> filter.getGroupIdIn() == null || filter.getGroupIdIn().contains(r.getPid()))
                    .filter(r -> filter.getGroupIdNotIn() == null || !filter.getGroupIdNotIn().contains(r.getPid()))
                    .collect(toList());
            filter.setRecommendations(filterRecommendations);

            filter.setCampaignIdIn(null);
            filter.setGroupIdIn(null);
            filter.setGroupIdNotIn(null);
        }

        boolean addWithTotalsToQuery = featureService
                .isEnabledForClientId(clientId, FeatureName.ADD_WITH_TOTALS_TO_GROUP_QUERY);

        boolean getRevenueOnlyByAvailableGoals = featureService
                .isEnabledForClientId(clientId, FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        Set<Long> availableGoalIds = null;
        if (getRevenueOnlyByAvailableGoals) {
            availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                    operator.getUid(), clientId);
        }

        GdiGroupsWithTotals adGroupsWithTotals = gridAdGroupYtRepository
                .getGroups(shard, filter, adGroupOrderByList, statRequirements.getFrom(), statRequirements.getTo(),
                        limited(GridAdGroupConstants.getMaxGroupRows()), goalIds, availableGoalIds,
                        adGroupFetchedFieldsResolver, addWithTotalsToQuery);
        List<GdiGroup> adGroups = adGroupsWithTotals.getGdiGroups();
        if (adGroups.isEmpty()) {
            return adGroupsWithTotals;
        }

        Set<Long> adGroupIds = listToSet(adGroups, GdiGroup::getId);
        Set<Long> campaignIds = listToSet(adGroups, GdiGroup::getCampaignId);

        boolean isNewGroupStatisticEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.NEW_GROUP_STATISTIC_ENABLED);
        if (isNewGroupStatisticEnabled && Boolean.TRUE.equals(statRequirements.getUseCampaignGoalIds())) {
            var conversionStats = gridAdGroupYtRepository
                    .getGroupsStatisticByUsedGoals(campaignIds, statRequirements.getFrom(), statRequirements.getTo(),
                            availableGoalIds, getRevenueOnlyByAvailableGoals);
            adGroups.forEach(group -> updateStatWithConversions(group.getStat(), conversionStats.get(group.getId())));
        }

        enrichGdiGroupStatuses(shard, adGroupIds, adGroups);

        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        GeoTree clientGeoTree = geoTreeFactory.getTranslocalGeoTree(clientCountryRegionId);
        AdGroupGeoTreeProvider geoTreeProvider = geoTreeProviderFactory.create(clientGeoTree, campaignTypes);

        List<AdGroup> coreAdGroups = getAdGroupsFromMysql(shard, adGroupIds);
        Map<Long, AdGroup> coreAdGroupsByIds = listToMap(coreAdGroups, AdGroup::getId);

        /*
         * Может быть несколько случаев, когда выгрузили группу из YT'я, но в MySQL не нашли:
         * 1. Группу удалили в MySQL, но до YT'я изменения еще не доехали
         * 2. Оператор запрашивает группу чужого клиента, у которого шард отличается от клиента для которого
         * выгружаются данные
         * Такое может быть, если оператор фильтровал по id групп и потом в урле сменил ulogin
         * 3. {добавьте свой кейс}
         */
        adGroups = filterList(adGroups, gdiGroup -> coreAdGroupsByIds.containsKey(gdiGroup.getId()));

        Map<Long, Long> adGroupIdToCampaignId = listToMap(adGroups, GdiGroup::getId, GdiGroup::getCampaignId);
        Map<Long, MultipliersBounds> adGroupsMultipliersBounds = bidModifierService
                .calculateMultipliersBoundsForAdGroups(clientId, operator.getUid(), adGroupIdToCampaignId);

        //Если для фильтра рекомендации не требовались, то выбираем их здесь для заполнения только по нужным кампаниям
        if (adGroupFetchedFieldsResolver.getRecommendations() && recommendationTypes.isEmpty()) {
            groupIdToRecommendations = getRecommendations(clientId, operator, recommendationTypes, campaignIds);
        }
        Map<Long, List<GdiRecommendation>> finalGroupIdToRecommendations = groupIdToRecommendations;

        Map<Long, GdiMinusKeywordsPackInfo> minusKeywordsPacks =
                getLinkedLibraryMinusKeywordsPacks(shard, clientId, coreAdGroups);

        Map<Long, BannerWithSystemFields> mainAds = getMainAds(clientId, adGroupIds);

        Map<Long, RelevanceMatch> relevanceMatches = relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard,
                clientId, adGroupIds, true);

        adGroups.forEach(gdiGroup -> {
            AdGroup coreAdGroup = coreAdGroupsByIds.get(gdiGroup.getId());

            gdiGroup
                    .withRegionsInfo(getRegionsInfo(coreAdGroup, geoTreeProvider))
                    .withHyperGeoId(coreAdGroup.getHyperGeoId())
                    .withMinusKeywords(coreAdGroup.getMinusKeywords())
                    .withRecommendations(finalGroupIdToRecommendations.getOrDefault(gdiGroup.getId(), emptyList()))
                    .withMultipliersBounds(adGroupsMultipliersBounds.get(gdiGroup.getId()))
                    .withMainAd(mainAds.get(gdiGroup.getId()))
                    .withLibraryMinusKeywordsPacks(
                            mapList(coreAdGroup.getLibraryMinusKeywordsIds(), minusKeywordsPacks::get))
                    .withPageGroupTags(coreAdGroup.getPageGroupTags())
                    .withTargetTags(coreAdGroup.getTargetTags())
                    .withProjectParamConditions(coreAdGroup.getProjectParamConditions())
                    .withContentCategoriesRetargetingConditionRules(
                            convertContentCategoriesRules(
                                    clientId, campaignTypes.get(coreAdGroup.getCampaignId()),
                                    coreAdGroup.getContentCategoriesRetargetingConditionRules()))
                    .withRelevanceMatch(toGdiGroupRelevanceMatch(relevanceMatches.get(gdiGroup.getId())));
        });

        Set<GdiGroupPrimaryStatus> primaryStatusSet = nvl(filter.getPrimaryStatusIn(), emptySet());

        adGroups = getAdGroupImplementations(adGroups, coreAdGroups);
        enrichGdiPerformanceGroups(shard, clientId, adGroups);
        enrichInternalGdiGroupTargeting(clientId, adGroups);

        adGroups = withFilterByAggrStatus ? adGroups : adGroups.stream()
                .filter(g -> filter.getArchived() == null || filter.getArchived()
                        .equals(g.getStatus().getStateFlags().getArchived()))
                .filter(g -> primaryStatusSet.isEmpty()
                        || primaryStatusSet.contains(g.getStatus().getPrimaryStatus()))
                .collect(toList());

        return adGroupsWithTotals.withGdiGroups(adGroups);
    }

    @Nullable
    private List<GdiRetargetingConditionRuleItem> convertContentCategoriesRules(
            ClientId clientId, CampaignType campaignType, @Nullable List<Rule> rules) {

        if (rules == null) {
            return null;
        }

        if (featureService.isEnabledForClientId(clientId, FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN)
                && campaignType != CampaignType.CPM_PRICE) {
            var contentCategoriesByParentId = cryptaSegmentRepository.getContentCategories()
                    .values()
                    .stream()
                    .filter(g -> g.getParentId() != null && g.getParentId() != 0)
                    .collect(groupingBy(GoalBase::getParentId));

            for (var rule : rules) {
                // replace parents with their children
                var newGoals = new ArrayList<Goal>();
                for (var goal : rule.getGoals()) {
                    // only one child means that it is "Rest" category, equals no children at all
                    if (contentCategoriesByParentId.containsKey(goal.getId())
                            && contentCategoriesByParentId.get(goal.getId()).size() > 1) {
                        newGoals.addAll(contentCategoriesByParentId.get(goal.getId()));
                    } else {
                        newGoals.add(goal);
                    }
                }

                rule.setGoals(newGoals);
            }
        }

        return toGdiRetargetingConditionRules(rules);
    }

    private void enrichGdiPerformanceGroups(int shard, ClientId clientId, List<GdiGroup> adGroups) {
        List<GdiPerformanceGroup> performanceAdGroups = selectList(adGroups, GdiPerformanceGroup.class);
        if (performanceAdGroups.isEmpty()) {
            return;
        }

        List<PerformanceBannerMain> banners = bannerTypedRepository.getBannersByGroupIds(shard,
                listToSet(performanceAdGroups, GdiPerformanceGroup::getId), clientId, PerformanceBannerMain.class);
        Map<Long, PerformanceBannerMain> bannerByAdGroupId = listToMap(banners, PerformanceBannerMain::getAdGroupId);

        Set<String> logoImageHashes = mapAndFilterToSet(banners,
                PerformanceBannerMain::getLogoImageHash, Objects::nonNull);
        List<GdiBannerLogo> bannerLogos = gridImageRepository.getBannerLogosByHash(shard, clientId, logoImageHashes);
        Map<String, GdiBannerLogo> bannerLogoByImageHash = listToMap(bannerLogos, GdiBannerLogo::getImageHash);

        performanceAdGroups.forEach(performanceAdGroup -> {
            PerformanceBannerMain banner = bannerByAdGroupId.get(performanceAdGroup.getId());
            if (banner == null) {
                return;
            }

            performanceAdGroup.setLogo(bannerLogoByImageHash.get(banner.getLogoImageHash()));
        });
    }

    private void enrichInternalGdiGroupTargeting(ClientId clientId, List<GdiGroup> adGroups) {
        List<GdiInternalGroup> internalAdGroups = StreamEx.of(adGroups)
                .select(GdiInternalGroup.class)
                .toList();
        Set<Long> internalAdGroupIds = listToSet(internalAdGroups, GdiInternalGroup::getId);
        if (internalAdGroupIds.isEmpty()) {
            return;
        }

        Map<Long, List<AdGroupAdditionalTargeting>> targetingsByAdGroup =
                adGroupAdditionalTargetingService.getTargetingMapByAdGroupIds(clientId, internalAdGroupIds);

        StreamEx.of(internalAdGroups)
                .mapToEntry(grp -> targetingsByAdGroup.getOrDefault(grp.getId(), emptyList()))
                .forKeyValue(this::fillAdditionalTargetings);
    }

    /**
     * Записать доп. таргетинги в поля GdiInternalGroup, соотвествующие типу таргетинга
     */
    private void fillAdditionalTargetings(GdiInternalGroup grp, List<AdGroupAdditionalTargeting> targetingList) {
        List<GdiAdditionalTargeting> convertedTargetings = StreamEx.of(targetingList)
                .map(AdditionalTargetingConverters::convert)
                .nonNull()
                .toList();
        grp.withTargetings(convertedTargetings);
    }

    private List<AdGroup> getAdGroupsFromMysql(int shard, Set<Long> adGroupIds) {
        List<AdGroup> result = new ArrayList<>(adGroupIds.size());
        LimitOffset limitOffset = limited(GridAdGroupConstants.getMaxGroupRows());
        for (List<Long> adGroupIdsPartition : Iterables.partition(adGroupIds, GridAdGroupConstants.getMaxGroupRows())) {
            result.addAll(adGroupService.getAdGroupsBySelectionCriteria(shard,
                    new AdGroupsSelectionCriteria()
                            .withAdGroupIds(ImmutableSet.copyOf(adGroupIdsPartition)),
                    limitOffset,
                    true
            ));
        }

        return result;
    }

    private Map<Long, BannerWithSystemFields> getMainAds(ClientId clientId, Collection<Long> adGroupIds) {
        try (TraceProfile profile = Trace.current().profile("adGroups:getMainAds")) {
            return bannerService.getMainBannerByAdGroupIds(clientId, adGroupIds);
        }
    }

    /**
     * Обновить фильтр adGroupIds, если задан фильтр по libraryMwId либо имени набора минус фраз.
     * <p>
     * Нужен т.к. мы не можем фильтровать сразу по атрибутам наборов минус фраз в YT.
     * Поэтому сначала достаем adGroupIds из mysql, а потом фильтруем по adGroupIds в YT.
     */
    void updateFilterIfMwFilterSpecified(int shard, ClientId clientId, GdiGroupFilter filter) {
        Set<Long> existingAdGroupIdsInFilter = filter.getGroupIdIn();
        Set<Long> libraryMwIdsInFilter = filter.getLibraryMwIdIn();
        String mwPackNameContains = filter.getMwPackNameContains();
        String mwPackNameNotContains = filter.getMwPackNameNotContains();
        if (libraryMwIdsInFilter == null && mwPackNameContains == null && mwPackNameNotContains == null) {
            return;
        }

        Set<Long> newAdGroupIdsFilter =
                new HashSet<>(minusKeywordsPackRepository.getLinkedAdGroupIds(shard, clientId, libraryMwIdsInFilter,
                        mwPackNameContains, mwPackNameNotContains));

        if (existingAdGroupIdsInFilter != null) {
            newAdGroupIdsFilter.retainAll(existingAdGroupIdsInFilter);
        }
        filter.setGroupIdIn(newAdGroupIdsFilter);
    }

    /**
     * Для заданных групп достаем из mySql все привязанные библиотечные наборы минус фраз
     */
    private Map<Long, GdiMinusKeywordsPackInfo> getLinkedLibraryMinusKeywordsPacks(int shard, ClientId clientId,
                                                                                   List<AdGroup> coreAdGroups) {
        Set<Long> minusKeywordsPacksIds = nullSafetyFlatMap(coreAdGroups, AdGroup::getLibraryMinusKeywordsIds, toSet());
        return EntryStream.of(minusKeywordsPackRepository.getMinusKeywordsPacks(shard, clientId, minusKeywordsPacksIds))
                .mapValues(this::toGdiMinusKeywordsPack)
                .toMap();
    }

    /**
     * Получить ограниченное представление групп из mySql
     */
    public List<GdiGroup> getTruncatedAdGroupsFromMysql(int shard, long clientCountryRegionId,
                                                        Collection<Long> adGroupIds,
                                                        Map<Long, CampaignType> campaignTypes) {
        GeoTree clientGeoTree = geoTreeFactory.getTranslocalGeoTree(clientCountryRegionId);
        AdGroupGeoTreeProvider geoTreeProvider = geoTreeProviderFactory.create(clientGeoTree, campaignTypes);

        List<AdGroup> coreAdGroups = getAdGroupsFromMysql(shard, ImmutableSet.copyOf(adGroupIds));
        List<GdiGroup> adGroups = mapList(coreAdGroups, coreAdGroup -> toGdiGroup(coreAdGroup, geoTreeProvider));

        enrichGdiGroupStatuses(shard, adGroupIds, adGroups);

        return getAdGroupImplementations(adGroups, coreAdGroups);
    }

    private void enrichGdiGroupStatuses(int shard, Collection<Long> adGroupIds, List<GdiGroup> adGroups) {
        Map<Long, AdGroupStates> adGroupStatuses = adGroupRepository.getAdGroupStatuses(shard, adGroupIds);

        adGroups.forEach(group -> group.setStatus(getGroupStatus(group, adGroupStatuses)));
    }

    GdiAdGroupRegionsInfo getRegionsInfo(AdGroup adGroup, AdGroupGeoTreeProvider geoTreeProvider) {
        var geoTree = geoTreeProvider.getGeoTree(adGroup);
        List<Long> regionIds = clientGeoService.convertForWeb(adGroup.getGeo(), geoTree);
        List<Long> effectiveRegionIds =
                ifNotNull(adGroup.getEffectiveGeo(), geo -> clientGeoService.convertForWeb(geo, geoTree));
        List<Long> restrictedRegionIds = adGroup.getRestrictedGeo();

        return new GdiAdGroupRegionsInfo()
                .withRegionIds(regionIds)
                .withEffectiveRegionIds(effectiveRegionIds)
                .withRestrictedRegionIds(restrictedRegionIds);
    }

    private GdiGroup toGdiGroup(AdGroup coreAdGroup, AdGroupGeoTreeProvider geoTreeProvider) {
        return new GdiGroup()
                .withId(coreAdGroup.getId())
                .withCampaignId(coreAdGroup.getCampaignId())
                .withType(coreAdGroup.getType())
                .withName(coreAdGroup.getName())
                .withMinusKeywords(coreAdGroup.getMinusKeywords())
                .withRegionsInfo(getRegionsInfo(coreAdGroup, geoTreeProvider))
                .withStatusModerate(coreAdGroup.getStatusModerate())
                .withStatusPostModerate(coreAdGroup.getStatusPostModerate())
                .withPageGroupTags(coreAdGroup.getPageGroupTags())
                .withProjectParamConditions(coreAdGroup.getProjectParamConditions())
                .withTargetTags(coreAdGroup.getTargetTags());
    }

    private GdiMinusKeywordsPackInfo toGdiMinusKeywordsPack(MinusKeywordsPack minusKeywordsPack) {
        return new GdiMinusKeywordsPackInfo()
                .withId(minusKeywordsPack.getId())
                .withName(minusKeywordsPack.getName());
    }

    private Map<Long, List<GdiRecommendation>> getRecommendations(
            ClientId clientId, User operator,
            Set<GdiRecommendationType> recommendationTypes, Collection<Long> campaignIds) {
        try (TraceProfile ignore = Trace.current().profile("recommendations:service:groups")) {
            return gridRecommendationService
                    .getGroupRecommendations(clientId, operator, recommendationTypes, campaignIds);
        }
    }

    private static List<GdiRetargetingConditionRuleItem> toGdiRetargetingConditionRules(
            List<Rule> retargetingConditionRules) {
        return mapList(retargetingConditionRules, rule ->
                new GdiRetargetingConditionRuleItem()
                        .withType(rule.getType())
                        .withGoals(rule.getGoals())
                        .withInterestType(rule.getInterestType())
                        .withSectionId(rule.getSectionId()));
    }

    @Nullable
    private static GdiGroupRelevanceMatch toGdiGroupRelevanceMatch(@Nullable RelevanceMatch relevanceMatch) {
        if (relevanceMatch == null) {
            return null;
        }
        return new GdiGroupRelevanceMatch()
                .withId(relevanceMatch.getId())
                .withIsActive(!relevanceMatch.getIsDeleted())
                .withRelevanceMatchCategories(
                        toGdiRelevanceMatchCategories(relevanceMatch.getRelevanceMatchCategories()));
    }
}
