package ru.yandex.direct.grid.core.entity.group.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStates;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmAudioAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmIndoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmYndxFrontpageAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentExternalWorldMoney;
import ru.yandex.direct.core.entity.mobilecontent.model.StoreActionForPrices;
import ru.yandex.direct.grid.core.entity.group.container.GdiGroupRelevanceMatch;
import ru.yandex.direct.grid.core.entity.group.model.GdiBaseGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiContentPromotionGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmAudioGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmBannerGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmGeoPinGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmGeoproductGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmIndoorGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmOutdoorGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmVideoGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiCpmYndxFrontpageGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiDynamicGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupModerationStatus;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupPrimaryStatus;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupStatus;
import ru.yandex.direct.grid.core.entity.group.model.GdiInternalGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiMcbannerGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiMobileContentAdGroupDeviceTypeTargeting;
import ru.yandex.direct.grid.core.entity.group.model.GdiMobileContentAdGroupNetworkTargeting;
import ru.yandex.direct.grid.core.entity.group.model.GdiMobileContentAdGroupOsType;
import ru.yandex.direct.grid.core.entity.group.model.GdiMobileContentGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiMobileContentPrices;
import ru.yandex.direct.grid.core.entity.group.model.GdiPerformanceGroup;

import static ru.yandex.direct.grid.core.entity.group.model.GdiGroupBlGenerationStatus.INAPPLICABLE;
import static ru.yandex.direct.grid.core.entity.group.repository.GridAdGroupStatusPredicates.ADGROUP_PRIMARY_STATUS_PREDICATES;
import static ru.yandex.direct.grid.core.entity.group.repository.GridAdGroupStatusPredicates.STATUS_MODERATE_PREDICATES;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupMapper.toGdiContentPromotionGroupType;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupMapper.toGdiGroupBlGenerationStatus;
import static ru.yandex.direct.grid.core.entity.group.service.GridAdGroupMapper.toGdiRelevanceMatchCategories;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class GridAdGroupUtils {
    private static final Logger logger = LoggerFactory.getLogger(GridAdGroupService.class);

    static GdiGroupStatus getGroupStatus(GdiGroup adGroup, Map<Long, AdGroupStates> adGroupStatuses) {
        GdiGroupStatus status = new GdiGroupStatus()
                .withStateFlags(adGroupStatuses.getOrDefault(adGroup.getId(), getDefaultAdGroupStates()))
                .withModerationStatus(getGroupModerationStatus(adGroup));
        enrichGroupPrimaryStatus(status, adGroup.getCampaignId(), adGroup.getId());
        return status;
    }

    private static AdGroupStates getDefaultAdGroupStates() {
        return new AdGroupStates()
                .withArchived(false)
                .withActive(false)
                .withShowing(false)
                .withBsEverSynced(false)
                .withHasDraftAds(false);
    }

    private static GdiGroupRelevanceMatch getGdiGroupRelevanceMatch(DynamicAdGroup dynamicAdGroup) {
        return new GdiGroupRelevanceMatch()
                .withId(null)
                .withIsActive(true)
                .withRelevanceMatchCategories(
                        toGdiRelevanceMatchCategories(dynamicAdGroup.getRelevanceMatchCategories()));
    }

    public static GdiGroupModerationStatus getGroupModerationStatus(GdiGroup adGroup) {
        Optional<GdiGroupModerationStatus> result = STATUS_MODERATE_PREDICATES.keySet().stream()
                .filter(st -> STATUS_MODERATE_PREDICATES.get(st).test(adGroup))
                .findFirst();
        if (!result.isPresent()) {
            logger.warn("AdGroup has undefined moderation status. Returning MODERATION. cid = {}, adgroup = {}",
                    adGroup.getCampaignId(),
                    adGroup.getId());
        }
        return result.orElse(GdiGroupModerationStatus.MODERATION);
    }

    private static void enrichGroupPrimaryStatus(GdiGroupStatus gdiGroupStatus, Long campaignId, Long adGroupId) {
        Optional<GdiGroupPrimaryStatus> primaryStatus = StreamEx.of(GdiGroupPrimaryStatus.values())
                .filter(state -> ADGROUP_PRIMARY_STATUS_PREDICATES.get(state)
                        .test(gdiGroupStatus))
                .findFirst();
        if (!primaryStatus.isPresent()) {
            logger.warn("AdGroup has undefined PrimaryStatus. Returning ACTIVE. cid = {}, adGroupId = {}",
                    campaignId, adGroupId);
        }

        gdiGroupStatus.setPrimaryStatus(primaryStatus.orElse(GdiGroupPrimaryStatus.ACTIVE));
    }

    static List<GdiGroup> getAdGroupImplementations(Collection<GdiGroup> adGroups, Collection<AdGroup> coreAdGroups) {
        Map<Long, AdGroup> coreAdGroupById = listToMap(coreAdGroups, AdGroup::getId);
        return mapAndFilterList(adGroups, adGroup -> toGdiAdGroupImplementation(coreAdGroupById, adGroup),
                Objects::nonNull);
    }

    @Nullable
    private static GdiGroup toGdiAdGroupImplementation(Map<Long, AdGroup> coreAdGroupById, GdiGroup adGroup) {
        GdiGroup typedGroup;
        Long id = adGroup.getId();
        AdGroup coreAdGroup = coreAdGroupById.get(id);
        switch (adGroup.getType()) {
            case DYNAMIC:
                if (coreAdGroup instanceof DynamicTextAdGroup) {
                    DynamicTextAdGroup dynamicTextAdGroup = (DynamicTextAdGroup) coreAdGroup;
                    typedGroup = new GdiDynamicGroup()
                            .withRelevanceMatch(getGdiGroupRelevanceMatch(dynamicTextAdGroup))
                            .withMainDomain(dynamicTextAdGroup.getDomainUrl())
                            .withFieldToUseAsName(dynamicTextAdGroup.getFieldToUseAsName())
                            .withFieldToUseAsBody(dynamicTextAdGroup.getFieldToUseAsBody())
                            .withTrackingParams(dynamicTextAdGroup.getTrackingParams())
                            .withStatus(adGroup.getStatus()
                                    .withBlGenerationStatus(toGdiGroupBlGenerationStatus(
                                            dynamicTextAdGroup.getStatusBLGenerated()))
                            );
                } else if (coreAdGroup instanceof DynamicFeedAdGroup) {
                    DynamicFeedAdGroup dynamicFeedAdGroup = (DynamicFeedAdGroup) coreAdGroup;
                    typedGroup = new GdiDynamicGroup()
                            .withRelevanceMatch(getGdiGroupRelevanceMatch(dynamicFeedAdGroup))
                            .withDynamicFeedId(dynamicFeedAdGroup.getFeedId())
                            .withFieldToUseAsName(dynamicFeedAdGroup.getFieldToUseAsName())
                            .withFieldToUseAsBody(dynamicFeedAdGroup.getFieldToUseAsBody())
                            .withTrackingParams(dynamicFeedAdGroup.getTrackingParams())
                            .withStatus(adGroup.getStatus()
                                    .withBlGenerationStatus(
                                            toGdiGroupBlGenerationStatus(
                                                    dynamicFeedAdGroup.getStatusBLGenerated())));
                } else {
                    logger.error("Unknown implementation of core dynamic adgroup: {}", adGroup.getClass().getName());
                    return null;
                }
                break;
            case MOBILE_CONTENT:
                MobileContentAdGroup mobileContentAdGroup = (MobileContentAdGroup) coreAdGroup;
                typedGroup = new GdiMobileContentGroup()
                        .withStoreHref(mobileContentAdGroup.getStoreUrl())
                        .withDeviceTypeTargeting(mapSet(mobileContentAdGroup.getDeviceTypeTargeting(),
                                deviceType -> GdiMobileContentAdGroupDeviceTypeTargeting.valueOf(deviceType.name())))
                        .withNetworkTargeting(mapSet(mobileContentAdGroup.getNetworkTargeting(),
                                networkTargeting -> GdiMobileContentAdGroupNetworkTargeting.valueOf(networkTargeting.name())))
                        .withCurrentMinimalOsVersion(mobileContentAdGroup.getMinimalOperatingSystemVersion())
                        .withMinimalOsVersionFromStore(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getMinOsVersion))
                        .withStoreContentId(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getStoreContentId))
                        .withStoreCountry(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getStoreCountry))
                        .withIconHash(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getIconHash))
                        .withMobileContentName(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getName))
                        .withBundleId(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getBundleId))
                        .withOsType(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                mobileContent -> GdiMobileContentAdGroupOsType.valueOf(
                                        mobileContent.getOsType().name())))
                        .withMobileContentPrices(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                mobileContentPrices -> toGdiMobileContentPrices(mobileContentPrices.getPrices())))
                        .withRating(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                mobileContent -> mobileContent.getRating() != null ?
                                        mobileContent.getRating().doubleValue() : null))
                        .withRatingVotes(ifNotNull(mobileContentAdGroup.getMobileContent(),
                                MobileContent::getRatingVotes));
                break;
            case BASE:
                TextAdGroup textAdGroup = (TextAdGroup) coreAdGroup;
                typedGroup = new GdiBaseGroup()
                        .withFieldToUseAsName(textAdGroup.getFieldToUseAsName())
                        .withFieldToUseAsBody(textAdGroup.getFieldToUseAsBody())
                        .withCollectAudience(CollectionUtils.isNotEmpty(textAdGroup.getUsersSegments()));
                break;
            case CPM_BANNER:
                CpmBannerAdGroup cpmBannerAdGroup = (CpmBannerAdGroup) coreAdGroup;
                typedGroup = new GdiCpmBannerGroup()
                        .withPriority(cpmBannerAdGroup.getPriority())
                        .withUsersSegments(cpmBannerAdGroup.getUsersSegments())
                        .withForInBanner(cpmBannerAdGroup.getForInBanners())
                        .withCriterionType(adGroup.getCriterionType());
                break;
            case CPM_VIDEO:
                CpmVideoAdGroup cpmVideoAdGroup = (CpmVideoAdGroup) coreAdGroup;
                typedGroup = new GdiCpmVideoGroup()
                        .withUsersSegments(cpmVideoAdGroup.getUsersSegments())
                        .withCriterionType(cpmVideoAdGroup.getCriterionType())
                        .withIsNonSkippable(cpmVideoAdGroup.getIsNonSkippable())
                        .withPriority(cpmVideoAdGroup.getPriority());
                break;
            case CPM_AUDIO:
                typedGroup = new GdiCpmAudioGroup()
                        .withPriority(((CpmAudioAdGroup) coreAdGroup).getPriority())
                        .withUsersSegments(((CpmAudioAdGroup) coreAdGroup).getUsersSegments());
                break;
            case CPM_INDOOR:
                typedGroup = new GdiCpmIndoorGroup()
                        .withPageBlocks(((CpmIndoorAdGroup) coreAdGroup).getPageBlocks())
                        .withUsersSegments(((CpmIndoorAdGroup) coreAdGroup).getUsersSegments());
                break;
            case CPM_OUTDOOR:
                typedGroup = new GdiCpmOutdoorGroup()
                        .withPageBlocks(((CpmOutdoorAdGroup) coreAdGroup).getPageBlocks())
                        .withUsersSegments(((CpmOutdoorAdGroup) coreAdGroup).getUsersSegments());
                break;
            case MCBANNER:
                typedGroup = new GdiMcbannerGroup();
                break;
            case PERFORMANCE:
                PerformanceAdGroup performanceAdGroup = (PerformanceAdGroup) coreAdGroup;
                typedGroup = new GdiPerformanceGroup()
                        .withFeedId(performanceAdGroup.getFeedId())
                        .withFieldToUseAsName(performanceAdGroup.getFieldToUseAsName())
                        .withFieldToUseAsBody(performanceAdGroup.getFieldToUseAsBody())
                        .withTrackingParams(performanceAdGroup.getTrackingParams())
                        .withStatus(adGroup.getStatus()
                                .withBlGenerationStatus(
                                        toGdiGroupBlGenerationStatus(performanceAdGroup.getStatusBLGenerated()))
                        );
                break;
            case INTERNAL:
                InternalAdGroup internalAdGroup = (InternalAdGroup) coreAdGroup;
                typedGroup = new GdiInternalGroup()
                        .withLevel(internalAdGroup.getLevel())
                        .withRf(internalAdGroup.getRf())
                        .withRfReset(internalAdGroup.getRfReset())
                        .withMaxClicksCount(internalAdGroup.getMaxClicksCount())
                        .withMaxClicksPeriod(internalAdGroup.getMaxClicksPeriod())
                        .withMaxStopsCount(internalAdGroup.getMaxStopsCount())
                        .withMaxStopsPeriod(internalAdGroup.getMaxStopsPeriod())
                        .withStartTime(internalAdGroup.getStartTime())
                        .withFinishTime(internalAdGroup.getFinishTime());
                break;
            case CPM_GEOPRODUCT:
                typedGroup = new GdiCpmGeoproductGroup();
                break;
            case CPM_GEO_PIN:
                typedGroup = new GdiCpmGeoPinGroup();
                break;
            case CONTENT_PROMOTION:
                ContentPromotionAdGroup contentPromotionAdGroup = (ContentPromotionAdGroup) coreAdGroup;
                typedGroup = new GdiContentPromotionGroup()
                        .withContentPromotionGroupType(
                                toGdiContentPromotionGroupType(contentPromotionAdGroup.getContentPromotionType()));
                break;
            case CPM_YNDX_FRONTPAGE:
                CpmYndxFrontpageAdGroup cpmYndxFrontpageAdGroup = (CpmYndxFrontpageAdGroup) coreAdGroup;
                typedGroup = new GdiCpmYndxFrontpageGroup()
                        .withPriority(cpmYndxFrontpageAdGroup.getPriority());
                break;
            case CONTENT_PROMOTION_VIDEO:
            default:
                logger.error("Unsupported adGroup type: {}", adGroup.getType());
                return null;
        }

        return typedGroup
                .withId(id)
                .withCampaignId(adGroup.getCampaignId())
                .withType(adGroup.getType())
                .withName(adGroup.getName())
                .withMainAd(adGroup.getMainAd())
                .withMinusKeywords(adGroup.getMinusKeywords())
                .withLibraryMinusKeywordsPacks(adGroup.getLibraryMinusKeywordsPacks())
                .withRegionsInfo(adGroup.getRegionsInfo())
                .withHyperGeoId(adGroup.getHyperGeoId())
                .withStatus(adGroup.getStatus()
                        .withBlGenerationStatus(nvl(adGroup.getStatus().getBlGenerationStatus(), INAPPLICABLE)))
                .withStatusModerate(adGroup.getStatusModerate())
                .withStatusPostModerate(adGroup.getStatusPostModerate())
                .withMultipliersBounds(adGroup.getMultipliersBounds())
                .withRecommendations(adGroup.getRecommendations())
                .withStat(adGroup.getStat())
                .withGoalStats(adGroup.getGoalStats())
                .withPageGroupTags(adGroup.getPageGroupTags())
                .withTargetTags(adGroup.getTargetTags())
                .withProjectParamConditions(adGroup.getProjectParamConditions())
                .withContentCategoriesRetargetingConditionRules(adGroup.getContentCategoriesRetargetingConditionRules())
                .withRelevanceMatch(nullableNvl(typedGroup.getRelevanceMatch(), adGroup.getRelevanceMatch()));
    }

    @Nullable
    private static List<GdiMobileContentPrices> toGdiMobileContentPrices(
            @Nullable Map<String, Map<StoreActionForPrices, MobileContentExternalWorldMoney>> prices) {
        if (prices == null) {
            return null;
        }
        var mobileContentPrices = new ArrayList<GdiMobileContentPrices>();
        for (var pricesInfo : prices.entrySet()) {
            var country = pricesInfo.getKey();
            for (var currencyAndSumInfo : pricesInfo.getValue().entrySet()) {
                StoreActionForPrices action = currencyAndSumInfo.getKey();
                var money = currencyAndSumInfo.getValue();
                var item = new GdiMobileContentPrices()
                        .withCountry(country)
                        .withStoreActionForPrices(action)
                        .withPrice(nvl(money.getSum(), BigDecimal.ZERO))
                        .withPriceCurrency(money.getCurrency());
                mobileContentPrices.add(item);
            }
        }
        return mobileContentPrices;
    }
}
