package ru.yandex.direct.grid.core.entity.offer.repository;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Functions;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.common.util.collections.CollectionUtils;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStatsFilter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOffer;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferFilter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferOrderBy;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferOrderByField;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatUtils;
import ru.yandex.direct.grid.core.util.yt.mapping.YtReader;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.schema.yt.Tables;
import ru.yandex.direct.grid.schema.yt.tables.OfferattributesBs;
import ru.yandex.direct.grid.schema.yt.tables.OfferstatBs;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.ytcomponents.service.OfferStatDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;

import static java.util.Map.entry;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.row;
import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.val;
import static ru.yandex.direct.common.util.RepositoryUtils.nullSafeReader;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.greaterOrEqual;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.lessOrEqual;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.not;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCase;
import static ru.yandex.direct.grid.core.util.stats.GridStatNew.getPeriodField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtUtils.effectiveCampaignIdField;
import static ru.yandex.direct.ytwrapper.YtUtils.periodCondition;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.PERCENT_DECIMAL_MULT;

/**
 * Репозиторий для получения данных и статистики офферов из YT
 */
@Repository
@ParametersAreNonnullByDefault
public class GridOfferYtRepository {
    private static final OfferstatBs OFFER_STAT = Tables.OFFERSTAT_BS.as("OS");
    private static final OfferattributesBs OFFER_ATTRIBUTES = Tables.OFFERATTRIBUTES_BS.as("OA");

    private static final Field<Long> OS_ORDER_ID = OFFER_STAT.ORDER_ID.as("OS_OrderID");

    private static final Field<Long> EFFECTIVE_ORDER_ID = field("EffectiveOrderID", Long.class);
    private static final Field<Long> DATE = field("Date", Long.class);

    private static final Field<BigDecimal> SHOWS = sum(OFFER_STAT.SHOWS).as(GdiOfferStats.SHOWS.name());
    private static final Field<BigDecimal> CLICKS = sum(OFFER_STAT.CLICKS).as(GdiOfferStats.CLICKS.name());
    private static final Field<BigDecimal> CTR = YtDSL.ytIf(SHOWS.ne(BigDecimal.ZERO),
            CLICKS.mul(PERCENT_DECIMAL_MULT).div(SHOWS), val(BigDecimal.ZERO)).as(GdiOfferStats.CTR.name());
    private static final Field<BigDecimal> COST = sum(OFFER_STAT.COST_CUR_TAX_FREE).as(GdiOfferStats.COST.name());
    private static final Field<BigDecimal> COST_WITH_TAX = sum(OFFER_STAT.COST_CUR).as(GdiOfferStats.COST_WITH_TAX.name());
    private static final Field<BigDecimal> REVENUE = sum(OFFER_STAT.SUM_PURCHASE_REVENUE).as(GdiOfferStats.REVENUE.name());
    private static final Field<BigDecimal> CRR = YtDSL.ytIf(COST.ne(BigDecimal.ZERO).and(REVENUE.ne(BigDecimal.ZERO)),
            YtDSL.toDouble(COST).mul(PERCENT_DECIMAL_MULT.setScale(1, RoundingMode.HALF_UP))
                    .div(YtDSL.toDouble(REVENUE)), YtDSL.ytNull(BigDecimal.class)).as(GdiOfferStats.CRR.name());
    private static final Field<BigDecimal> CARTS = sum(OFFER_STAT.BUCKETS).as(GdiOfferStats.CARTS.name());
    private static final Field<BigDecimal> PURCHASES = sum(OFFER_STAT.PURCHASES).as(GdiOfferStats.PURCHASES.name());
    private static final Field<BigDecimal> AVG_CLICK_COST = YtDSL.ytIf(CLICKS.ne(BigDecimal.ZERO),
            COST.div(CLICKS), val(BigDecimal.ZERO)).as(GdiOfferStats.AVG_CLICK_COST.name());
    private static final Field<BigDecimal> AVG_PRODUCT_PRICE = YtDSL.ytIf(PURCHASES.ne(BigDecimal.ZERO),
            sum(OFFER_STAT.SUM_AVG_PRODUCT_PRICE).div(PURCHASES), YtDSL.ytNull(BigDecimal.class)).as(GdiOfferStats.AVG_PRODUCT_PRICE.name());
    private static final Field<BigDecimal> AVG_PURCHASE_REVENUE = YtDSL.ytIf(PURCHASES.ne(BigDecimal.ZERO),
            sum(OFFER_STAT.SUM_PURCHASE_REVENUE).div(PURCHASES), YtDSL.ytNull(BigDecimal.class)).as(GdiOfferStats.AVG_PURCHASE_REVENUE.name());
    private static final Field<BigDecimal> AUTOBUDGET_GOALS = sum(OFFER_STAT.ABCONVERSIONS).as(GdiOfferStats.AUTOBUDGET_GOALS.name());
    private static final Field<BigDecimal> MEANINGFUL_GOALS = sum(OFFER_STAT.KEY_CONVERSIONS).as(GdiOfferStats.MEANINGFUL_GOALS.name());

    private static final YtReader<GdiOfferId> OS_OFFER_ID_READER = new YtReader<>(
            JooqReaderWithSupplierBuilder.builder(GdiOfferId::new)
                    .readProperty(GdiOfferId.BUSINESS_ID, fromField(OFFER_STAT.BUSINESS_ID))
                    .readProperty(GdiOfferId.SHOP_ID, fromField(OFFER_STAT.SHOP_ID))
                    .readProperty(GdiOfferId.OFFER_YABS_ID, fromField(OFFER_STAT.OFFER_YABS_ID).by(ULong::longValue))
                    .build());

    private static final YtReader<GdiOfferStats> OS_STATS_READER = new YtReader<>(
            JooqReaderWithSupplierBuilder.builder(GdiOfferStats::new)
                    .readProperty(GdiOfferStats.SHOWS, fromField(SHOWS).by(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.CLICKS, fromField(CLICKS).by(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.CTR, fromField(CTR)
                            .by(YtMappingUtils::fromMicros).andThen(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.COST, fromField(COST)
                            .by(YtMappingUtils::fromMicros).andThen(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.COST_WITH_TAX, fromField(COST_WITH_TAX)
                            .by(YtMappingUtils::fromMicros).andThen(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.REVENUE, fromField(REVENUE).by(YtMappingUtils::fromMicros))
                    .readProperty(GdiOfferStats.CRR, fromField(CRR).by(YtMappingUtils::fromMicros))
                    .readProperty(GdiOfferStats.CARTS, fromField(CARTS))
                    .readProperty(GdiOfferStats.PURCHASES, fromField(PURCHASES))
                    .readProperty(GdiOfferStats.AVG_CLICK_COST, fromField(AVG_CLICK_COST)
                            .by(YtMappingUtils::fromMicros).andThen(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.AVG_PRODUCT_PRICE, fromField(AVG_PRODUCT_PRICE)
                            .by(YtMappingUtils::fromMicros))
                    .readProperty(GdiOfferStats.AVG_PURCHASE_REVENUE, fromField(AVG_PURCHASE_REVENUE)
                            .by(YtMappingUtils::fromMicros))
                    .readProperty(GdiOfferStats.AUTOBUDGET_GOALS, fromField(AUTOBUDGET_GOALS)
                            .by(RepositoryUtils::nullToZero))
                    .readProperty(GdiOfferStats.MEANINGFUL_GOALS, fromField(MEANINGFUL_GOALS)
                            .by(RepositoryUtils::nullToZero))
                    .build());

    private static final YtReader<GdiOffer> OA_OFFER_READER = new YtReader<>(
            JooqReaderWithSupplierBuilder.builder(GdiOffer::new)
                    .readProperty(GdiOffer.ID, JooqReaderWithSupplierBuilder.builder(GdiOfferId::new)
                            .readProperty(GdiOfferId.BUSINESS_ID, fromField(OFFER_ATTRIBUTES.BUSINESS_ID))
                            .readProperty(GdiOfferId.SHOP_ID, fromField(OFFER_ATTRIBUTES.SHOP_ID))
                            .readProperty(GdiOfferId.OFFER_YABS_ID, fromField(OFFER_ATTRIBUTES.OFFER_YABS_ID)
                                    .by(ULong::longValue))
                            .build())
                    .readPropertyForFirst(GdiOffer.URL, fromField(OFFER_ATTRIBUTES.URL))
                    .readPropertyForFirst(GdiOffer.NAME, fromField(OFFER_ATTRIBUTES.NAME))
                    .readPropertyForFirst(GdiOffer.IMAGE_URL, fromField(OFFER_ATTRIBUTES.PICTURE_URL))
                    .readPropertyForFirst(GdiOffer.PRICE, fromField(OFFER_ATTRIBUTES.PRICE)
                            .by(nullSafeReader(price -> price / 10)).andThen(YtMappingUtils::fromMicros))
                    .readPropertyForFirst(GdiOffer.CURRENCY_ISO_CODE, fromField(OFFER_ATTRIBUTES.CURRENCY_NAME))
                    .build());

    private static final JooqFilterProcessor<GdiOfferStatsFilter> STATS_FILTER_PROCESSOR =
            JooqFilterProcessor.<GdiOfferStatsFilter>builder()
                    .withFilter(GdiOfferStatsFilter::getMinShows, lessOrEqual(SHOWS))
                    .withFilter(GdiOfferStatsFilter::getMaxShows, greaterOrEqual(SHOWS))
                    .withFilter(GdiOfferStatsFilter::getMinClicks, lessOrEqual(CLICKS))
                    .withFilter(GdiOfferStatsFilter::getMaxClicks, greaterOrEqual(CLICKS))
                    .withFilter(GdiOfferStatsFilter::getMinCtr, lessOrEqual(CTR, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxCtr, greaterOrEqual(CTR, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinCost, lessOrEqual(COST, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxCost, greaterOrEqual(COST, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinCostWithTax, lessOrEqual(COST_WITH_TAX, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxCostWithTax, greaterOrEqual(COST_WITH_TAX, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinRevenue, lessOrEqual(REVENUE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxRevenue, greaterOrEqual(REVENUE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinCrr, lessOrEqual(CRR, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxCrr, greaterOrEqual(CRR, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinCarts, lessOrEqual(CARTS))
                    .withFilter(GdiOfferStatsFilter::getMaxCarts, greaterOrEqual(CARTS))
                    .withFilter(GdiOfferStatsFilter::getMinPurchases, lessOrEqual(PURCHASES))
                    .withFilter(GdiOfferStatsFilter::getMaxPurchases, greaterOrEqual(PURCHASES))
                    .withFilter(GdiOfferStatsFilter::getMinAvgClickCost, lessOrEqual(AVG_CLICK_COST, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxAvgClickCost, greaterOrEqual(AVG_CLICK_COST, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinAvgProductPrice, lessOrEqual(AVG_PRODUCT_PRICE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxAvgProductPrice, greaterOrEqual(AVG_PRODUCT_PRICE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinAvgPurchaseRevenue, lessOrEqual(AVG_PURCHASE_REVENUE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMaxAvgPurchaseRevenue, greaterOrEqual(AVG_PURCHASE_REVENUE, DECIMAL_MULT))
                    .withFilter(GdiOfferStatsFilter::getMinAutobudgetGoals, lessOrEqual(AUTOBUDGET_GOALS))
                    .withFilter(GdiOfferStatsFilter::getMaxAutobudgetGoals, greaterOrEqual(AUTOBUDGET_GOALS))
                    .withFilter(GdiOfferStatsFilter::getMinMeaningfulGoals, lessOrEqual(MEANINGFUL_GOALS))
                    .withFilter(GdiOfferStatsFilter::getMaxMeaningfulGoals, greaterOrEqual(MEANINGFUL_GOALS))
                    .build();

    // campaignIdIn и adGroupIdIn обрабатываются отдельно, т.к. они попадают в другую секцию запроса
    private static final JooqFilterProcessor<GdiOfferFilter> FILTER_PROCESSOR =
            JooqFilterProcessor.<GdiOfferFilter>builder()
                    .withFilter(GdiOfferFilter::getUrlContains,
                            substringIgnoreCase(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.URL)))
                    .withFilter(GdiOfferFilter::getUrlNotContains,
                            not(substringIgnoreCase(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.URL))))
                    .withFilter(GdiOfferFilter::getNameContains,
                            substringIgnoreCase(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.NAME)))
                    .withFilter(GdiOfferFilter::getNameNotContains,
                            not(substringIgnoreCase(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.NAME))))
                    .withFilter(GdiOfferFilter::getStats, STATS_FILTER_PROCESSOR)
                    .build();

    private static final Map<GdiOfferOrderByField, List<Field<?>>> ORDER_BY_FIELDS_MAP = Map.ofEntries(
            entry(GdiOfferOrderByField.ID, List.of(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.BUSINESS_ID),
                    OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.SHOP_ID),
                    OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.OFFER_YABS_ID))),
            entry(GdiOfferOrderByField.NAME, List.of(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.NAME))),
            entry(GdiOfferOrderByField.STAT_SHOWS, List.of(SHOWS)),
            entry(GdiOfferOrderByField.STAT_CLICKS, List.of(CLICKS)),
            entry(GdiOfferOrderByField.STAT_CTR, List.of(CTR)),
            entry(GdiOfferOrderByField.STAT_COST, List.of(COST)),
            entry(GdiOfferOrderByField.STAT_COST_WITH_TAX, List.of(COST_WITH_TAX)),
            entry(GdiOfferOrderByField.STAT_REVENUE, List.of(REVENUE)),
            entry(GdiOfferOrderByField.STAT_CRR, List.of(CRR)),
            entry(GdiOfferOrderByField.STAT_CARTS, List.of(CARTS)),
            entry(GdiOfferOrderByField.STAT_PURCHASES, List.of(PURCHASES)),
            entry(GdiOfferOrderByField.STAT_AVG_CLICK_COST, List.of(AVG_CLICK_COST)),
            entry(GdiOfferOrderByField.STAT_AVG_PRODUCT_PRICE, List.of(AVG_PRODUCT_PRICE)),
            entry(GdiOfferOrderByField.STAT_AVG_PURCHASE_REVENUE, List.of(AVG_PURCHASE_REVENUE)),
            entry(GdiOfferOrderByField.STAT_AUTOBUDGET_GOALS, List.of(AUTOBUDGET_GOALS)),
            entry(GdiOfferOrderByField.STAT_MEANINGFUL_GOALS, List.of(MEANINGFUL_GOALS))
    );

    private static final List<Field<?>> DEFAULT_ORDER_BY_FIELDS = ORDER_BY_FIELDS_MAP.get(GdiOfferOrderByField.ID);

    private final OfferStatDynContextProvider offerStatDynContextProvider;
    private final CampaignService campaignService;

    @Autowired
    public GridOfferYtRepository(OfferStatDynContextProvider offerStatDynContextProvider,
                                 CampaignService campaignService) {
        this.offerStatDynContextProvider = offerStatDynContextProvider;
        this.campaignService = campaignService;
    }

    /**
     * Получить данные об офферах
     *
     * @param filter      фильтр для выборки данных
     * @param orderByList список полей с порядком сортировки данных
     * @param statFrom    начало периода, за который нужно получать статистику
     * @param statTo      конец периода, за который нужно получать статистику (включительно)
     * @param limitOffset верхний предел количества полученных офферов и смещение относительно начала выборки
     */
    public List<GdiOffer> getOffers(GdiOfferFilter filter,
                                    List<GdiOfferOrderBy> orderByList,
                                    LocalDate statFrom, LocalDate statTo,
                                    LimitOffset limitOffset) {
        var orderIds = campaignService.getOrderIdByCampaignId(filter.getCampaignIdIn()).values();

        var orderByFields = flatMap(orderByList, orderBy -> {
            var fields = ORDER_BY_FIELDS_MAP.get(orderBy.getField());
            return orderBy.getOrder() == Order.DESC ? mapList(fields, Field::desc) : fields;
        });

        var query = YtDSL.ytContext()
                .select(OA_OFFER_READER.getFieldsToRead())
                .select(OA_OFFER_READER.getFieldsToReadForFirst())
                .select(OS_STATS_READER.getFieldsToRead())
                .from(OFFER_STAT)
                .join(OFFER_ATTRIBUTES).on(row(OFFER_STAT.BUSINESS_ID, OFFER_STAT.SHOP_ID, OFFER_STAT.OFFER_YABS_ID)
                        .eq(row(OFFER_ATTRIBUTES.BUSINESS_ID, OFFER_ATTRIBUTES.SHOP_ID, OFFER_ATTRIBUTES.OFFER_YABS_ID)))
                .where(OFFER_STAT.ORDER_ID.in(orderIds)
                        .and(nvl(ifNotNull(filter.getAdGroupIdIn(), OFFER_STAT.GROUP_EXPORT_ID::in), noCondition()))
                        .and(periodCondition(OFFER_STAT.EVENT_DATE, statFrom, statTo)))
                .groupBy(OA_OFFER_READER.getFieldsToRead())
                .having(nvl(FILTER_PROCESSOR.apply(filter), noCondition())
                        // TODO(DIRECT-166870) - убрать этот костыль
                        .andNot(YtDSL.isNull(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.NAME))
                                .and(YtDSL.isNull(OA_OFFER_READER.aliasedField(OFFER_ATTRIBUTES.URL)))))
                .orderBy(isEmpty(orderByFields) ? DEFAULT_ORDER_BY_FIELDS : orderByFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        return mapList(offerStatDynContextProvider.getContext().executeSelect(query).getYTreeRows(),
                n -> OA_OFFER_READER.fromYTreeRow(n).withStats(OS_STATS_READER.fromYTreeRow(n)));
    }

    /**
     * Получить данные о конкретных офферах (без статистики)
     *
     * @param offerIds id офферов
     */
    public Map<GdiOfferId, GdiOffer> getOfferById(Collection<GdiOfferId> offerIds) {
        var query = YtDSL.ytContext()
                .select(OA_OFFER_READER.getFieldsToRead())
                .select(OA_OFFER_READER.getFieldsToReadForFirst())
                .from(OFFER_ATTRIBUTES)
                .where(row(OFFER_ATTRIBUTES.BUSINESS_ID, OFFER_ATTRIBUTES.SHOP_ID, OFFER_ATTRIBUTES.OFFER_YABS_ID)
                        .in(mapList(offerIds, id -> row(id.getBusinessId(), id.getShopId(), ULong.valueOf(id.getOfferYabsId())))))
                .groupBy(OA_OFFER_READER.getFieldsToRead());

        return StreamEx.of(offerStatDynContextProvider.getContext().executeSelect(query).getYTreeRows())
                .map(OA_OFFER_READER::fromYTreeRow)
                .toMap(GdiOffer::getId, Functions.identity());
    }

    /**
     * Для каждого оффера получить список id заказов, в рамках которых он показывался в заданный период
     *
     * @param orderIds id возможных заказов (обязательны для оптимизации)
     * @param offerIds id офферов
     * @param statFrom начало периода
     * @param statTo   конец периода
     */
    public Map<GdiOfferId, List<Long>> getOrderIdsByOfferId(Collection<Long> orderIds, Collection<GdiOfferId> offerIds, LocalDate statFrom, LocalDate statTo) {
        var query = YtDSL.ytContext()
                .select(CollectionUtils.union(OS_OFFER_ID_READER.getFieldsToRead(), List.of(OS_ORDER_ID)))
                .from(OFFER_STAT)
                .where(OFFER_STAT.ORDER_ID.in(orderIds))
                .and(row(OFFER_STAT.BUSINESS_ID, OFFER_STAT.SHOP_ID, OFFER_STAT.OFFER_YABS_ID)
                        .in(mapList(offerIds, id -> row(id.getBusinessId(), id.getShopId(), ULong.valueOf(id.getOfferYabsId())))))
                .and(periodCondition(OFFER_STAT.EVENT_DATE, statFrom, statTo))
                .groupBy(CollectionUtils.union(OS_OFFER_ID_READER.getFieldsToRead(), List.of(OS_ORDER_ID)));

        return StreamEx.of(offerStatDynContextProvider.getContext().executeSelect(query).getYTreeRows())
                .mapToEntry(OS_OFFER_ID_READER::fromYTreeRow, r -> r.getLong(OS_ORDER_ID.getName()))
                .grouping();
    }

    /**
     * Для каждого заказа получить сгруппированную по дням, неделям или месяцам офферную статистику за заданный период
     *
     * @param orderIds        список id заказов
     * @param masterIdBySubId отображение из id подлежащих заказов в id мастер-заказов, учитывается в том числе в группировке
     * @param statFrom        начало периода, за который нужно получать статистику
     * @param statTo          конец периода, за который нужно получать статистику (включительно)
     * @param groupByDate     вид группировки
     */
    public Map<Long, Map<LocalDate, GdiOfferStats>> getOfferStatsByDateByOrderId(Collection<Long> orderIds,
                                                                                 @Nullable Map<Long, Long> masterIdBySubId,
                                                                                 LocalDate statFrom,
                                                                                 LocalDate statTo,
                                                                                 ReportOptionGroupByDate groupByDate) {
        var query = YtDSL.ytContext()
                .select(CollectionUtils.union(List.of(
                        effectiveCampaignIdField(OFFER_STAT.ORDER_ID, masterIdBySubId).as(EFFECTIVE_ORDER_ID),
                        getPeriodField(OFFER_STAT.EVENT_DATE, nvl(groupByDate, ReportOptionGroupByDate.NONE)).as(DATE)),
                        OS_STATS_READER.getFieldsToRead()))
                .from(OFFER_STAT)
                .where(OFFER_STAT.ORDER_ID.in(GridStatUtils.getAllCampaignIds(orderIds, masterIdBySubId)))
                .and(periodCondition(OFFER_STAT.EVENT_DATE, statFrom, statTo))
                .groupBy(EFFECTIVE_ORDER_ID, DATE);

        return StreamEx.of(offerStatDynContextProvider.getContext().executeSelect(query).getYTreeRows())
                .mapToEntry(r -> r.getLong(EFFECTIVE_ORDER_ID.getName()),
                        r -> Map.entry(YtMappingUtils.fromEpochSecond(r.getLong(DATE.getName())),
                                OS_STATS_READER.fromYTreeRow(r)))
                .grouping(toMap(Map.Entry::getKey, Map.Entry::getValue));
    }
}
