package ru.yandex.direct.grid.core.entity.offer.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.converter.FeedConverter;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.offer.converter.GridOfferConverter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOffer;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferFilter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferOrderBy;
import ru.yandex.direct.grid.core.entity.offer.repository.GridOfferYtRepository;
import ru.yandex.direct.grid.core.util.stats.GridStatUtils;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CollectionUtils.isAllEmpty;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

/**
 * Сервис для получения данных и статистики офферов
 */
@Service
@ParametersAreNonnullByDefault
public class GridOfferService {
    /** Повышение этого лимита может существенно замедлить запрос к YT. */
    public static final Integer MAX_OFFER_ROWS = 1_000;

    private final GridOfferYtRepository gridOfferYtRepository;
    private final FeedService feedService;
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignService campaignService;

    @Autowired
    public GridOfferService(GridOfferYtRepository gridOfferYtRepository,
                            FeedService feedService,
                            ShardHelper shardHelper,
                            AdGroupRepository adGroupRepository,
                            CampaignRepository campaignRepository,
                            CampaignService campaignService) {
        this.gridOfferYtRepository = gridOfferYtRepository;
        this.feedService = feedService;
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignService = campaignService;
    }

    /**
     * Получить все данные об офферах из YT.
     *
     * @param clientId         идентификатор клиента для проверки доступов
     * @param filter           настройки фильтрации выбранных офферов
     * @param offerOrderByList настройки упорядочивания офферов
     * @param statStartDay     начало периода, за который мы получаем статистику по офферам
     * @param statEndDay       конец периода, за который мы получаем статистику по офферам
     */
    public List<GdiOffer> getOffers(ClientId clientId, GdiOfferFilter filter, List<GdiOfferOrderBy> offerOrderByList,
                                    LocalDate statStartDay, LocalDate statEndDay) {
        // Т.к. в YT выгрузке нет никакой привязки к клиенту, аккуратно отберем принадлежащие клиенту кампании и группы,
        // чтобы случайно не отдать чужие данные. Заодно учтем подлежащие кампании.
        if (!isAllEmpty(filter.getCampaignIdIn(), filter.getAdGroupIdIn())) {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            if (!isEmpty(filter.getCampaignIdIn())) {
                var campaignIds = campaignRepository.getExistingCampaignIds(shard, clientId, filter.getCampaignIdIn());
                var masterIdBySubId = campaignRepository.getSubCampaignIdsWithMasterIds(shard, campaignIds);
                var allCampaignIds = GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId);
                filter.setCampaignIdIn(Set.copyOf(allCampaignIds));
            }
            if (!isEmpty(filter.getAdGroupIdIn())) {
                filter.setAdGroupIdIn(adGroupRepository.getClientExistingAdGroupIds(shard, clientId, filter.getAdGroupIdIn()));
            }
        }

        return gridOfferYtRepository.getOffers(filter, offerOrderByList, statStartDay, statEndDay, limited(MAX_OFFER_ROWS));
    }

    /**
     * Получить данные о конкретных офферах (без статистики) из YT.
     *
     * @param clientId идентификатор клиента для проверки доступов
     * @param offerIds id офферов
     */
    public Map<GdiOfferId, GdiOffer> getOfferById(ClientId clientId, Collection<GdiOfferId> offerIds) {
        // В YT выгрузке нет никакой явной привязки к клиенту, но каждый оффер происходит из конкретного магазина.
        // По businessId и shopId, являющимся частью id оффера, мы можем найти фиды Директа, соответствующие магазину
        // (таковых может быть несколько, если их экспортировали из Маркета).
        // Соответственно, клиенту принадлежат оффера, которые происходят из принадлежащих клиенту фидов.
        var businessIdAndShopIdByOfferId = listToMap(offerIds, Function.identity(),
                GridOfferConverter::extractBusinessIdAndShopId);
        var clientFeeds = feedService.getFeedsSimple(clientId, FeedQueryFilter.newBuilder()
                .withBusinessIdsAndShopIds(businessIdAndShopIdByOfferId.values())
                .build());
        var clientBusinessAndShopIds = listToSet(clientFeeds, FeedConverter::extractBusinessIdAndShopId);
        var clientOfferIds = EntryStream.of(businessIdAndShopIdByOfferId)
                .filterValues(clientBusinessAndShopIds::contains)
                .keys()
                .toSet();

        return gridOfferYtRepository.getOfferById(clientOfferIds);
    }

    /**
     * Для каждого оффера получить список идентификаторов кампаний, в рамках которых он показывался в заданный период.
     *
     * @param clientId     идентификатор клиента
     * @param campaignIds  идентификаторы рассматриваемых кампаний (обязательны для оптимизации)
     * @param offerIds     композитные идентификаторы офферов
     * @param statStartDay начало периода
     * @param statEndDay   конец периода
     */
    public Map<GdiOfferId, Set<Long>> getCampaignIdsByOfferId(ClientId clientId, Collection<Long> campaignIds,
                                                              Collection<GdiOfferId> offerIds,
                                                              LocalDate statStartDay, LocalDate statEndDay) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds, clientId);
        var allCampaignIds = campaignRepository.getExistingCampaignIds(shard, clientId,
                GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId));
        var orderIdByCampaignId = EntryStream.of(campaignService.getOrderIdByCampaignId(allCampaignIds))
            .filterValues(orderId -> orderId != 0L)
            .toMap();
        var orderIdsByOfferId = gridOfferYtRepository.getOrderIdsByOfferId(orderIdByCampaignId.values(),
                offerIds, statStartDay, statEndDay);
        var campaignIdByOrderId = EntryStream.of(orderIdByCampaignId)
                .invert()
                .toMap();
        return EntryStream.of(orderIdsByOfferId)
                .mapValues(orderIds -> StreamEx.of(orderIds)
                        .map(campaignIdByOrderId::get)
                        .nonNull()
                        .mapToEntry(Function.identity())
                        .mapKeyValue(masterIdBySubId::getOrDefault)
                        .toSet())
                .removeValues(CollectionUtils::isEmpty)
                .toMap();
    }

    /**
     * Для каждой кампании получить сгруппированную по дням, неделям или месяцам офферную статистику за заданный период.
     *
     * @param clientId     идентификатор клиента для проверки доступов
     * @param campaignIds  список идентификаторов кампаний
     * @param statStartDay начало периода, за который мы получаем статистику по офферам
     * @param statEndDay   конец периода, за который мы получаем статистику по офферам
     * @param groupByDate  вид группировки
     */
    public Map<Long, Map<LocalDate, GdiOfferStats>> getOfferStatsByDateByCampaignId(ClientId clientId,
                                                                                    Collection<Long> campaignIds,
                                                                                    LocalDate statStartDay,
                                                                                    LocalDate statEndDay,
                                                                                    ReportOptionGroupByDate groupByDate) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(campaignIds, clientId);
        var allCampaignIds = campaignRepository.getExistingCampaignIds(shard, clientId,
                GridStatUtils.getAllCampaignIds(campaignIds, masterIdBySubId));
        var orderIdByCampaignId = EntryStream.of(campaignService.getOrderIdByCampaignId(allCampaignIds))
            .filterValues(orderId -> orderId != 0L)
            .toMap();
        var orderIds = mapAndFilterList(campaignIds, orderIdByCampaignId::get, Objects::nonNull);
        var masterOrderIdBySubOrderId = EntryStream.of(masterIdBySubId)
                .mapKeys(orderIdByCampaignId::get)
                .nonNullKeys()
                .mapValues(orderIdByCampaignId::get)
                .nonNullValues()
                .toMap();
        var offerStatsByDayByOrderId = gridOfferYtRepository.getOfferStatsByDateByOrderId(orderIds,
                masterOrderIdBySubOrderId, statStartDay, statEndDay, groupByDate);
        var campaignIdByOrderId = EntryStream.of(orderIdByCampaignId)
                .invert()
                .toMap();
        return EntryStream.of(offerStatsByDayByOrderId)
                .mapKeys(campaignIdByOrderId::get)
                .nonNullKeys()
                .toMap();
    }
}
