package ru.yandex.direct.grid.core.entity.recommendation;

import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.AbstractMap;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.operations.specs.MapSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MergeMode;
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec;
import ru.yandex.inside.yt.kosher.operations.specs.SortSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.GetNode;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public class RecommendationTablesUtils {
    private RecommendationTablesUtils() {
    }

    // Название атрибута с типом ноды Кипариса
    public static final String TYPE = "type";
    // Пользовательский атрибут показывает, откуда приехала таблица. Например, "hahn"
    public static final String ATTR_SOURCE = "attr_source";
    // Long timestamp в формате unix time - время генерации таблицы
    public static final String ATTR_TIMESTAMP = "attr_timestamp";
    // Список типов рекомендаций, сгенерированных в этой таблице (их может быть больше 1)
    public static final String ATTR_TYPES = "attr_types";
    // Минимальные ts по типам рекомендаций - этот атрибут установлен на таблице с основными данными
    public static final String ATTR_MIN_TIMESTAMPS = "attr_min_timestamps";

    public static final String ACCESS_TIME = "access_time";

    public static final String CREATION_TIME = "creation_time";

    public static final String MODIFICATION_TIME = "modification_time";

    public static final List<ColumnInfo> COLUMNS = asList(
            new ColumnInfo("hash", "int64", "int64(farm_hash(client_id))", true),
            new ColumnInfo("client_id", "int64", true),
            new ColumnInfo("type", "int64", true),
            new ColumnInfo("cid", "int64", true),
            new ColumnInfo("pid", "int64", true),
            new ColumnInfo("bid", "int64", true),
            new ColumnInfo("user_key_1", "string", true),
            new ColumnInfo("user_key_2", "string", true),
            new ColumnInfo("user_key_3", "string", true),
            new ColumnInfo("timestamp", "int64", true),
            new ColumnInfo("data", "string", false)
    );

    public static final List<String> KEY_COLUMNS = COLUMNS.stream()
            .filter(c -> c.key).map(c -> c.name).collect(toList());

    public static final String CONF_BASE_DIR = "dynamic-yt.tables.recommendations.base-dir";

    /**
     * Создаёт пустую таблицу для хранения рекомендаций. В зависимости от передаваемых опций может быть
     * создана статическая или динамическая таблица, сортированная или несортированная.
     * minTimestamps записывается в user-атрибуты таблицы.
     */
    public static void createRecommendationsTable(
            YtOperator ytOperator,
            Optional<GUID> transactionId,
            YPath path,
            boolean dynamic,
            boolean sorted,
            String primaryMedium,
            Map<Integer, Long> minTimestamps) {
        Map<String, YTreeNode> attrs = ImmutableMap.of(
                // Динамизм
                "dynamic", YTree.booleanNode(dynamic),
                "primary_medium", YTree.stringNode(primaryMedium),
                "optimize_for", YTree.stringNode("lookup"),
                // Схема
                YtUtils.SCHEMA_ATTR, buildSchema(sorted),
                // Пользовательские атрибуты
                ATTR_MIN_TIMESTAMPS, serializeMinTimestamps(minTimestamps)
        );
        ytOperator.getYt().cypress().create(transactionId, true, path, CypressNodeType.TABLE, true, false, attrs);
    }

    /**
     * Создаёт статическую сортированную таблицу рекомендаций с указанными атрибутами и содержимым.
     * Если таблица с этим именем уже была, она будет заменена новой
     */
    public static void createIncomingTable(
            YtOperator ytOperator,
            Optional<GUID> transactionId,
            YPath path,
            String attrSource,
            long attrTimestamp,
            List<Integer> attrTypes,
            List<RecommendationYtRecord> records
    ) {
        Map<String, YTreeNode> attrs = ImmutableMap.of(
                "dynamic", YTree.booleanNode(false),
                YtUtils.SCHEMA_ATTR, buildSchema(true),
                // Пользовательские атрибуты
                ATTR_SOURCE, YTree.stringNode(attrSource),
                ATTR_TIMESTAMP, YTree.integerNode(attrTimestamp),
                ATTR_TYPES, YTree.builder().value(attrTypes.stream().map(YTree::integerNode).collect(toList())).build()
        );
        ytOperator.getYt().cypress().create(transactionId, true, path, CypressNodeType.TABLE, true, false, true, attrs);

        List<YTreeMapNode> ytRecords = records.stream().map(RecommendationYtRecord::buildMapNode).collect(toList());
        ytOperator.getYt().tables().write(
                transactionId, true, path, YTableEntryTypes.YSON, Cf.wrap(ytRecords).iterator());
    }

    /**
     * Возвращает схему для таблицы рекомендаций.
     *
     * @param sorted true, если нужна сортированная таблица
     */
    public static YTreeNode buildSchema(boolean sorted) {
        YTreeBuilder schemaBuilder = YTree.listBuilder();
        for (ColumnInfo column : COLUMNS) {
            addColumn(schemaBuilder, column.name, column.type, column.expression, column.key && sorted);
        }
        return YTree.builder()
                .beginAttributes()
                .key("strict").value(true)
                .key("unique_keys").value(sorted)
                .endAttributes()
                .value(schemaBuilder.buildList())
                .build();
    }

    private static void addColumn(YTreeBuilder builder, String name, String type,
                                  @Nullable String expression, boolean sorted) {
        builder.beginMap()
                .key("name").value(name)
                .key("type").value(type);
        if (expression != null) {
            builder.key("expression").value(expression);
        }
        if (sorted) {
            builder.key("sort_order").value("ascending");
        }
        builder.key("required").value(false);
        builder.endMap();
    }

    /**
     * Загружает из Yt значение атрибута ATTR_MIN_TIMESTAMPS и конвертирует его в мапу
     */
    public static Map<Integer, Long> loadMinTimestamps(ApiServiceTransaction transaction, YPath table) {
        YTreeNode node = transaction.getNode(new GetNode(table.toString())
                .setAttributes(ColumnFilter.of(ATTR_MIN_TIMESTAMPS)))
                .join(); // IGNORE-BAD-JOIN DIRECT-149116
        return deserializeMinTimestamps(node.getAttributeOrThrow(ATTR_MIN_TIMESTAMPS));
    }

    /**
     * Десериализует значение minTimestamps из Yt атрибута в мапу
     * В значениях соответствия должны быть секунды (unix timestamp).
     * Если значения будут содержать миллисекунды, метод будет возвращать пустую мапу.
     */
    public static Map<Integer, Long> deserializeMinTimestamps(YTreeNode attributeValue) {
        Map<Integer, Long> minTimestamps = EntryStream.of(attributeValue.asMap())
                .mapKeyValue((k, v) -> new AbstractMap.SimpleImmutableEntry<>(Integer.parseInt(k), v.longValue()))
                .toMap(AbstractMap.SimpleImmutableEntry::getKey, AbstractMap.SimpleImmutableEntry::getValue);
        // Если получилось так, что значения отрицательны или не влезают в 32 бита (это может быть из-за того, что
        // были записаны миллисекундные таймстемпы вместо секундных), возвращаем пустую мапу
        if (minTimestamps.values().stream().anyMatch(ts -> ts < 0 || log2(ts) > 32)) {
            return emptyMap();
        }
        return minTimestamps;
    }

    /**
     * Возвращает логарифм по основанию 2 переданного числа (число должно быть неотрицательным).
     */
    public static int log2(long num) {
        checkArgument(num >= 0);
        return 64 - Long.numberOfLeadingZeros(num);
    }

    public static YTreeNode serializeMinTimestamps(Map<Integer, Long> minTimestamps) {
        return YTree.builder().value(
                EntryStream.of(minTimestamps)
                        .mapKeys(type -> Integer.toString(type))
                        .mapValues(YTree::integerNode)
                        .toMap())
                .build();
    }

    /**
     * Перекладывает все данные из таблички А в табличку Б.
     * При этом будут автоматически вычислены вычисляемые колонки (что нам и нужно).
     */
    public static void map(YtOperator ytOperator, YPath inputTable, YPath outputTable, Duration timeout,
                           @Nullable String ytPool) {
        MapSpec.Builder mapOperationBuilder = MapSpec.builder()
                .setInputTables(Cf.wrap(singletonList(inputTable)))
                .setOutputTables(Cf.wrap(singletonList(outputTable)))
                .setMapperCommand("cat");
        if (ytPool != null) {
            mapOperationBuilder.setPool(ytPool);
        }
        MapSpec mapOperation = mapOperationBuilder.build();
        GUID operationId = ytOperator.getYt().operations().map(mapOperation);
        ytOperator.getYt().operations().getOperation(operationId).awaitAndThrowIfNotSuccess(timeout);
    }

    /**
     * Сортирует таблицу inputTable по полям fields, результат записывается в таблицу outputTable.
     */
    public static void sort(YtOperator ytOperator, YPath inputTable, YPath outputTable, List<String> fields,
                            Duration timeout, @Nullable String ytPool) {
        SortSpec sortOperation = new SortSpec(
                Cf.wrap(singletonList(inputTable)),
                outputTable,
                Cf.wrap(fields));
        if (ytPool != null) {
            sortOperation = sortOperation.pool(ytPool);
        }
        GUID operationId = ytOperator.getYt().operations().sort(sortOperation);
        ytOperator.getYt().operations().getOperation(operationId).awaitAndThrowIfNotSuccess(timeout);
    }

    /**
     * Выполняет merge --mode sorted для указанных таблиц, результат записывается в таблицу outputTable.
     */
    public static void mergeSort(YtOperator ytOperator, List<YPath> inputTables, YPath outputTable,
                                 Duration timeout, @Nullable String ytPool) {
        MergeSpec.Builder mergeOperationBuilder = MergeSpec.builder()
                .setMergeMode(MergeMode.SORTED)
                .setCombineChunks(true)
                .setMergeBy(KEY_COLUMNS)
                .setInputTables(inputTables)
                .setOutputTable(outputTable);
        if (ytPool != null) {
            mergeOperationBuilder.setPool(ytPool);
        }
        GUID operationId = ytOperator.getYt().operations().merge(mergeOperationBuilder.build());
        ytOperator.getYt().operations().getOperation(operationId).awaitAndThrowIfNotSuccess(timeout);
    }

    public static YPath generateTableName(YPath directory) {
        return directory.child("recommendations_" + new SimpleDateFormat("yyyyMMdd_HHmmss").format(new Date()));
    }
}
