package ru.yandex.direct.grid.core.entity.recommendation.repository;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Row9;
import org.jooq.Select;
import org.jooq.TableField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.RecommendationsStatusStatus;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.util.yt.YtRecommendationsDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.schema.yt.tables.BannerstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.CurrentRecommendations;
import ru.yandex.direct.grid.schema.yt.tables.PhrasestableDirect;
import ru.yandex.direct.grid.schema.yt.tables.RecommendationsStatustableDirect;
import ru.yandex.direct.grid.schema.yt.tables.records.CurrentRecommendationsRecord;
import ru.yandex.direct.grid.schema.yt.tables.records.RecommendationsStatustableDirectRecord;
import ru.yandex.direct.inventori.model.request.BlockSize;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.maxBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_BANNER;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.ALLOWED_BLOCK_SIZES;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.grid.schema.yt.Tables.BANNERSTABLE_DIRECT;
import static ru.yandex.direct.grid.schema.yt.Tables.CURRENT_RECOMMENDATIONS;
import static ru.yandex.direct.grid.schema.yt.Tables.PHRASESTABLE_DIRECT;
import static ru.yandex.direct.grid.schema.yt.Tables.RECOMMENDATIONS_STATUSTABLE_DIRECT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для получения данных рекоммендаций из YT
 */
@Repository
@ParametersAreNonnullByDefault
public class GridRecommendationYtRepository {
    private static final PhrasestableDirect PHRASES = PHRASESTABLE_DIRECT.as("P");
    private static final BannerstableDirect BANNERS = BANNERSTABLE_DIRECT.as("B");
    private static final CurrentRecommendations RECOMMENDATIONS = CURRENT_RECOMMENDATIONS.as("R");
    private static final RecommendationsStatustableDirect RECOMMENDATIONS_STATUS =
            RECOMMENDATIONS_STATUSTABLE_DIRECT.as("S");
    private static final List<TableField<?, ?>> DEFAULT_ORDER = ImmutableList.of(RECOMMENDATIONS.CID);
    private static final Function<GdiRecommendation, List<Object>> COMPOSITE_KEY = r ->
            asList(r.getClientId(), r.getType(), r.getCid(), r.getPid(), r.getBid(), r.getUserKey1(), r.getUserKey2(),
                    r.getUserKey3());
    private static final Gson GSON = new Gson();

    private static final Row9<Long, Long, Long, Long, Long, String, String, String, Long> RECOMMENDATIONS_KEY_COLUMNS =
            row(RECOMMENDATIONS.CLIENT_ID,
                    RECOMMENDATIONS.TYPE,
                    RECOMMENDATIONS.CID,
                    RECOMMENDATIONS.PID,
                    RECOMMENDATIONS.BID,
                    RECOMMENDATIONS.USER_KEY_1,
                    RECOMMENDATIONS.USER_KEY_2,
                    RECOMMENDATIONS.USER_KEY_3,
                    RECOMMENDATIONS.TIMESTAMP);

    private static final Row9<Long, Long, Long, Long, Long, String, String, String, Long>
            RECOMMENDATIONS_STATUS_KEY_COLUMNS =
            row(RECOMMENDATIONS_STATUS.CLIENT_ID,
                    RECOMMENDATIONS_STATUS.TYPE,
                    RECOMMENDATIONS_STATUS.CID,
                    RECOMMENDATIONS_STATUS.PID,
                    RECOMMENDATIONS_STATUS.BID,
                    RECOMMENDATIONS_STATUS.USER_KEY_1,
                    RECOMMENDATIONS_STATUS.USER_KEY_2,
                    RECOMMENDATIONS_STATUS.USER_KEY_3,
                    RECOMMENDATIONS_STATUS.TIMESTAMP);

    // Стараемся отправлять в YT за раз ключей не более этого количества
    private static final int MAX_KEYS_IN_QUERY = 500;
    private static final Set<GdiRecommendationType> ONLINE_RECOMMENDATION_TYPES = ImmutableSet.of(
            GdiRecommendationType.changeAdGroupForModeration,
            GdiRecommendationType.changeBannerForModeration,
            GdiRecommendationType.uploadAppropriateCreatives,
            GdiRecommendationType.chooseAppropriatePlacementsForAdGroup,
            GdiRecommendationType.chooseAppropriatePlacementsForBanner,
            GdiRecommendationType.addBannerFormatsForPriceSalesCorrectness,
            GdiRecommendationType.addBannerFormatsForPriceSalesCorrectnessInParentCampaign);

    private final YtRecommendationsDynamicSupport ytSupport;
    private final YtFieldMapper<GdiRecommendation, CurrentRecommendationsRecord> recommendationMapper;
    private final YtFieldMapper<GdiRecommendation, RecommendationsStatustableDirectRecord> recommendationStatusMapper;
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final CreativeRepository creativeRepository;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public GridRecommendationYtRepository(YtRecommendationsDynamicSupport gridYtSupport,
                                          ShardHelper shardHelper, AdGroupRepository adGroupRepository,
                                          CreativeRepository creativeRepository,
                                          DslContextProvider dslContextProvider) {
        this.ytSupport = gridYtSupport;
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.creativeRepository = creativeRepository;
        this.dslContextProvider = dslContextProvider;
        JooqReaderWithSupplier<GdiRecommendation> internalReader =
                JooqReaderWithSupplierBuilder.builder(GdiRecommendation::new)
                        .readProperty(GdiRecommendation.CLIENT_ID, fromField(RECOMMENDATIONS.CLIENT_ID))
                        .readProperty(GdiRecommendation.TYPE, fromField(RECOMMENDATIONS.TYPE)
                                .by(GdiRecommendationType::fromId))
                        .readProperty(GdiRecommendation.CID, fromField(RECOMMENDATIONS.CID))
                        .readProperty(GdiRecommendation.PID, fromField(RECOMMENDATIONS.PID))
                        .readProperty(GdiRecommendation.BID, fromField(RECOMMENDATIONS.BID))
                        .readProperty(GdiRecommendation.USER_KEY1, fromField(RECOMMENDATIONS.USER_KEY_1))
                        .readProperty(GdiRecommendation.USER_KEY2, fromField(RECOMMENDATIONS.USER_KEY_2))
                        .readProperty(GdiRecommendation.USER_KEY3, fromField(RECOMMENDATIONS.USER_KEY_3))
                        .readProperty(GdiRecommendation.TIMESTAMP, fromField(RECOMMENDATIONS.TIMESTAMP))
                        .readProperty(GdiRecommendation.KPI, fromField(RECOMMENDATIONS.DATA))
                        .readProperty(GdiRecommendation.STATUS, fromField(RECOMMENDATIONS_STATUS.STATUS)
                                .by(s -> s == null ? null
                                        : RecommendationStatus
                                        .fromSource(RecommendationsStatusStatus.valueOf(s))))
                        .build();

        JooqReaderWithSupplier<GdiRecommendation> internalStatusReader =
                JooqReaderWithSupplierBuilder.builder(GdiRecommendation::new)
                        .readProperty(GdiRecommendation.CLIENT_ID, fromField(RECOMMENDATIONS_STATUS.CLIENT_ID))
                        .readProperty(GdiRecommendation.TYPE, fromField(RECOMMENDATIONS_STATUS.TYPE)
                                .by(GdiRecommendationType::fromId))
                        .readProperty(GdiRecommendation.CID, fromField(RECOMMENDATIONS_STATUS.CID))
                        .readProperty(GdiRecommendation.PID, fromField(RECOMMENDATIONS_STATUS.PID))
                        .readProperty(GdiRecommendation.BID, fromField(RECOMMENDATIONS_STATUS.BID))
                        .readProperty(GdiRecommendation.USER_KEY1, fromField(RECOMMENDATIONS_STATUS.USER_KEY_1))
                        .readProperty(GdiRecommendation.USER_KEY2, fromField(RECOMMENDATIONS_STATUS.USER_KEY_2))
                        .readProperty(GdiRecommendation.USER_KEY3, fromField(RECOMMENDATIONS_STATUS.USER_KEY_3))
                        .readProperty(GdiRecommendation.TIMESTAMP, fromField(RECOMMENDATIONS_STATUS.TIMESTAMP))
                        .readProperty(GdiRecommendation.STATUS, fromField(RECOMMENDATIONS_STATUS.STATUS)
                                .by(s -> s == null ? null
                                        : RecommendationStatus
                                        .fromSource(RecommendationsStatusStatus.valueOf(s))))
                        .build();

        this.recommendationMapper = new YtFieldMapper<>(internalReader, RECOMMENDATIONS);
        this.recommendationStatusMapper = new YtFieldMapper<>(internalStatusReader, RECOMMENDATIONS_STATUS);
    }

    public List<GdiRecommendation> getRecommendations(Long clientId,
                                                      Collection<GdiRecommendationType> types,
                                                      @Nullable Collection<Long> campaignIds,
                                                      @Nullable Collection<Long> groupIds,
                                                      @Nullable Collection<Long> bannerIds,
                                                      @Nullable Collection<String> userKeys1,
                                                      @Nullable Collection<String> userKeys2,
                                                      @Nullable Collection<String> userKeys3) {
        Condition baseYtCondition = RECOMMENDATIONS.CLIENT_ID.eq(clientId)
                .and(YtDSL.isNull(RECOMMENDATIONS_STATUS.TIMESTAMP));
        Condition baseMysqlCondition = CAMPAIGNS.CLIENT_ID.eq(clientId)
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.TYPE.notIn(CampaignsType.cpm_banner, CampaignsType.cpm_deals));

        if (types.isEmpty()) {
            return emptyList();
        }

        final List<Long> ytRecommendationTypes = types.stream()
                .filter(t -> !ONLINE_RECOMMENDATION_TYPES.contains(t))
                .map(GdiRecommendationType::getId)
                .collect(toList());
        baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.TYPE.in(ytRecommendationTypes));

        Collection<Long> campIds = campaignIds;
        if (campIds != null && !campIds.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.CID.in(campIds));
        }
        if (groupIds != null && !groupIds.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.PID.in(groupIds));
        }
        if (bannerIds != null && !bannerIds.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.BID.in(bannerIds));
        }
        if (userKeys1 != null && !userKeys1.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.USER_KEY_1.in(userKeys1));
        }
        if (userKeys2 != null && !userKeys2.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.USER_KEY_2.in(userKeys2));
        }
        if (userKeys3 != null && !userKeys3.isEmpty()) {
            baseYtCondition = baseYtCondition.and(RECOMMENDATIONS.USER_KEY_3.in(userKeys3));
        }
        List<GdiRecommendation> recommendations = new ArrayList<>();
        // changeAdGroupForModeration - тип рекомендаций, расчитываемый "на лету" запросом в mysql
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
        if (types.contains(GdiRecommendationType.changeAdGroupForModeration)
                || types.contains(GdiRecommendationType.changeBannerForModeration)
                || types.contains(GdiRecommendationType.addBannerFormats)) {
            long now = new Date().getTime() / 1000;
            if (isEmpty(campIds)) {
                try (TraceProfile ignore = Trace.current().profile("recommendations:mysql")) {
                    campIds = dslContextProvider.ppc(shard)
                            .select(CAMPAIGNS.CID)
                            .from(CAMPAIGNS)
                            .where(baseMysqlCondition)
                            .fetch(CAMPAIGNS.CID);
                }
            }
            if (!campIds.isEmpty()) {
                if (types.contains(GdiRecommendationType.addBannerFormats)) {
                    Collection<Long> adGroupIds = groupIds;
                    if (isEmpty(adGroupIds)) {
                        adGroupIds = flatMap(adGroupRepository.getAdGroupIdsByCampaignIds(shard, campIds).values(),
                                identity());
                    }
                    Map<Long, AdGroupSimple> adGroupsByIds = adGroupRepository.getAdGroupSimple(shard,
                            ClientId.fromLong(clientId), adGroupIds);
                    Map<Long, List<Creative>> creativesByPerformanceAdGroupIds =
                            creativeRepository.getCreativesByPerformanceAdGroupIds(shard, ClientId.fromLong(clientId),
                                    campIds, adGroupIds);

                    for (Long groupId : adGroupIds) {
                        List<Creative> creatives = creativesByPerformanceAdGroupIds.getOrDefault(groupId, emptyList());
                        boolean hasAdaptive = creatives.stream().anyMatch(Creative::getIsAdaptive);

                        if (adGroupsByIds.get(groupId).getType() == CPM_BANNER && !hasAdaptive) {
                            Set<BlockSize> blockSizes = listToSet(creatives,
                                    c -> new BlockSize(c.getWidth().intValue(), c.getHeight().intValue()));
                            Set<BlockSize> difference = Sets.difference(ALLOWED_BLOCK_SIZES, blockSizes);

                            if (!difference.isEmpty()) {
                                String formats = GSON.toJson(singletonMap("formats", mapList(difference,
                                        b -> "" + b.getWidth() + "x" + b.getHeight())));

                                GdiRecommendation recommendation =
                                        new GdiRecommendation()
                                                .withClientId(clientId)
                                                .withType(GdiRecommendationType.addBannerFormats)
                                                .withCid(adGroupsByIds.get(groupId).getCampaignId())
                                                .withPid(groupId)
                                                .withBid(0L)
                                                .withUserKey1("")
                                                .withUserKey2("")
                                                .withUserKey3("")
                                                .withTimestamp(now)
                                                .withKpi(formats);

                                recommendations.add(recommendation);
                            }
                        }
                    }
                }

                if (types.contains(GdiRecommendationType.changeAdGroupForModeration)) {
                    List<Field<Long>> selectFields = Arrays.asList(BANNERS.CID_HASH.as(BANNERS.CID_HASH.getName()),
                            BANNERS.CID.as(BANNERS.CID.getName()), BANNERS.PID.as(BANNERS.PID.getName()));
                    List<GdiRecommendation> onlineRecommendations;
                    try (TraceProfile ignore = Trace.current().profile("recommendations:yt",
                            GdiRecommendationType.changeAdGroupForModeration.name())) {
                        onlineRecommendations = ytSupport.selectRows(shard, YtDSL.ytContext()
                                .select(selectFields)
                                .from(PHRASES)
                                .join(BANNERS).on(row(PHRASES.CID, PHRASES.PID)
                                        .eq(row(BANNERS.CID, BANNERS.PID)))
                                .where(PHRASES.CID.in(campIds)
                                        .and(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.No.getLiteral()))
                                        .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No.getLiteral())))
                                .groupBy(selectFields))
                                .getYTreeRows()
                                .stream()
                                .map(r -> new GdiRecommendation()
                                        .withClientId(clientId)
                                        .withType(GdiRecommendationType.changeAdGroupForModeration)
                                        .withCid(r.getLong(BANNERS.CID.getName()))
                                        .withPid(r.getLong(BANNERS.PID.getName()))
                                        .withBid(0L)
                                        .withUserKey1("")
                                        .withUserKey2("")
                                        .withUserKey3("")
                                        .withTimestamp(now)
                                )
                                .collect(toList());
                    }
                    recommendations.addAll(onlineRecommendations);
                }
                if (types.contains(GdiRecommendationType.changeBannerForModeration)) {
                    try (TraceProfile ignore = Trace.current().profile("recommendations:yt",
                            GdiRecommendationType.changeBannerForModeration.name())) {
                        List<Field<Long>> selectFields = Arrays.asList(BANNERS.CID_HASH.as(BANNERS.CID_HASH.getName()),
                                BANNERS.CID.as(BANNERS.CID.getName()), BANNERS.PID.as(BANNERS.PID.getName()),
                                BANNERS.BID.as(BANNERS.BID.getName()));
                        List<GdiRecommendation> onlineRecommendations = ytSupport.selectRows(shard, YtDSL.ytContext()
                                .select(selectFields)
                                .from(BANNERS)
                                .where(BANNERS.CID.in(campIds)
                                        .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.No.getLiteral()))
                                        .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No.getLiteral())))
                                .groupBy(selectFields))
                                .getYTreeRows()
                                .stream()
                                .map(r -> new GdiRecommendation()
                                        .withClientId(clientId)
                                        .withType(GdiRecommendationType.changeBannerForModeration)
                                        .withCid(r.getLong(BANNERS.CID.getName()))
                                        .withPid(r.getLong(BANNERS.PID.getName()))
                                        .withBid(r.getLong(BANNERS.BID.getName()))
                                        .withUserKey1("")
                                        .withUserKey2("")
                                        .withUserKey3("")
                                        .withTimestamp(now)
                                )
                                .collect(toList());
                        recommendations.addAll(onlineRecommendations);
                    }
                }
            }
        }

        try (TraceProfile ignore = Trace.current().profile("recommendations:yt:get")) {
            final List<GdiRecommendation> ytRecommendations = ytSupport.selectRows(
                    shard,
                    YtDSL.ytContext()
                            .select(recommendationMapper.getFieldsToRead())
                            .from(RECOMMENDATIONS)
                            .leftJoin(RECOMMENDATIONS_STATUS)
                            .on(RECOMMENDATIONS_KEY_COLUMNS.eq(RECOMMENDATIONS_STATUS_KEY_COLUMNS))
                            .where(baseYtCondition))
                    .getYTreeRows().stream()
                    .map(recommendationMapper::fromNode)
                    .collect(toList());
            recommendations.addAll(ytRecommendations);
        }

        // выбрать самую "свежую" рекомендацию (с максимальным timestamp)
        recommendations = recommendations.stream()
                .collect(groupingBy(COMPOSITE_KEY, maxBy(Comparator.comparing(GdiRecommendation::getTimestamp))))
                .values().stream()
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(toList());
        return recommendations;
    }

    /**
     * Загружает из yt и mysql самые свежие рекомендации для указанных префиксов ключей
     *
     * @param keys        ключи рекомендаций (timestamp не учитывается, заполнять его не нужно)
     * @param clientShard clientID -> shard
     */
    public List<GdiRecommendation> getRecommendationsByKeyPrefixes(Set<RecommendationKey> keys,
                                                                   Map<Long, Integer> clientShard) {
        if (keys.isEmpty()) {
            return emptyList();
        }

        // NB: Наверное, можно более умно разбить этот список на части, используя знания, какие шарды
        // в каких кластерах сейчас нужно запрашивать (с учётом свежести).
        List<List<RecommendationKey>> chunks = Lists.partition(new ArrayList<>(keys), MAX_KEYS_IN_QUERY);
        List<GdiRecommendation> resultList = new ArrayList<>();
        for (List<RecommendationKey> chunk : chunks) {
            Set<Integer> shardsInChunk =
                    chunk.stream().map(RecommendationKey::getClientId).map(clientShard::get).collect(toSet());
            UnversionedRowset rows = ytSupport.selectRows(
                    shardsInChunk,
                    shards -> {
                        List<RecommendationKey> keysOfShard = chunk.stream()
                                .filter(k -> shards.contains(clientShard.get(k.getClientId())))
                                .collect(toList());

                        Preconditions.checkState(!keysOfShard.isEmpty());
                        return getRecommendationsByKeyPrefixesQuery(keysOfShard);
                    });

            resultList.addAll(
                    rows.getYTreeRows().stream()
                            .map(recommendationMapper::fromNode)
                            .collect(toList())
            );
        }

        // Когда будем добавлять поддержку recommendations_online, нужно будет дополнительно получить записи и оттуда
        // тоже
        // И после из тех, которые отличаются только таймстемпом, убрать старые
        return resultList;
    }

    /**
     * Метод, аналогичный {@link #getRecommendationsByKeyPrefixes(Set, Map)}, но для одного шарда
     *
     * @param keys ключи рекомендаций (timestamp не учитывается, заполнять его не нужно)
     */
    public Set<GdiRecommendation> getRecommendationsByKeyPrefixes(Set<RecommendationKey> keys,
                                                                  Integer shard) {
        if (keys.isEmpty()) {
            return emptySet();
        }
        List<List<RecommendationKey>> chunks = Lists.partition(new ArrayList<>(keys), MAX_KEYS_IN_QUERY);
        Set<GdiRecommendation> result = new HashSet<>();
        for (List<RecommendationKey> chunk : chunks) {
            UnversionedRowset rows = ytSupport.selectRows(
                    shard,
                    getRecommendationsByKeyPrefixesQuery(chunk)
            );
            result.addAll(rows.getYTreeRows().stream()
                    .map(recommendationMapper::fromNode)
                    .collect(toSet()));
        }
        return result;
    }

    private Select getRecommendationsByKeyPrefixesQuery(List<RecommendationKey> keys) {
        return YtDSL.ytContext()
                .select(
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.CLIENT_ID),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.TYPE),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.CID),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.PID),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.BID),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.USER_KEY_1),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.USER_KEY_2),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.USER_KEY_3),
                        recommendationMapper.fieldAlias(RECOMMENDATIONS.TIMESTAMP)
                )
                .from(RECOMMENDATIONS)
                .leftJoin(RECOMMENDATIONS_STATUS)
                .on(RECOMMENDATIONS_KEY_COLUMNS.eq(RECOMMENDATIONS_STATUS_KEY_COLUMNS))
                .where(recommendationKeyIn(keys)
                        .and(YtDSL.isNull(RECOMMENDATIONS_STATUS.STATUS)));
    }


    /**
     * Получает статусы рекомендаций по ключам из таблицы "RECOMMENDATIONS_STATUS" без учета статуса "IN_PROGRESS"
     *
     * @param clientId идентификатор клиента
     * @param keys     ключи рекомендаций
     */
    public List<GdiRecommendation> getRecommendationsStatuses(Long clientId, Collection<RecommendationKey> keys) {
        final int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));

        List<List<RecommendationKey>> chunks = Lists.partition(new ArrayList<>(keys), MAX_KEYS_IN_QUERY);
        List<GdiRecommendation> resultList = new ArrayList<>();
        for (List<RecommendationKey> chunk : chunks) {
            UnversionedRowset rows = ytSupport.selectRows(
                    shard,
                    YtDSL.ytContext()
                            .select(recommendationStatusMapper.getFieldsToRead())
                            .from(RECOMMENDATIONS_STATUS)
                            .where(recommendationStatusKeyIn(chunk))
                            //'ne' сериализуется как '<>' вместо '!=', что не принимает YT
                            // поэтому используем not(status = "in_progress")
                            .andNot(RECOMMENDATIONS_STATUS.STATUS
                                    .eq(RecommendationsStatusStatus.in_progress.getLiteral())));

            resultList.addAll(
                    rows.getYTreeRows().stream()
                            .map(recommendationStatusMapper::fromNode)
                            .collect(toList())
            );
        }

        return resultList;
    }

    /**
     * Возвращает condition для предиката вида
     * WHERE (client_id, type, cid, pid, bid, user_key_1, user_key_2, user_key_3) in ((…), (…), …, (…))
     * timestamp в этом предикате не учитывается
     * это наиболее подходящий способ выбрать из Yt массово по перечислению ключей за минимум текста YtQL
     */
    private Condition recommendationKeyIn(Collection<RecommendationKey> keys) {
        if (keys.isEmpty()) {
            // Если элементов нет, вернём false-предикат
            return YtDSL.nativeCondition("1 = 0");
        }

        Map<Integer, List<RecommendationKey>> keysByPrefixLen =
                keys.stream().collect(groupingBy(this::getKeyPrefixLen));

        List<? extends TableField<CurrentRecommendationsRecord, ? extends Serializable>> fields =
                asList(RECOMMENDATIONS.CLIENT_ID, RECOMMENDATIONS.TYPE, RECOMMENDATIONS.CID, RECOMMENDATIONS.PID,
                        RECOMMENDATIONS.BID, RECOMMENDATIONS.USER_KEY_1, RECOMMENDATIONS.USER_KEY_2,
                        RECOMMENDATIONS.USER_KEY_3, RECOMMENDATIONS.TIMESTAMP);
        List<Function<RecommendationKey, Object>> fieldExtractors = asList(
                RecommendationKey::getClientId,
                RecommendationKey::getType,
                RecommendationKey::getCampaignId,
                RecommendationKey::getAdGroupId,
                RecommendationKey::getBannerId,
                RecommendationKey::getUserKey1,
                RecommendationKey::getUserKey2,
                RecommendationKey::getUserKey3,
                RecommendationKey::getTimestamp
        );
        List<Condition> conditions = new ArrayList<>();
        for (Integer prefixLen : keysByPrefixLen.keySet()) {
            List<RecommendationKey> keysOfThisLen = keysByPrefixLen.get(prefixLen);
            if (!keysOfThisLen.isEmpty()) {
                Condition condition = row(fields.subList(0, prefixLen))
                        .in(keysOfThisLen.stream().map(
                                key -> row(fieldExtractors.subList(0, prefixLen).stream().map(fx -> fx.apply(key))
                                        .collect(toList()))
                        ).collect(toList()));
                conditions.add(condition);
            }
        }
        return conditions.stream().reduce(Condition::or).get();
    }

    /**
     * todo : unit test
     * Возвращает длину префикса ключа (кол-во колонок префиксе) для переданной модели.
     * Хотя бы 1 колонка должна быть не равна null.
     */
    private int getKeyPrefixLen(RecommendationKey key) {
        if (key.getClientId() == null) {
            throw new IllegalArgumentException("client_id shouldn't be null");
        }
        if (key.getType() == null) {
            return 1;
        }
        if (key.getCampaignId() == null) {
            return 2;
        }
        if (key.getAdGroupId() == null) {
            return 3;
        }
        if (key.getBannerId() == null) {
            return 4;
        }
        if (key.getUserKey1() == null) {
            return 5;
        }
        if (key.getUserKey2() == null) {
            return 6;
        }
        if (key.getUserKey3() == null) {
            return 7;
        }
        if (key.getTimestamp() == null) {
            return 8;
        }
        return 9;
    }

    /**
     * Возвращает condition для таблицы "RECOMMENDATIONS_STATUS" для предиката вида
     * WHERE (client_id, type, cid, pid, bid, user_key_1, user_key_2, user_key_3, timestamp) in ((…), (…), …, (…))
     */
    private Condition recommendationStatusKeyIn(Collection<RecommendationKey> keys) {
        if (keys.isEmpty()) {
            // Если элементов нет, вернём false-предикат
            return YtDSL.nativeCondition("1 = 0");
        }

        return row(
                RECOMMENDATIONS_STATUS.CLIENT_ID, RECOMMENDATIONS_STATUS.TYPE, RECOMMENDATIONS_STATUS.CID,
                RECOMMENDATIONS_STATUS.PID,
                RECOMMENDATIONS_STATUS.BID, RECOMMENDATIONS_STATUS.USER_KEY_1, RECOMMENDATIONS_STATUS.USER_KEY_2,
                RECOMMENDATIONS_STATUS.USER_KEY_3,
                RECOMMENDATIONS_STATUS.TIMESTAMP

        ).in(
                keys.stream().map(key ->
                        row(key.getClientId(), key.getType(), key.getCampaignId(), key.getAdGroupId(),
                                key.getBannerId(), key.getUserKey1(), key.getUserKey2(), key.getUserKey3(),
                                key.getTimestamp()))
                        .collect(toList())
        );
    }
}
