package ru.yandex.direct.grid.core.entity.recommendation.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.gson.Gson;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.service.CommunicationChannelService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.campdaybudgethistory.repository.CampDayBudgetStopHistoryRepository;
import ru.yandex.direct.core.entity.client.model.ClientsOptions;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.payment.service.AutopayService;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.utils.UserUtil;
import ru.yandex.direct.core.entity.walletparams.repository.WalletPaymentTransactionsRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.entity.recommendation.repository.GridRecommendationYtRepository;
import ru.yandex.direct.grid.core.entity.recommendation.service.cpmprice.GridBannerFormatsForPriceSalesRecommendationService;
import ru.yandex.direct.grid.core.entity.recommendation.service.outdoor.GridOutdoorVideoRecommendationForAdGroupService;
import ru.yandex.direct.grid.core.entity.recommendation.service.outdoor.GridOutdoorVideoRecommendationForBannersService;
import ru.yandex.direct.grid.core.entity.recommendation.service.outdoor.GridOutdoorVideoRecommendationForPlacementsService;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.model.recommendation.GdAutopayErrors;
import ru.yandex.direct.grid.processing.model.recommendation.GdDateFlag;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationItem;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationItems;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKey;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiAddBannerFormats;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiAddBannerFormatsForPriceSalesCorrectness;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiAddTurboWebSite;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiAutopayStopped;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiChooseAppropriatePlacementsForAdGroup;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiChooseAppropriatePlacementsForBanner;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiDailyBudget;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiGraph;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiGraphPoint;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiMainInvoice;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiOverdraftDebt;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiRemovePagesFromBlackList;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiRemovePagesFromBlackListOfACampaign;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiSwitchOnAutotargeting;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiUploadAppropriateCreatives;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiWeeklyBudget;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSign;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationStatus;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSummary;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSummaryWithKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSummaryWithoutKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationWithKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationWithoutKpi;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.utils.StringUtils;

import static java.math.BigDecimal.ZERO;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_OUTDOOR;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.WEB_EDIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DAILY_BUDGET_STOP_WARNING_PERIOD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DAILY_BUDGET_STOP_WARNING_TIME_DEFAULT;
import static ru.yandex.direct.currency.CurrencyCode.RUB;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.dailyBudget;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.increaseStrategyWeeklyBudget;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Сервис за рекоммендации
 */
@Repository
@ParametersAreNonnullByDefault
public class GridRecommendationService {

    private static final Logger logger = LoggerFactory.getLogger(GridRecommendationService.class);

    private static final String AUTOPAY_NOT_ENOUGH_FUNDS_ERROR = "not_enough_funds";
    private static final String AUTOPAY_EXPIRED_CARD_ERROR = "expired_card";
    private static final Integer DEFAULT_RECOMMENDATION_PERIOD = 0;
    protected static final String DAILY_BUDGET_RECOMMENDATION_MESSAGE_NAME = "dailyBudget";
    protected static final String WEEKLY_BUDGET_RECOMMENDATION_MESSAGE_NAME = "weekBudget";

    public static final Function<List<GdRecommendation>, BigDecimal>
            RECOMMENDATIONS_COST_EXTRACTOR = recommendations -> recommendations.stream()
            .filter(r -> r.getClass().isAssignableFrom(GdRecommendationWithKpi.class))
            .map(GdRecommendationWithKpi.class::cast)
            .map(GdRecommendationWithKpi::getKpi)
            .filter(Objects::nonNull)
            .map(GdRecommendationKpi::getCost)
            .reduce(ZERO, BigDecimal::add);

    public static final Function<List<GdRecommendation>, BigDecimal>
            RECOMMENDATIONS_CLICKS_EXTRACTOR = recommendations -> recommendations.stream()
            .filter(r -> r.getClass().isAssignableFrom(GdRecommendationWithKpi.class))
            .map(GdRecommendationWithKpi.class::cast)
            .map(GdRecommendationWithKpi::getKpi)
            .filter(Objects::nonNull)
            .map(GdRecommendationKpi::getClicks)
            .reduce(BigDecimal::add).orElse(ZERO);

    private static final Set<GdiRecommendationType> ALL_ALLOWED_IN_CAMPAIGN_GRID_RECOMMENDATION_TYPES =
            EnumSet.of(GdiRecommendationType.autopayStopped,
                    GdiRecommendationType.overdraftDebt,
                    GdiRecommendationType.videoButton,
                    GdiRecommendationType.mainInvoice
            );

    private static final Set<GdiRecommendationType> ALLOWED_SUMMARY_RECOMMENDATION_TYPES =
            EnumSet.of(dailyBudget,
                    GdiRecommendationType.weeklyBudget,
                    GdiRecommendationType.removePagesFromBlackList,
                    GdiRecommendationType.changeAdGroupForModeration,
                    GdiRecommendationType.changeAdGroupWithLowStat,
                    GdiRecommendationType.addMorePicturesToAdgroup,
                    GdiRecommendationType.switchOnAutotargeting,
                    GdiRecommendationType.changeBannerForModeration,
                    GdiRecommendationType.sendLicensesBanks,
                    GdiRecommendationType.sendLicensesMedServices,
                    GdiRecommendationType.sendLicensesPharmacy,
                    GdiRecommendationType.addTitleExtension,
                    GdiRecommendationType.addSiteLinks,
                    GdiRecommendationType.addAdditionItemCallouts,
                    GdiRecommendationType.addBannerDisplayHrefs,
                    GdiRecommendationType.addImageToBanner,
                    GdiRecommendationType.addTurboWebSiteToBanner,
                    GdiRecommendationType.addBannerFormats,
                    GdiRecommendationType.increaseStrategyWeeklyBudget,
                    GdiRecommendationType.increaseStrategyTargetCPA,
                    GdiRecommendationType.decreaseStrategyTargetROI);

    protected static final Set<GdiRecommendationType> ALL_ALLOWED_SUMMARY_RECOMMENDATION_TYPES =
            SetUtils.union(ALLOWED_SUMMARY_RECOMMENDATION_TYPES, ALL_ALLOWED_IN_CAMPAIGN_GRID_RECOMMENDATION_TYPES);

    private static final Set<GdiRecommendationType> ALL_ALLOWED_GROUP_RECOMMENDATION_TYPES =
            EnumSet.of(GdiRecommendationType.changeAdGroupForModeration,
                    GdiRecommendationType.changeAdGroupWithLowStat,
                    GdiRecommendationType.addMorePicturesToAdgroup,
                    GdiRecommendationType.uploadAppropriateCreatives,
                    GdiRecommendationType.chooseAppropriatePlacementsForAdGroup,
                    GdiRecommendationType.switchOnAutotargeting,
                    GdiRecommendationType.addBannerFormats,
                    GdiRecommendationType.addBannerFormatsForPriceSalesCorrectnessInParentCampaign);

    public static final Set<GdiRecommendationType> ALL_ALLOWED_CAMPAIGN_RECOMMENDATION_TYPES =
            EnumSet.of(dailyBudget,
                    GdiRecommendationType.removePagesFromBlackListOfACampaign,
                    GdiRecommendationType.weeklyBudget,
                    GdiRecommendationType.increaseStrategyWeeklyBudget,
                    GdiRecommendationType.increaseStrategyTargetCPA,
                    GdiRecommendationType.decreaseStrategyTargetROI,
                    GdiRecommendationType.addBannerFormatsForPriceSalesCorrectness);

    private static final Set<GdiRecommendationType> ALL_ALLOWED_BANNER_RECOMMENDATION_TYPES =
            EnumSet.of(GdiRecommendationType.changeBannerForModeration,
                    GdiRecommendationType.sendLicensesBanks,
                    GdiRecommendationType.sendLicensesMedServices,
                    GdiRecommendationType.sendLicensesPharmacy,
                    GdiRecommendationType.addTitleExtension,
                    GdiRecommendationType.addSiteLinks,
                    GdiRecommendationType.addAdditionItemCallouts,
                    GdiRecommendationType.addBannerDisplayHrefs,
                    GdiRecommendationType.addImageToBanner,
                    GdiRecommendationType.addTurboWebSiteToBanner,
                    GdiRecommendationType.chooseAppropriatePlacementsForBanner);

    private static final Set<GdiRecommendationType> ALLOWED_UNIVERSAL_CAMPAIGN_RECOMMENDATION_TYPES =
            EnumSet.noneOf(GdiRecommendationType.class);

    private static final Set<GdiRecommendationType> OUTDOOR_VIDEO_RECOMMENDATIONS_TYPES =
            EnumSet.of(GdiRecommendationType.uploadAppropriateCreatives,
                    GdiRecommendationType.chooseAppropriatePlacementsForAdGroup,
                    GdiRecommendationType.chooseAppropriatePlacementsForBanner);

    private static final Gson GSON = new Gson();

    public static final Function<GdiRecommendation, GdRecommendation>
            GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION = gdi -> {

        final GdRecommendation gdRecommendation;
        if (gdi.getKpi() != null) {
            final GdRecommendationKpi kpi;
            if (gdi.getType() == dailyBudget) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiDailyBudget.class);
            } else if (gdi.getType() == GdiRecommendationType.weeklyBudget
                    || gdi.getType() == GdiRecommendationType.increaseStrategyWeeklyBudget
                    || gdi.getType() == GdiRecommendationType.increaseStrategyTargetCPA
                    || gdi.getType() == GdiRecommendationType.decreaseStrategyTargetROI) {
                GdRecommendationKpiWeeklyBudget weeklyBudgetKpi
                        = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiWeeklyBudget.class);

                if (weeklyBudgetKpi.getCurrentWeeklyBudget() != null
                        && weeklyBudgetKpi.getRecommendedWeeklyBudget() != null) {
                    weeklyBudgetKpi.withDiffWeeklyBudget(weeklyBudgetKpi.getRecommendedWeeklyBudget()
                            .subtract(weeklyBudgetKpi.getCurrentWeeklyBudget()));
                }

                if (weeklyBudgetKpi.getCurrentTargetCPA() != null
                        && weeklyBudgetKpi.getRecommendedTargetCPA() != null) {
                    weeklyBudgetKpi.withDiffTargetCPA(weeklyBudgetKpi.getRecommendedTargetCPA()
                            .subtract(weeklyBudgetKpi.getCurrentTargetCPA()));
                }

                if (weeklyBudgetKpi.getCurrentTargetROI() != null
                        && weeklyBudgetKpi.getRecommendedTargetROI() != null) {
                    weeklyBudgetKpi.withDiffTargetROI(weeklyBudgetKpi.getCurrentTargetROI()
                            .subtract(weeklyBudgetKpi.getRecommendedTargetROI()));
                }

                kpi = weeklyBudgetKpi;
            } else if (gdi.getType() == GdiRecommendationType.removePagesFromBlackListOfACampaign) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiRemovePagesFromBlackList.class);
            } else if (gdi.getType() == GdiRecommendationType.switchOnAutotargeting) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiSwitchOnAutotargeting.class);
            } else if (gdi.getType() == GdiRecommendationType.addTurboWebSiteToBanner) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiAddTurboWebSite.class);
            } else if (gdi.getType() == GdiRecommendationType.uploadAppropriateCreatives) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiUploadAppropriateCreatives.class);
            } else if (gdi.getType() == GdiRecommendationType.chooseAppropriatePlacementsForAdGroup) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiChooseAppropriatePlacementsForAdGroup.class);
            } else if (gdi.getType() == GdiRecommendationType.chooseAppropriatePlacementsForBanner) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiChooseAppropriatePlacementsForBanner.class);
            } else if (gdi.getType() == GdiRecommendationType.addBannerFormatsForPriceSalesCorrectness) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiAddBannerFormatsForPriceSalesCorrectness.class);
            } else if (gdi.getType() == GdiRecommendationType.addBannerFormats) {
                kpi = GSON.fromJson(gdi.getKpi(), GdRecommendationKpiAddBannerFormats.class)
                        .withClicks(ZERO)
                        .withGoals(ZERO)
                        .withCost(ZERO)
                        .withCurrency(RUB)
                        .withSign(GdRecommendationSign.CPA)
                        .withPeriod(DEFAULT_RECOMMENDATION_PERIOD)
                        .withGraph(new GdRecommendationKpiGraph().withTypeX("").withTypeY("")
                                .withPoints(singletonList(
                                        new GdRecommendationKpiGraphPoint()
                                                .withCurrentValue(ZERO)
                                                .withProfitValue(ZERO)
                                                .withHoliday("")
                                                .withTypeValueX("")
                                                .withValueX(""))));
            } else {
                kpi = null;
            }

            if (kpi != null) {
                kpi.withClicks(nvl(kpi.getClicks(), ZERO))
                        .withGoals(nvl(kpi.getGoals(), ZERO))
                        .withCost(nvl(kpi.getCost(), ZERO));
            }

            gdRecommendation = new GdRecommendationWithKpi().withKpi(kpi);
        } else {
            gdRecommendation = new GdRecommendationWithoutKpi();
        }
        return gdRecommendation
                .withKeys(
                        singletonList(new GdRecommendationKey()
                                .withType(gdi.getType())
                                .withCid(gdi.getCid())
                                .withPid(gdi.getPid())
                                .withBid(gdi.getBid())
                                .withUserKey1(gdi.getUserKey1())
                                .withUserKey2(gdi.getUserKey2())
                                .withUserKey3(gdi.getUserKey3())
                                .withTimestamp(gdi.getTimestamp()))
                )
                .withType(gdi.getType())
                .withIsApplicable(gdi.getIsApplicable())
                .withStatus(Optional.ofNullable(gdi.getStatus())
                        .map(Enum::name)
                        .map(GdRecommendationStatus::valueOf)
                        .orElse(GdRecommendationStatus.READY));
    };

    private static final BiFunction<GdiRecommendationType, List<GdRecommendation>, GdRecommendationSummary>
            GDI_RECOMMENDATION_TYPE_LIST_GD_RECOMMENDATION_SUMMARY_BI_FUNCTION = (type, gdRecommendations) -> {
        final GdRecommendationKpi totalKpi = getTotalKpi(gdRecommendations);

        final GdRecommendationItems items = new GdRecommendationItems()
                .withRowset(gdRecommendations.stream()
                        .filter(r -> GdRecommendationWithKpi.class.isAssignableFrom(r.getClass()))
                        .map(GdRecommendationWithKpi.class::cast)
                        .map(GdRecommendationWithKpi::getKpi)
                        .filter(Objects::nonNull)
                        .sorted(Comparator.comparing(GdRecommendationKpi::getClicks)
                                .thenComparing(GdRecommendationKpi::getGoals).reversed())
                        .limit(3)
                        .map(k -> new GdRecommendationItem()
                                .withName(getRecommendationItemName(k))
                                .withClicks(k.getClicks().longValue())
                                .withGoals(k.getGoals().longValue())
                                .withCost(k.getCost())
                                .withCurrency(k.getCurrency())
                        ).collect(toList()))
                .withTotalCount(gdRecommendations.size());

        final GdRecommendationSummary gdRecommendationSummary =
                totalKpi != null
                        ? new GdRecommendationSummaryWithKpi().withKpi(totalKpi).withItems(items)
                        : new GdRecommendationSummaryWithoutKpi();

        final List<GdRecommendationKey> keys = gdRecommendations.stream()
                .map(GdRecommendation::getKeys)
                .flatMap(Collection::stream)
                .collect(toList());
        final GdRecommendationStatus status = getTotalStatus(gdRecommendations);


        final Set<Long> campaignIds = keys.stream()
                .map(GdRecommendationKey::getCid)
                .collect(toSet());

        // если хотя бы одна рекомендация данного типа 'isApplicable',
        // то суммарный isApplicable = true,
        // иначе суммарная возможность применения - false
        final boolean isApplicable = gdRecommendations.stream()
                .anyMatch(GdRecommendation::getIsApplicable);

        return gdRecommendationSummary
                .withType(type)
                // ключи отдаем только для "применябельных" типов рекомендаций
                .withKeys(type.getCoreType().isAppliable() ? keys : emptyList())
                .withCampaignIds(campaignIds)
                .withStatus(status)
                .withIsApplicable(isApplicable)
                .withTotalCount(gdRecommendations.size());
    };

    private static String getRecommendationItemName(GdRecommendationKpi kpi) {
        if (GdRecommendationKpiDailyBudget.class.isAssignableFrom(kpi.getClass())) {
            return ((GdRecommendationKpiDailyBudget) kpi).getName();
        }

        if (GdRecommendationKpiSwitchOnAutotargeting.class.isAssignableFrom(kpi.getClass())) {
            return ((GdRecommendationKpiSwitchOnAutotargeting) kpi).getName();
        }

        if (GdRecommendationKpiWeeklyBudget.class.isAssignableFrom(kpi.getClass())) {
            return ((GdRecommendationKpiWeeklyBudget) kpi).getName();
        }

        if (GdRecommendationKpiRemovePagesFromBlackList.class.isAssignableFrom(kpi.getClass())) {
            return ((GdRecommendationKpiRemovePagesFromBlackList) kpi).getPageName();
        }

        return "";
    }

    private static GdRecommendationStatus getTotalStatus(List<GdRecommendation> gdRecommendations) {
        // если хотя бы одна рекомендация данного типа 'in_progress',
        // то суммарный статус тоже 'in_progress',
        // иначе суммарный статус - 'ready'
        return gdRecommendations.stream()
                .map(GdRecommendation::getStatus)
                .filter(GdRecommendationStatus.IN_PROGRESS::equals)
                .findFirst()
                .orElse(GdRecommendationStatus.READY);
    }

    private static GdRecommendationKpi getTotalKpi(List<GdRecommendation> gdRecommendations) {
        final List<GdRecommendationKpiGraphPoint> totalPoints = gdRecommendations.stream()
                .filter(r -> GdRecommendationWithKpi.class.isAssignableFrom(r.getClass()))
                .map(GdRecommendationWithKpi.class::cast)
                .map(GdRecommendationWithKpi::getKpi)
                .filter(Objects::nonNull)
                .map(GdRecommendationKpi::getGraph)
                .filter(Objects::nonNull)
                .map(GdRecommendationKpiGraph::getPoints)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .map(GdRecommendationKpiGraphPoint::copy)
                .collect(toMap(GdRecommendationKpiGraphPoint::getValueX, identity(),
                        (p1, p2) -> p1.withCurrentValue(p1.getCurrentValue().add(p2.getCurrentValue()))
                                .withProfitValue(p1.getProfitValue().add(p2.getProfitValue()))))
                .values().stream()
                .map(p -> p
                        .withCurrentValue(p.getCurrentValue().setScale(0, RoundingMode.HALF_UP))
                        .withProfitValue(p.getProfitValue().setScale(0, RoundingMode.HALF_UP))
                )
                .sorted(Comparator.comparing(GdRecommendationKpiGraphPoint::getValueX))
                .collect(toList());

        return gdRecommendations.stream()
                .filter(r -> GdRecommendationWithKpi.class.isAssignableFrom(r.getClass()))
                .map(GdRecommendationWithKpi.class::cast)
                .map(GdRecommendationWithKpi::getKpi)
                .filter(Objects::nonNull)
                .map(GdRecommendationKpi::copy)
                .map(GdRecommendationKpi.class::cast)
                .reduce((kpi1, kpi2) -> kpi1
                        .withClicks(kpi1.getClicks().add(kpi2.getClicks()))
                        .withGoals(kpi1.getGoals().add(kpi2.getGoals()))
                        .withCost(kpi1.getCost().add(kpi2.getCost()))
                )
                .map(k -> k.withClicks(k.getClicks().setScale(0, RoundingMode.UP)))
                .map(t -> t.withGraph(t.getGraph().copy().withPoints(totalPoints)))
                .orElse(null);
    }

    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;
    private final GridRecommendationYtRepository recommendationYtRepository;
    private final FeatureService featureService;
    private final RbacService rbacService;
    private final GridOutdoorVideoRecommendationForAdGroupService gridOutdoorVideoRecommendationForAdGroupService;
    private final GridOutdoorVideoRecommendationForBannersService gridOutdoorVideoRecommendationForBannersService;
    private final GridOutdoorVideoRecommendationForPlacementsService gridOutdoorVideoRecommendationForPlacementsService;
    private final GridBannerFormatsForPriceSalesRecommendationService gridBannerFormatsForPriceSalesRecommendationService;
    private final WalletPaymentTransactionsRepository walletPaymentTransactionsRepository;
    private final WalletRepository walletRepository;
    private final ClientService clientService;
    private final CampDayBudgetStopHistoryRepository campDayBudgetStopHistoryRepository;
    private final AutopayService autopayService;
    private final CommunicationChannelService communicationChannelService;

    @Autowired
    public GridRecommendationService(
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository, ShardHelper shardHelper,
            GridRecommendationYtRepository recommendationYtRepository,
            FeatureService featureService, RbacService rbacService,
            GridOutdoorVideoRecommendationForAdGroupService gridOutdoorVideoRecommendationForAdGroupService,
            GridOutdoorVideoRecommendationForBannersService gridOutdoorVideoRecommendationForBannersService,
            GridOutdoorVideoRecommendationForPlacementsService gridOutdoorVideoRecommendationForPlacementsService,
            GridBannerFormatsForPriceSalesRecommendationService gridBannerFormatsForPriceSalesRecommendationService,
            WalletPaymentTransactionsRepository walletPaymentTransactionsRepository, WalletRepository walletRepository,
            ClientService clientService, CampDayBudgetStopHistoryRepository campDayBudgetStopHistoryRepository,
            AutopayService autopayService, CommunicationChannelService communicationChannelService) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.recommendationYtRepository = recommendationYtRepository;
        this.rbacService = rbacService;
        this.gridOutdoorVideoRecommendationForAdGroupService = gridOutdoorVideoRecommendationForAdGroupService;
        this.gridOutdoorVideoRecommendationForBannersService = gridOutdoorVideoRecommendationForBannersService;
        this.gridOutdoorVideoRecommendationForPlacementsService = gridOutdoorVideoRecommendationForPlacementsService;
        this.gridBannerFormatsForPriceSalesRecommendationService = gridBannerFormatsForPriceSalesRecommendationService;
        this.walletPaymentTransactionsRepository = walletPaymentTransactionsRepository;
        this.clientService = clientService;
        this.walletRepository = walletRepository;
        this.campDayBudgetStopHistoryRepository = campDayBudgetStopHistoryRepository;
        this.autopayService = autopayService;
        this.communicationChannelService = communicationChannelService;
    }

    public List<GdiRecommendation> getAvailableRecommendations(Long clientId,
                                                               @Nullable User operator,
                                                               @Nullable Collection<GdiRecommendationType> types,
                                                               @Nullable Collection<Long> campIds) {
        return getAvailableRecommendations(clientId, operator, types, campIds, null, null, null, null, null);
    }

    public List<GdiRecommendation> getAvailableRecommendations(Long clientId,
                                                               @Nullable User operator,
                                                               @Nullable Collection<GdiRecommendationType> types,
                                                               @Nullable Collection<Long> campIds,
                                                               @Nullable Collection<Long> groupIds,
                                                               @Nullable Collection<Long> bannerIds,
                                                               @Nullable Collection<String> userKeys1,
                                                               @Nullable Collection<String> userKeys2,
                                                               @Nullable Collection<String> userKeys3) {
        try (TraceProfile ignore = Trace.current().profile("recommendations:service:get")) {
            Set<GdiRecommendationType> filterTypes;

            ClientId client = ClientId.fromLong(clientId);
            int shard = shardHelper.getShardByClientIdStrictly(client);

            // [todo] Проверку на оператора (возможно!) нужно убрать будет после того как откроем рекомендации на
            //  всех клиентов.
            if (operator != null && UserUtil.hasOneOfRoles(operator, SUPER, SUPERREADER, PLACER, MANAGER)) {
                filterTypes = EnumSet.allOf(GdiRecommendationType.class);
            } else {
                Set<String> availableFeatures = featureService.getEnabledForClientId(client);

                filterTypes = EnumSet.allOf(GdiRecommendationType.class).stream()
                        .filter(type -> type.getFeature() == null || availableFeatures.contains(type.getFeature().getName()))
                        .collect(toSet());
            }

            if (types != null && !types.isEmpty()) {
                filterTypes.retainAll(types);
            }

            List<GdiRecommendation> recommendations = new ArrayList<>();
            try (TraceProfile ignoreMe = Trace.current().profile("recommendations:ytRepository:get")) {
                List<GdiRecommendation> recommendationsFromYt = recommendationYtRepository
                        .getRecommendations(client.asLong(), filterTypes, campIds, groupIds, bannerIds,
                                userKeys1, userKeys2, userKeys3);
                recommendations.addAll(recommendationsFromYt);
            }

            try (TraceProfile ignoreMe = Trace.current().profile("recommendations:outdoorVideoRecommendations")) {
                var outdoorVideoRecommendations =
                        getOutdoorVideoRecommendations(shard, clientId, filterTypes, campIds, groupIds);
                recommendations.addAll(outdoorVideoRecommendations);
            }

            try (TraceProfile ignoreMe = Trace.current().profile("recommendations:priceSalesRecommendations")) {
                var priceSalesRecommendations = getPriceSalesRecommendations(shard, client, filterTypes, campIds);
                recommendations.addAll(priceSalesRecommendations);
            }

            Set<Long> campaignIds = recommendations.stream()
                    .map(GdiRecommendation::getCid)
                    .filter(Objects::nonNull)
                    .collect(toSet());

            Set<Long> archivedCampaignIds = campaignRepository.getArchivedCampaigns(shard, campaignIds);
            campaignIds.removeAll(archivedCampaignIds);

            Set<Long> universalCampaignIds = campaignRepository.getUniversalCampaigns(shard, campaignIds);

            final Set<Long> writableCampaignIds;
            if (operator != null) {
                campaignIds.retainAll(rbacService.getVisibleCampaigns(operator.getUid(), campaignIds));
                writableCampaignIds = rbacService.getWritableCampaigns(operator.getUid(), campaignIds);
            } else {
                writableCampaignIds = null;
            }

            //Отбрасываем недоступные кампании и заполняем поле "IsApplicable" - право оператора выполнять рекомендацию
            return recommendations.stream()
                    .filter(rec -> rec.getCid() == null || rec.getCid() == 0 || (campaignIds.contains(rec.getCid())
                            && (!universalCampaignIds.contains(rec.getCid())
                                    || ALLOWED_UNIVERSAL_CAMPAIGN_RECOMMENDATION_TYPES.contains(rec.getType()))
                    ))
                    .peek(rec -> rec.setIsApplicable(rec.getCid() == null || rec.getCid() == 0
                            || writableCampaignIds == null || writableCampaignIds.contains(rec.getCid())))
                    .collect(toList());
        }
    }

    /**
     * Получить рекомендации про полноту прайсовых кампаний
     */
    List<GdiRecommendation> getPriceSalesRecommendations(int shard, ClientId client,
                                                         Set<GdiRecommendationType> types,
                                                         @Nullable Collection<Long> campIds) {
        if (isEmpty(campIds)) {
            return emptyList();
        }

        List<GdiRecommendation> result = new ArrayList<>();

        if (types.contains(GdiRecommendationType.addBannerFormatsForPriceSalesCorrectness)) {
            result.addAll(gridBannerFormatsForPriceSalesRecommendationService
                    .getRecommendationsForCampaigns(shard, client, campIds));
        }
        if (types.contains(GdiRecommendationType.addBannerFormatsForPriceSalesCorrectnessInParentCampaign)) {
            result.addAll(gridBannerFormatsForPriceSalesRecommendationService
                    .getRecommendationsForAdGroups(shard, client, campIds));
        }

        return result;
    }

    /**
     * Получить OUTDOOR_VIDEO_RECOMMENDATIONS_TYPES рекомендации
     */
    List<GdiRecommendation> getOutdoorVideoRecommendations(int shard, Long clientId,
                                                           Set<GdiRecommendationType> types,
                                                           @Nullable Collection<Long> campIds,
                                                           @Nullable Collection<Long> adGroupIds) {

        if (Collections.disjoint(OUTDOOR_VIDEO_RECOMMENDATIONS_TYPES, types) || (isEmpty(adGroupIds) && isEmpty(campIds))) {
            return emptyList();
        }

        Map<Long, Long> campaignIdsByOutdoorAdGroupIds;
        if (!isEmpty(adGroupIds)) {
            campaignIdsByOutdoorAdGroupIds =
                    adGroupRepository.getAdGroupSimple(shard, ClientId.fromLong(clientId), adGroupIds)
                            .values()
                            .stream()
                            .filter(x -> x.getType().equals(CPM_OUTDOOR))
                            .collect(toMap(AdGroupSimple::getId, AdGroupSimple::getCampaignId));
        } else {
            campaignIdsByOutdoorAdGroupIds =
                    adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, campIds)
                            .values()
                            .stream()
                            .flatMap(Collection::stream)
                            .filter(x -> x.getType().equals(CPM_OUTDOOR))
                            .collect(toMap(AdGroupSimple::getId, AdGroupSimple::getCampaignId));
        }

        List<GdiRecommendation> result = new ArrayList<>();

        if (types.contains(GdiRecommendationType.uploadAppropriateCreatives)) {
            List<GdiRecommendation> recommendations =
                    gridOutdoorVideoRecommendationForPlacementsService.getRecommendations(shard, clientId,
                            campaignIdsByOutdoorAdGroupIds);

            result.addAll(recommendations);
        }

        if (types.contains(GdiRecommendationType.chooseAppropriatePlacementsForAdGroup)) {
            List<GdiRecommendation> recommendations =
                    gridOutdoorVideoRecommendationForAdGroupService.getRecommendations(shard, clientId,
                            campaignIdsByOutdoorAdGroupIds);

            result.addAll(recommendations);
        }

        if (types.contains(GdiRecommendationType.chooseAppropriatePlacementsForBanner)) {
            List<GdiRecommendation> recommendations =
                    gridOutdoorVideoRecommendationForBannersService.getRecommendations(shard, clientId,
                            campaignIdsByOutdoorAdGroupIds);

            result.addAll(recommendations);
        }
        return result;
    }

    /*
     * Получает статусы рекомендаций по ключам из таблицы "RECOMMENDATIONS_STATUS" без учета статуса "IN_PROGRESS"
     *
     * @param clientId идентификатор клиента
     * @param keys     ключи рекомендаций
     */
    public List<GdiRecommendation> getRecommendationsStatuses(Long clientId, Collection<RecommendationKey> keys) {
        return recommendationYtRepository.getRecommendationsStatuses(clientId, keys);
    }

    public Map<Long, List<GdRecommendation>> getBannerRecommendations(Long clientId,
                                                                      User operator,
                                                                      Set<GdiRecommendationType> recommendationTypes,
                                                                      Collection<Long> campaignIds) {
        final Set<GdiRecommendationType> validTypes = recommendationTypes
                .stream()
                .filter(ALL_ALLOWED_BANNER_RECOMMENDATION_TYPES::contains)
                .collect(toSet());

        final List<GdiRecommendation> recommendations = getAvailableRecommendations(clientId, operator,
                validTypes.isEmpty() ? ALL_ALLOWED_BANNER_RECOMMENDATION_TYPES : validTypes,
                campaignIds);

        return recommendations.stream()
                .collect(groupingBy(
                        GdiRecommendation::getBid, mapping(
                                GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION, toList())));
    }

    public List<GdRecommendationSummary> getRecommendationSummary(Long clientId,
                                                                  User operator,
                                                                  @Nullable Set<GdiRecommendationType> types) {
        final Set<GdiRecommendationType> validTypes = Optional.ofNullable(types)
                .filter(t -> !t.isEmpty())
                .orElse(ALL_ALLOWED_SUMMARY_RECOMMENDATION_TYPES)
                .stream()
                .filter(ALL_ALLOWED_SUMMARY_RECOMMENDATION_TYPES::contains)
                .collect(toSet());

        boolean disableOldDailyBudgetRecommendation = featureService.isEnabledForClientId(ClientId.fromLong(clientId),
                FeatureName.DISABLE_OLD_DAILY_BUDGET_RECOMMENDATION);
        boolean disableOldWeeklyBudgetRecommendation = featureService.isEnabledForClientId(ClientId.fromLong(clientId),
                FeatureName.DISABLE_OLD_WEEKLY_BUDGET_RECOMMENDATION);
        var oldDisabledTypes = new HashSet<GdiRecommendationType>();
        var newRecommendations = List.<GdRecommendation>of();
        if (disableOldDailyBudgetRecommendation || disableOldWeeklyBudgetRecommendation) {
            if (disableOldDailyBudgetRecommendation) {
                oldDisabledTypes.add(dailyBudget);
            }

            if (disableOldWeeklyBudgetRecommendation){
                oldDisabledTypes.add(increaseStrategyWeeklyBudget);
            }
            boolean enableNewDailyBudgetRecommendation = isNewRecommendationEnabled(clientId,
                    FeatureName.ENABLE_NEW_DAILY_BUDGET_RECOMMENDATION, disableOldDailyBudgetRecommendation,
                    validTypes, dailyBudget);
            boolean enableNewWeeklyBudgetRecommendation = isNewRecommendationEnabled(clientId,
                    FeatureName.ENABLE_NEW_WEEKLY_BUDGET_RECOMMENDATION, disableOldWeeklyBudgetRecommendation,
                    validTypes, increaseStrategyWeeklyBudget);
            if (enableNewDailyBudgetRecommendation || enableNewWeeklyBudgetRecommendation) {
                var client = ClientId.fromLong(clientId);
                var shard = shardHelper.getShardByClientId(client);
                var campaigns = listToMap(campaignRepository.getCampaigns(shard, new CampaignsSelectionCriteria()
                                .withClientIds(List.of(client))
                                .withStatusArchived(false)
                                .withCampaignTypes(WEB_EDIT)),
                        Campaign::getId);
                List<Long> campaignIds = new ArrayList<>(campaigns.keySet());
                campaignIds.retainAll(rbacService.getVisibleCampaigns(operator.getUid(), campaignIds));
                var writableCampaignIds = rbacService.getWritableCampaigns(operator.getUid(), campaignIds);
                try {
                    newRecommendations = communicationChannelService.getCommunicationMessage(
                            ClientId.fromLong(clientId), operator.getUid(),
                            Set.of("recommendation_dashboard_old"), new ArrayList<>(campaigns.keySet()), "ru")
                            .stream()
                            .filter(m -> isMessageWithGridRecommendations(m, enableNewDailyBudgetRecommendation,
                                    enableNewWeeklyBudgetRecommendation))
                            .map(r -> {
                                var cid = r.getTargetObject().getId();
                                return toRecommendationWithKpi(r, campaigns.get(cid), writableCampaignIds.contains(cid));
                            })
                            .collect(toList());
                } catch (InterruptedRuntimeException ex) {
                    Thread.currentThread().interrupt();
                    throw ex;
                } catch (RuntimeException ex) {
                    logger.error("Error on getting recommendations", ex);
                }
            }
        }

        final Map<GdiRecommendationType, List<GdRecommendation>> recommendationsByType = StreamEx.of(
                getAvailableRecommendations(clientId, operator, validTypes, null))
                .filter(r -> !oldDisabledTypes.contains(r.getType()))
                .map(GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION)
                .append(newRecommendations)
                .collect(groupingBy(GdRecommendation::getType, toList()));

        // рекомендациям по площадкам требуется особая предагрегация
        if (validTypes.contains(GdiRecommendationType.removePagesFromBlackList)) {
            final List<GdRecommendation> pageRecommendations = this.getPageRecommendationsGroupedByPage(clientId,
                    operator);
            if (!pageRecommendations.isEmpty()) {
                recommendationsByType.put(GdiRecommendationType.removePagesFromBlackList, pageRecommendations);
            }
        }

        return EntryStream.of(recommendationsByType)
                .mapKeyValue(GDI_RECOMMENDATION_TYPE_LIST_GD_RECOMMENDATION_SUMMARY_BI_FUNCTION)
                .append(getRecommendationSummaryCampaignGrid(clientId, types))
                .collect(toList());
    }

    public List<GdRecommendationSummary> getRecommendationSummaryCampaignGrid(Long clientId,
                                                                              @Nullable Set<GdiRecommendationType> types) {
        ClientId client = ClientId.fromLong(clientId);
        List<GdRecommendationSummary> result = new ArrayList<>();

        var shard = shardHelper.getShardByClientId(client);
        CurrencyCode clientCurrencyCode = clientService.getWorkCurrency(client).getCode();
        List<Wallet> wallets = walletRepository.getAllWalletExistingCampByClientId(shard, List.of(client));
        Long walletId = wallets == null || wallets.isEmpty() ? null : wallets.get(0).getWalletCampaignId();

        Set<GdiRecommendationType> validTypes = Optional.ofNullable(types)
                .filter(t -> !t.isEmpty())
                .orElse(ALL_ALLOWED_IN_CAMPAIGN_GRID_RECOMMENDATION_TYPES)
                .stream()
                .filter(ALL_ALLOWED_IN_CAMPAIGN_GRID_RECOMMENDATION_TYPES::contains)
                .collect(toSet());
        if (walletId != null) {
            if (validTypes.contains(GdiRecommendationType.autopayStopped)) {
                addAutopayStoppedRecommendation(shard, clientCurrencyCode, walletId, result);
            }
            if (validTypes.contains(GdiRecommendationType.mainInvoice)) {
                addMainInvoiceRecommendation(shard, clientCurrencyCode, walletId, result);
            }
        }
        if (validTypes.contains(GdiRecommendationType.overdraftDebt)) {
            addOverdraftDebtRecommendation(client, clientCurrencyCode, walletId, result);
        }
        if (validTypes.contains(GdiRecommendationType.videoButton)) {
            addVideoButtonRecommendation(client, result);
        }
        return result;
    }

    private void addVideoButtonRecommendation(ClientId clientId, List<GdRecommendationSummary> result) {
        var features = featureService.getEnabledForClientId(clientId);
        if (!features.contains(FeatureName.GREAT_BUTTON_COMING_SOON.getName())
                && !features.contains(FeatureName.GREAT_BUTTON_ENABLED.getName()))
            return;//не нужно показывать рекомендацию
        var recommendations = recommendationYtRepository.getRecommendations(clientId.asLong(),
                List.of(GdiRecommendationType.videoButton),
                null, null, null, null, null, null);
        for (GdiRecommendation recom : recommendations) {
            result.add(new GdRecommendationSummaryWithoutKpi()
                    .withType(recom.getType())
                    .withStatus(GdRecommendationStatus.READY)
                    .withIsApplicable(false)
                    .withCampaignIds(emptySet())
                    .withKeys(emptyList())
                    .withTotalCount(1)
            );
        }
    }

    private void addOverdraftDebtRecommendation(ClientId clientId, CurrencyCode clientCurrencyCode,
                                                @Nullable Long walletId, List<GdRecommendationSummary> result) {
        ClientsOptions clientsOptions = clientService.getClientOptions(clientId);
        if (!NumberUtils.greaterThanZero(clientsOptions.getDebt())) {
            return;
        }

        var kpi = new GdRecommendationKpiOverdraftDebt()
                .withNextPayDate(clientsOptions.getNextPayDate())
                .withDateFlag(getDateFlag(clientsOptions))
                .withDebt(clientsOptions.getDebt());
        var recommendation = getBaseRecommendationWithKpi(kpi, clientCurrencyCode,
                nvl(ifNotNull(walletId, Set::of), emptySet()),
                GdiRecommendationType.overdraftDebt, null);
        result.add(recommendation);
    }

    private void addAutopayStoppedRecommendation(int shard, CurrencyCode clientCurrencyCode,
                                                 Long walletId, List<GdRecommendationSummary> result) {
        Long triesNum = autopayService.getAutopayTriesNum(shard, walletId);
        if (triesNum == null || triesNum == 0) {
            return;
        }
        String autopayLastTransactionCode = walletPaymentTransactionsRepository.getLastBalanceStatusCode(shard,
                walletId);
        if (triesNum < 0) {
            autopayLastTransactionCode = "Error";
        }

        GdAutopayErrors autopayError = getAutopayError(autopayLastTransactionCode);
        if (autopayError == null) {
            return;
        }

        var kpi = new GdRecommendationKpiAutopayStopped().withAutopayError(autopayError);
        var recommendation = getBaseRecommendationWithKpi(kpi, clientCurrencyCode, Set.of(walletId),
                GdiRecommendationType.autopayStopped, null);
        result.add(recommendation);
    }

    private void addMainInvoiceRecommendation(int shard, CurrencyCode clientCurrencyCode,
                                              Long walletId, List<GdRecommendationSummary> result) {
        LocalDateTime stopDateTime = campDayBudgetStopHistoryRepository
                .getDayBudgetStopHistoryForNotification(shard, walletId, DAILY_BUDGET_STOP_WARNING_PERIOD);
        if (stopDateTime == null) {
            return;
        }
        var yesterday = LocalDate.now().minusDays(1);
        if (stopDateTime.toLocalDate().isBefore(yesterday)
                || !DAILY_BUDGET_STOP_WARNING_TIME_DEFAULT.isAfter(stopDateTime.toLocalTime())) {
            return;
        }

        var kpi = new GdRecommendationKpiMainInvoice()
                .withToday(LocalDateTime.of(LocalDate.now(), LocalTime.MIDNIGHT))
                .withStopTime(stopDateTime);
        var recommendation = getBaseRecommendationWithKpi(kpi, clientCurrencyCode, Set.of(walletId),
                GdiRecommendationType.mainInvoice, (int) DAILY_BUDGET_STOP_WARNING_PERIOD.toDays());
        result.add(recommendation);
    }

    private GdDateFlag getDateFlag(ClientsOptions clientsOptions) {
        var today = LocalDate.now();
        if (clientsOptions.getNextPayDate() == null || clientsOptions.getNextPayDate().isAfter(today)) {
            return GdDateFlag.FUTURE;
        } else if (clientsOptions.getNextPayDate().isBefore(today)) {
            return GdDateFlag.PAST;
        } else {
            return GdDateFlag.PRESENT;
        }
    }

    private static GdAutopayErrors getAutopayError(@Nullable String error) {
        return StringUtils.ifNotBlank(error, s -> {
            //noinspection ConstantConditions
            switch (error) {
                case AUTOPAY_NOT_ENOUGH_FUNDS_ERROR:
                    return GdAutopayErrors.NOT_ENOUGH_FUNDS;
                case AUTOPAY_EXPIRED_CARD_ERROR:
                    return GdAutopayErrors.EXPIRED_CARD;
                default:
                    return GdAutopayErrors.OTHER;
            }
        });

    }

    private static GdRecommendationSummaryWithKpi getBaseRecommendationWithKpi(GdRecommendationKpi recommendationKpi,
                                                                               CurrencyCode currencyCode,
                                                                               Set<Long> campaignIds,
                                                                               GdiRecommendationType type,
                                                                               @Nullable Integer period) {
        return new GdRecommendationSummaryWithKpi()
                .withKpi(recommendationKpi
                        .withCurrency(currencyCode)
                        .withClicks(ZERO)
                        .withCost(ZERO)
                        .withGraph(new GdRecommendationKpiGraph()
                                .withPoints(emptyList())
                                .withTypeX("")
                                .withTypeY(""))
                        .withPeriod(period == null ? DEFAULT_RECOMMENDATION_PERIOD : period)
                        .withSign(GdRecommendationSign.CPC)
                )
                .withItems(new GdRecommendationItems()
                        .withTotalCount(1)
                        .withRowset(List.of(new GdRecommendationItem()
                                .withName("")
                                .withClicks(0L)
                                .withCost(ZERO)
                                .withCurrency(currencyCode)
                                .withGoals(0L))))
                .withCampaignIds(campaignIds)
                .withIsApplicable(true)
                .withKeys(emptyList())
                .withStatus(GdRecommendationStatus.READY)
                .withTotalCount(1)
                .withType(type);
    }

    public Map<Long, List<GdRecommendation>> getCampaignRecommendations(
            Long clientId, User operator,
            Set<GdiRecommendationType> recommendationTypes,
            Set<Long> campaignIds) {
        var oldRecommendations = getCampaignOldRecommendations(
                clientId, operator, recommendationTypes, campaignIds);
        boolean disableOldDailyBudgetRecommendation = featureService.isEnabledForClientId(ClientId.fromLong(clientId),
                FeatureName.DISABLE_OLD_DAILY_BUDGET_RECOMMENDATION);
        boolean disableOldWeeklyBudgetRecommendation = featureService.isEnabledForClientId(ClientId.fromLong(clientId),
                FeatureName.DISABLE_OLD_WEEKLY_BUDGET_RECOMMENDATION);
        if (!disableOldDailyBudgetRecommendation && !disableOldWeeklyBudgetRecommendation) {
            return oldRecommendations;
        }

        Predicate<GdRecommendation> filterAllExceptDailyRecommendations =
                r -> !disableOldDailyBudgetRecommendation || !dailyBudget.equals(r.getType());
        Predicate<GdRecommendation> filterAllExceptWeeklyRecommendations =
                r -> !disableOldWeeklyBudgetRecommendation || !increaseStrategyWeeklyBudget.equals(r.getType());
        var filteredResult = EntryStream.of(oldRecommendations)
                .mapValues(list -> filterList(list,
                        filterAllExceptDailyRecommendations.and(filterAllExceptWeeklyRecommendations)))
                .filterValues(list -> !list.isEmpty())
                .mapValues(list -> (List<GdRecommendation>) new ArrayList<>(list))
                .toMap();
        var enableNewDailyBudgetRecommendation = isNewRecommendationEnabled(clientId,
                FeatureName.ENABLE_NEW_DAILY_BUDGET_RECOMMENDATION, disableOldDailyBudgetRecommendation,
                recommendationTypes, dailyBudget);
        var enableNewWeeklyBudgetRecommendation = isNewRecommendationEnabled(clientId,
                FeatureName.ENABLE_NEW_WEEKLY_BUDGET_RECOMMENDATION, disableOldWeeklyBudgetRecommendation,
                recommendationTypes, increaseStrategyWeeklyBudget);
        if (!enableNewDailyBudgetRecommendation && !enableNewWeeklyBudgetRecommendation) {
            return filteredResult;
        }
        var shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
        var campaigns = listToMap(campaignRepository.getCampaigns(shard, new CampaignsSelectionCriteria()
                        .withClientIds(List.of(ClientId.fromLong(clientId)))
                        .withStatusArchived(false)
                        .withCampaignTypes(WEB_EDIT)),
                Campaign::getId);
        var writableCampaignIds = rbacService.getWritableCampaigns(operator.getUid(), campaignIds);
        try {
            communicationChannelService.getCommunicationMessage(
                    ClientId.fromLong(clientId), operator.getUid(),
                    Set.of("camp_grid"), new ArrayList<>(campaignIds), "ru"
            ).stream()
                    .filter(m -> isMessageWithGridRecommendations(m, enableNewDailyBudgetRecommendation,
                            enableNewWeeklyBudgetRecommendation))
                    .forEach(msg -> {
                        var cid = msg.getTargetObject().getId();
                        var rec = toRecommendationWithKpi(msg, campaigns.get(cid),
                                writableCampaignIds.contains(cid));
                        if (!filteredResult.containsKey(cid)) {
                            filteredResult.put(cid, new ArrayList<>());
                        }
                        filteredResult.get(cid).add(rec);
                    });
        } catch (InterruptedRuntimeException ex) {
            Thread.currentThread().interrupt();
            throw ex;
        } catch (RuntimeException ex) {
            logger.error("Error on getting recommendations", ex);
        }
        return filteredResult;
    }


    /**
     * @return новая рекомендация доступна только если соблюдены оба условия:
     * - отключены старые рекомендации такого же типа (иначе они могут противоречить новым)
     * - в запросе за рекомендациями не задан конкретный их тип (то есть нужны рекомендации всех типов)
     * либо в запросе присутствует тип новой рекомендации, иначе будут ошибки на фронте
     */
    private boolean isNewRecommendationEnabled(Long clientId, FeatureName featureName,
                                               boolean isOldRecommendationDisabled,
                                               Set<GdiRecommendationType> requestedTypes,
                                               GdiRecommendationType targetType){
        return isOldRecommendationDisabled &&
                featureService.isEnabledForClientId(ClientId.fromLong(clientId), featureName) &&
                (requestedTypes == null || requestedTypes.isEmpty() || requestedTypes.contains(targetType));
    }

    private GdRecommendationWithKpi toRecommendationWithKpi(CommunicationMessage message,
                                                            @Nullable Campaign campaign,
                                                            boolean isApplicable){
        GdiRecommendationType recommendationType;
        GdRecommendationKpi recommendationKpi;
        switch(message.getName()){
            case DAILY_BUDGET_RECOMMENDATION_MESSAGE_NAME:
                recommendationType = dailyBudget;
                recommendationKpi = toGdRecommendationKpiDailyBudget(message, campaign);
                break;
            case WEEKLY_BUDGET_RECOMMENDATION_MESSAGE_NAME:
                recommendationType = increaseStrategyWeeklyBudget;
                recommendationKpi = toGdRecommendationKpiWeeklyBudget(message, campaign);
                break;
            default:
                throw new IllegalArgumentException(String.format("Unsupported message name %s", message.getName()));
        }

        var eventKey = String.format("%d-%d",
                message.getEventId(),
                message.getEventVersionId());
        var dataKey = String.format("%d-%d",
                message.getMajorVersion(),
                message.getMinorVersion());
        var reqSlotKey = message.getRequestId() + "-" + message.getSlot().getName();
        var recommendation = new GdRecommendationWithKpi()
                .withIsApplicable(isApplicable)
                .withStatus(GdRecommendationStatus.READY)
                .withType(recommendationType)
                .withKeys(List.of(new GdRecommendationKey()
                        .withType(recommendationType)
                        .withCid(message.getTargetObject().getId())
                        .withPid(0L)
                        .withBid(0L)
                        .withUserKey1(eventKey)
                        .withUserKey2(dataKey)
                        .withUserKey3(reqSlotKey)
                        .withTimestamp(System.currentTimeMillis() / 1000)
                ))
                .withKpi(recommendationKpi);

        return recommendation;
    }

    private GdRecommendationKpi toGdRecommendationKpiDailyBudget(CommunicationMessage message,
                                                                 @Nullable Campaign campaign) {
        var messageData = message.getData();
        return new GdRecommendationKpiDailyBudget()
                        .withName(campaign == null ? "Campaign" : campaign.getName())
                        .withPeriod(7)
                        .withCurrency(((Currency) messageData.get("client.currency")).getCode())
                        .withCurrentDailyBudget(campaign == null ? ZERO : campaign.getDayBudget())
                        .withGoals(ZERO)
                        .withClicks(BigDecimal.valueOf((Long) messageData.get("clicks_increase")))
                        .withCost((BigDecimal) messageData.get("budget_spendings_increase"))
                        .withRecommendedDailyBudget((BigDecimal) messageData.get("new_daily_budget"))
                        .withSign(GdRecommendationSign.CPC)
                        .withGraph(new GdRecommendationKpiGraph()
                                .withTypeX("date")
                                .withTypeY("clicks")
                                .withPoints(List.of())
                        );
    }

    private GdRecommendationKpi toGdRecommendationKpiWeeklyBudget(CommunicationMessage message,
                                                                  @Nullable Campaign campaign) {
        var messageData = message.getData();
        var recommendedWeeklyBudget = (BigDecimal) messageData.get("new_week_budget");
        var currentWeeklyBudget = campaign == null ? ZERO : campaign.getStrategy().getStrategyData().getSum();

        return new GdRecommendationKpiWeeklyBudget()
                .withName(campaign == null ? "Campaign" : campaign.getName())
                .withPeriod(7)
                .withCurrency(((Currency) messageData.get("client.currency")).getCode())
                .withCurrentWeeklyBudget(currentWeeklyBudget)
                .withRecommendedWeeklyBudget(recommendedWeeklyBudget)
                .withDiffWeeklyBudget(recommendedWeeklyBudget == null || currentWeeklyBudget == null ? null :
                        recommendedWeeklyBudget.subtract(currentWeeklyBudget))
                .withCurrentTargetCPA(campaign == null ? null : campaign.getStrategy().getStrategyData().getAvgCpa())
                .withCurrentTargetROI(campaign == null ? null : campaign.getStrategy().getStrategyData().getRoiCoef())
                .withGoals(BigDecimal.valueOf((Long) messageData.get("conversions_increase")))
                .withClicks(ZERO)
                .withCost((BigDecimal) messageData.get("budget_spendings_increase"))
                .withRecommendedTargetCPA(campaign == null ? null : campaign.getStrategy().getStrategyData().getAvgCpa())
                .withRecommendedTargetROI(campaign == null ? null : campaign.getStrategy().getStrategyData().getRoiCoef())
                .withSign(GdRecommendationSign.CPC)
                .withGraph(new GdRecommendationKpiGraph()
                        .withTypeX("date")
                        .withTypeY("ctr")
                        .withPoints(List.of())
                );
    }

    private Map<Long, List<GdRecommendation>> getCampaignOldRecommendations(Long clientId,
                                                                        User operator,
                                                                        Set<GdiRecommendationType> recommendationTypes,
                                                                        Set<Long> campaignIds) {

        if (recommendationTypes.isEmpty() || recommendationTypes.stream()
                .anyMatch(ALL_ALLOWED_CAMPAIGN_RECOMMENDATION_TYPES::contains)) {
            final Set<GdiRecommendationType> validTypes =
                    recommendationTypes.stream()
                            .filter(ALL_ALLOWED_CAMPAIGN_RECOMMENDATION_TYPES::contains)
                            .collect(toSet());

            final List<GdiRecommendation> recommendations =
                    getAvailableRecommendations(clientId,
                            operator, validTypes.isEmpty() ? ALL_ALLOWED_CAMPAIGN_RECOMMENDATION_TYPES : validTypes,
                            campaignIds);

            // рекомендациям по площадкам требуется особая предагрегация
            final List<GdiRecommendation> pageRecommendations = recommendations.stream()
                    .filter(r -> GdiRecommendationType.removePagesFromBlackListOfACampaign.equals(r.getType()))
                    .collect(toList());

            final Map<Long, List<GdRecommendation>> pageRecommendationsByCid = pageRecommendations.isEmpty()
                    ? emptyMap() : this.getPageRecommendationsGroupedByCampaign(pageRecommendations);

            final Map<Long, List<GdRecommendation>> otherRecommendationsByCid = recommendations.stream()
                    .filter(r -> !GdiRecommendationType.removePagesFromBlackListOfACampaign.equals(r.getType()))
                    .collect(groupingBy(
                            GdiRecommendation::getCid, mapping(
                                    GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION, toList())));

            return EntryStream.of(otherRecommendationsByCid)
                    .append(pageRecommendationsByCid)
                    .toMap((list1, list2) -> {
                        list1.addAll(list2);
                        return list1;
                    });
        } else {
            Set<Long> availableCampaignIds =
                    getAvailableRecommendations(clientId, operator, recommendationTypes, null)
                            .stream()
                            .map(GdiRecommendation::getCid)
                            .collect(toSet());
            return listToMap(availableCampaignIds, identity(), r -> emptyList());
        }
    }

    public List<GdRecommendation> getPageRecommendationsGroupedByPage(Long clientId, User operator) {
        final List<GdiRecommendation> recommendations =
                getAvailableRecommendations(clientId, operator,
                        singleton(GdiRecommendationType.removePagesFromBlackListOfACampaign), null);

        final Map<String, List<GdRecommendation>> recommendationsGroupedByPage = recommendations.stream()
                // group by page name which is in user_key_1
                .collect(groupingBy(GdiRecommendation::getUserKey1,
                        mapping(GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION, toList())));

        return EntryStream.of(recommendationsGroupedByPage)
                .mapKeyValue((pageName, rList) -> {
                    final List<GdRecommendationKey> keys = rList.stream()
                            .map(GdRecommendation::getKeys)
                            .flatMap(Collection::stream)
                            .collect(toList());

                    final GdRecommendationKpi totalKpi = getTotalKpi(rList);

                    // если хотя бы одна рекомендация данного типа 'isApplicable',
                    // то суммарный isApplicable = true,
                    // иначе суммарная возможность применения - false
                    final boolean isApplicable = rList.stream()
                            .anyMatch(GdRecommendation::getIsApplicable);

                    return new GdRecommendationWithKpi()
                            .withType(GdiRecommendationType.removePagesFromBlackList)
                            .withStatus(getTotalStatus(rList))
                            .withIsApplicable(isApplicable)
                            .withKpi(new GdRecommendationKpiRemovePagesFromBlackList()
                                    .withPageName(pageName)
                                    .withGraph(totalKpi.getGraph())
                                    .withClicks(totalKpi.getClicks())
                                    .withGoals(ZERO)
                                    .withCost(totalKpi.getCost())
                                    .withCurrency(totalKpi.getCurrency())
                                    .withPeriod(totalKpi.getPeriod())
                                    .withSign(totalKpi.getSign()))
                            .withKeys(keys);
                })
                .collect(toList());
    }

    private Map<Long, List<GdRecommendation>> getPageRecommendationsGroupedByCampaign(
            List<GdiRecommendation> recommendations) {

        final Map<Long, List<GdRecommendation>> recommendationsGroupedByPage = recommendations.stream()
                // group by campaign id
                .collect(groupingBy(GdiRecommendation::getCid,
                        mapping(GDI_RECOMMENDATION_GD_RECOMMENDATION_FUNCTION, toList())));

        return EntryStream.of(recommendationsGroupedByPage)
                .mapValues(rList -> {
                    final List<GdRecommendationKey> keys = rList.stream()
                            .map(GdRecommendation::getKeys)
                            .flatMap(Collection::stream)
                            .collect(toList());

                    final GdRecommendationKpi totalKpi = getTotalKpi(rList);

                    // если хотя бы одна рекомендация данного типа 'isApplicable',
                    // то суммарный isApplicable = true,
                    // иначе суммарная возможность применения - false
                    final boolean isApplicable = rList.stream()
                            .anyMatch(GdRecommendation::getIsApplicable);

                    final GdRecommendation recommendation = new GdRecommendationWithKpi()
                            .withType(GdiRecommendationType.removePagesFromBlackListOfACampaign)
                            .withStatus(getTotalStatus(rList))
                            .withIsApplicable(isApplicable)
                            .withKpi(new GdRecommendationKpiRemovePagesFromBlackListOfACampaign()
                                    .withPageNames(keys.stream().map(GdRecommendationKey::getUserKey1).collect(toSet()))
                                    .withGraph(totalKpi.getGraph())
                                    .withClicks(totalKpi.getClicks())
                                    .withCost(totalKpi.getCost())
                                    .withCurrency(totalKpi.getCurrency())
                                    .withPeriod(totalKpi.getPeriod())
                                    .withSign(totalKpi.getSign()))
                            .withKeys(keys);

                    return Stream.of(recommendation).collect(toList());
                })
                .toMap();
    }

    public Map<Long, List<GdiRecommendation>> getGroupRecommendations(ClientId clientId, User operator,
                                                                      Set<GdiRecommendationType> types,
                                                                      Collection<Long> campaignIds) {
        final Set<GdiRecommendationType> recommendationTypes = types.stream()
                .filter(ALL_ALLOWED_GROUP_RECOMMENDATION_TYPES::contains)
                .collect(toSet());

        final List<GdiRecommendation> recommendations =
                getAvailableRecommendations(clientId.asLong(), operator,
                        recommendationTypes.isEmpty() ? ALL_ALLOWED_GROUP_RECOMMENDATION_TYPES : recommendationTypes,
                        campaignIds);

        return recommendations.stream()
                .collect(groupingBy(GdiRecommendation::getPid));
    }


    /**
     * Эти поля должны быть удалены в DIRECT-101841
     */
    public static void addUnnecessaryFields(GdRecommendationKpi kpi) {
        kpi
                .withClicks(BigDecimal.ZERO)
                .withCost(BigDecimal.ZERO)
                .withCurrency(CurrencyCode.RUB)
                .withGraph(new GdRecommendationKpiGraph().withPoints(emptyList()).withTypeX("").withTypeY(""))
                .withPeriod(0)
                .withSign(GdRecommendationSign.CPA);
    }

    private static boolean isMessageWithGridRecommendations(CommunicationMessage m,
                                                            boolean enableNewDailyBudgetRecommendation,
                                                            boolean enableNewWeeklyBudgetRecommendation){
        return (enableNewDailyBudgetRecommendation && DAILY_BUDGET_RECOMMENDATION_MESSAGE_NAME.equals(m.getName())) ||
                (enableNewWeeklyBudgetRecommendation && WEEKLY_BUDGET_RECOMMENDATION_MESSAGE_NAME.equals(m.getName()));
    }
}
