package ru.yandex.direct.grid.core.entity.recommendation.service.cpmprice;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSize;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSizeWithExpand;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.model.recommendation.GdPricePackageRecommendationFormat;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiAddBannerFormatsForPriceSalesCorrectness;
import ru.yandex.direct.utils.Checked;

import static java.time.LocalDateTime.now;
import static java.time.ZoneOffset.UTC;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.function.Predicate.not;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultAdGroup;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.collectCampaignCreativeSizesWithExpand;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.addUnnecessaryFields;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.addBannerFormatsForPriceSalesCorrectness;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Рекомендация: "Добавьте креативы всех необходимых форматов" (addBannerFormatsForPriceSalesCorrectness)
 */
@Service
@ParametersAreNonnullByDefault
public class GridBannerFormatsForPriceSalesRecommendationService {

    private static final Logger logger =
            LoggerFactory.getLogger(GridBannerFormatsForPriceSalesRecommendationService.class);
    private static final GdiRecommendationType recommendationType = addBannerFormatsForPriceSalesCorrectness;
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private final BannerTypedRepository bannerTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CreativeRepository creativeRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;
    private final FeatureService featureService;

    @Autowired
    public GridBannerFormatsForPriceSalesRecommendationService(BannerTypedRepository bannerTypedRepository,
                                                               AdGroupRepository adGroupRepository,
                                                               CampaignRepository campaignRepository,
                                                               CreativeRepository creativeRepository,
                                                               CampaignTypedRepository campaignTypedRepository,
                                                               PricePackageRepository pricePackageRepository,
                                                               FeatureService featureService) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.creativeRepository = creativeRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.featureService = featureService;
    }

    /**
     * Получить рекомендации в кампаниях про полноту прайсовых кампаний
     *
     * @param shard       - шард
     * @param clientId    - id клиента
     * @param campaignIds - список id кампаний, для которых нужно получить рекомендацию
     * @return - список рекомендаций
     */
    public List<GdiRecommendation> getRecommendationsForCampaigns(int shard,
                                                                  ClientId clientId,
                                                                  Collection<Long> campaignIds)
    {
        Map<Long, CpmPriceCampaign> campaigns = getCpmPriceCampaigns(shard, campaignIds);
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(
                mapList(campaigns.values(), CpmPriceCampaign::getPricePackageId));


        // Проверка и рекомендация добавить дефолтные группы в кампании
        List<GdiRecommendation> recommendations = new ArrayList<>(
                getRecommendationForDefaultAdGroupForCampaigns(shard, clientId, campaigns, pricePackages)
        );

        return recommendations;
    }

    private List<GdiRecommendation> getRecommendationForDefaultAdGroupForCampaigns(int shard,
            ClientId clientId,
            Map<Long, CpmPriceCampaign> campaigns,
            Map<Long, PricePackage> pricePackages) {
        var cpmYndxFrontpageCampaigns = filterList(campaigns.values(),
                campaign -> pricePackages.get(campaign.getPricePackageId()).isFrontpagePackage());

        Map<Long, List<Long>> campaignIdToAdGroupIds =
                adGroupRepository.getAdGroupIdsByCampaignIds(shard, mapList(cpmYndxFrontpageCampaigns, CpmPriceCampaign::getId));
        Map<Long, Long> campaignIdToDefaultAdGroupId = getCampaignDefaultAdGroupIds(shard, campaignIdToAdGroupIds);

        Map<Long, Set<CreativeSizeWithExpand>> campaignIdToMissingFormats =
                getCampaignMissingFormats(shard,
                        listToMap(cpmYndxFrontpageCampaigns, CpmPriceCampaign::getId), campaignIdToDefaultAdGroupId);

        return EntryStream.of(campaignIdToMissingFormats)
                .filterValues(not(Set::isEmpty))
                .mapKeyValue((campaignId, missingFormats) ->
                        toRecommendation(clientId.asLong(), campaignId, 0L, missingFormats))
                .toList();
    }

    /**
     * Получить рекомендации в группах про полноту прайсовых кампаний
     *
     * @param shard       - шард
     * @param clientId    - id клиента
     * @param campaignIds - список id кампаний, для которых нужно получить рекомендацию
     * @return - список рекомендаций
     */
    public List<GdiRecommendation> getRecommendationsForAdGroups(int shard,
                                                                 ClientId clientId,
                                                                 Collection<Long> campaignIds)
    {
        Map<Long, CpmPriceCampaign> campaigns = getCpmPriceCampaigns(shard, campaignIds);
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(
                mapList(campaigns.values(), CpmPriceCampaign::getPricePackageId));

        // Проверка и рекомендация добавить дефолтные группы в кампании
        List<GdiRecommendation> recommendations = new ArrayList<>(
                getRecommendationForDefaultAdGroupForAdGroups(shard, clientId, campaigns, pricePackages)
        );

        return recommendations;
    }

    private List<GdiRecommendation> getRecommendationForDefaultAdGroupForAdGroups(int shard,
            ClientId clientId,
            Map<Long, CpmPriceCampaign> campaigns,
            Map<Long, PricePackage> pricePackages) {
        var cpmYndxFrontpageCampaigns = filterList(campaigns.values(),
                campaign -> pricePackages.get(campaign.getPricePackageId()).isFrontpagePackage());

        Map<Long, List<Long>> campaignIdToAdGroupIds =
                adGroupRepository.getAdGroupIdsByCampaignIds(shard, mapList(cpmYndxFrontpageCampaigns, CpmPriceCampaign::getId));
        Map<Long, Long> campaignIdToDefaultAdGroupId = getCampaignDefaultAdGroupIds(shard, campaignIdToAdGroupIds);

        Map<Long, Set<CreativeSizeWithExpand>> campaignIdToMissingFormats =
                getCampaignMissingFormats(shard,
                        listToMap(cpmYndxFrontpageCampaigns, CpmPriceCampaign::getId), campaignIdToDefaultAdGroupId);

        return EntryStream.of(campaignIdToMissingFormats)
                .filterValues(not(Set::isEmpty))
                .flatMapKeyValue((campaignId, missingFormats) -> {
                    List<Long> campaignAdGroupIds = campaignIdToAdGroupIds.getOrDefault(campaignId, emptyList());
                    return mapList(campaignAdGroupIds,
                            adGroupId -> toRecommendation(clientId.asLong(), campaignId, adGroupId, missingFormats))
                            .stream();

                })
                .toList();
    }

    private Map<Long, Long> getCampaignDefaultAdGroupIds(int shard, Map<Long, List<Long>> campaignIdToAdGroupIds) {
        List<Long> adGroupIds = flatMap(campaignIdToAdGroupIds.values(), identity());
        Map<Long, Long> adGroupPriorities = adGroupRepository.getAdGroupsPriority(shard, adGroupIds);
        return EntryStream.of(campaignIdToAdGroupIds)
                .mapValues(campaignAdGroupIds -> filterList(campaignAdGroupIds, isDefaultAdGroup(adGroupPriorities)))
                .mapToValue((campaignId, campaignAdGroupIds) -> {
                    if (campaignAdGroupIds.size() > 1) {
                        logger.warn("Campaign {} has more than one default group: {}", campaignId, campaignAdGroupIds);
                    }
                    return campaignAdGroupIds.isEmpty() ? null : campaignAdGroupIds.get(0);
                })
                .filterValues(Objects::nonNull)
                .toMap();
    }

    private Map<Long, Set<CreativeSizeWithExpand>> getCampaignMissingFormats(
            int shard,
            Map<Long, CpmPriceCampaign> campaigns,
            Map<Long, Long> campaignIdToDefaultAdGroupId) {
        Map<Long, Set<CreativeSizeWithExpand>> adGroupIdToCreativeSizes =
                getAdGroupIdToCurrentSizes(shard, campaignIdToDefaultAdGroupId.values());

        return EntryStream.of(campaigns)
                .mapValues(campaign -> {
                    Long defaultAdGroupId = campaignIdToDefaultAdGroupId.get(campaign.getId());
                    Set<CreativeSizeWithExpand> currentSizes = adGroupIdToCreativeSizes.getOrDefault(defaultAdGroupId, emptySet());
                    Set<CreativeSizeWithExpand> missedSizes = new HashSet<>();

                    for (var equivalentFormatsSet : collectCampaignCreativeSizesWithExpand(campaign)) {
                        if (Sets.intersection(equivalentFormatsSet, currentSizes).isEmpty()) {
                            missedSizes.addAll(equivalentFormatsSet);
                        }
                    }

                    Set<CreativeSize> hiddenFormats = new HashSet<>(4);//Форматы по фиче, скрываем из рекомендаций
                    if (!featureService.isEnabledForClientId(ClientId.fromLong(campaign.getClientId()), FeatureName.CPM_PRICE_PREMIUM_FORMAT)) {
                        hiddenFormats.add(CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE);
                    }
                    if (!featureService.isEnabledForClientId(ClientId.fromLong(campaign.getClientId()), FeatureName.CPM_PRICE_PREMIUM_MOBILE)) {
                        hiddenFormats.addAll(CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE);
                    }
                    return filterToSet(missedSizes,
                            size -> !hiddenFormats.contains(new CreativeSize(size.getWidth(), size.getHeight()))
                    );
                })
                .toMap();
    }


    /**
     * Получить мапу adGroupId -> Set<CreativeSizeWithExpand>
     */
    private Map<Long, Set<CreativeSizeWithExpand>> getAdGroupIdToCurrentSizes(int shard, Collection<Long> adGroupIds) {

        List<BannerWithPricePackage> bannersWithSystemFields =
                bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds, BannerWithPricePackage.class);

        Map<Long, Set<Long>> adGroupIdToBannerIds = EntryStream.of(bannersWithSystemFields)
                .mapToKey((index, banner) -> banner.getAdGroupId())
                .filterValues(CampaignWithPricePackageUtils::isBannerNonRejectedOrArchived)
                .mapValues(BannerWithPricePackage::getId)
                .groupingTo(HashSet::new);

        List<Long> allBannerIds = flatMap(adGroupIdToBannerIds.values(), identity());
        Map<Long, Creative> creativesByBannerIds = creativeRepository.getCreativesByBannerIds(shard, allBannerIds);

        return EntryStream.of(adGroupIdToBannerIds)
                .mapValues(bannerIds -> bannerIds.stream()
                        .map(creativesByBannerIds::get)
                        .filter(Objects::nonNull)
                        .map(CreativeSizeWithExpand::new)
                        .collect(toSet()))
                .toMap();
    }

    /**
     * Сформировать объект GdiRecommendation для рекомендации
     *
     * @param clientId   - id клиента
     * @param campaignId - id кампании
     * @param adGroupId  - id дефолтной группы
     * @param difference - список разрешений, которых не хватает в кампании
     * @return - рекомендация
     */
    private GdiRecommendation toRecommendation(Long clientId, Long campaignId, Long adGroupId,
                                               Set<CreativeSizeWithExpand> difference) {
        List<GdPricePackageRecommendationFormat> formats = difference.stream()
                .map(x -> new GdPricePackageRecommendationFormat()
                        .withWidth(x.getWidth())
                        .withHeight(x.getHeight())
                        .withHasExpand(x.getHasExpand()))
                .collect(toList());

        GdRecommendationKpiAddBannerFormatsForPriceSalesCorrectness gdKpi =
                new GdRecommendationKpiAddBannerFormatsForPriceSalesCorrectness()
                        .withPricePackageFormats(formats);
        addUnnecessaryFields(gdKpi);
        String kpi = Checked.get(() -> OBJECT_MAPPER.writeValueAsString(gdKpi));

        return new GdiRecommendation()
                .withClientId(clientId)
                .withType(recommendationType)
                .withCid(campaignId)
                .withPid(adGroupId)
                .withBid(0L)
                .withUserKey1("")
                .withUserKey2("")
                .withUserKey3("")
                .withKpi(kpi)
                .withTimestamp(now().toEpochSecond(UTC));
    }

    /**
     * Получить только CPM_PRICE кампании из списка campaignIds
     */
    private Map<Long, CpmPriceCampaign> getCpmPriceCampaigns(int shard, Collection<Long> campaignIds) {
        Map<Long, CampaignType> campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
        return campaignTypedRepository.getTypedCampaignsMap(shard, campaignTypes, CampaignType.CPM_PRICE);
    }

}
