package ru.yandex.direct.grid.core.entity.recommendation.service.outdoor;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.model.recommendation.GdOutdoorVideoFormat;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpi;
import ru.yandex.direct.utils.Checked;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.time.LocalDateTime.now;
import static java.time.ZoneOffset.UTC;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.addUnnecessaryFields;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Абстрактный класс для outdoor рекомендаций:
 * <p>
 * Рекомендация 1: (uploadAppropriateCreatives)
 * "Загрузите креативы следующих форматов"
 * (Загрузите креативы для того чтобы на щите что-то крутилось)
 * <p>
 * Рекомендация 2: (chooseAppropriatePlacementsForBanner)
 * "Выберите щит для баннера на группе этого баннера"
 * (Выберите щиты любого разрешения из разрешний креатива баннера)
 * <p>
 * Рекомендация 3: (chooseAppropriatePlacementsForAdGroup)
 * "Выберите щиты следующих форматов""
 * (Выберите щиты определенных соотношений)
 */
@ParametersAreNonnullByDefault
abstract class AbstractGridOutdoorVideoRecommendationService {
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final PlacementBlockRepository blockRepository;
    private final CreativeRepository creativeRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final AdGroupRepository adGroupRepository;
    private final GdiRecommendationType recommendationType;

    AbstractGridOutdoorVideoRecommendationService(PlacementBlockRepository blockRepository,
                                                  CreativeRepository creativeRepository,
                                                  BannerRelationsRepository bannerRelationsRepository,
                                                  AdGroupRepository adGroupRepository,
                                                  GdiRecommendationType recommendationType) {
        this.blockRepository = blockRepository;
        this.creativeRepository = creativeRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.adGroupRepository = adGroupRepository;
        this.recommendationType = recommendationType;
    }

    /**
     * Основной метод рекомендаций.
     * <p>
     * 1. Получить разрешения креативов (adGroupId->bannerId->[resolutions])
     * 2. Получить разрешения щитов (adGroupId->[resolutions])
     * 3. Получить сырые рекомендации (реализуется в потомке)
     * 4. Сконвертировать сырые рекомендации в GdiRecommendation и вернуть
     *
     * @param shard                          - шард
     * @param clientId                       - клиент
     * @param campaignIdsByOutdoorAdGroupIds - adGroupId -> campaignId
     * @return рекомендации
     */
    public List<GdiRecommendation> getRecommendations(int shard, Long clientId,
                                                      Map<Long, Long> campaignIdsByOutdoorAdGroupIds) {
        Set<Long> outdoorAdGroupIds = campaignIdsByOutdoorAdGroupIds.keySet();

        Map<Long, Map<Long, Set<OutdoorVideoFormat>>> adGroupCreativesVideoFormats =
                getCreativesVideoFormats(shard, outdoorAdGroupIds);

        Map<Long, Set<OutdoorVideoFormat>> adGroupIdToPlacementsVideoFormats =
                getPlacementsVideoFormats(shard, outdoorAdGroupIds);

        List<RawRecommendation> rawRecommendations =
                getRawRecommendations(adGroupIdToPlacementsVideoFormats, adGroupCreativesVideoFormats);
        return rawsToGdiRecommendations(rawRecommendations, clientId, campaignIdsByOutdoorAdGroupIds);
    }


    /**
     * Получить сырые рекомендации на основе разрешений креативов и разрешений щитов.
     * Метод должен быть реализован в потомке.
     *
     * @param adGroupIdToPlacementsVideoFormats разрешения креативов (adGroupId->bannerId->[resolutions])
     * @param adGroupCreativesVideoFormats      разрешения щитов (adGroupId->[resolutions])
     * @return сырые рекомендации
     */
    abstract List<RawRecommendation> getRawRecommendations(Map<Long, Set<OutdoorVideoFormat>> adGroupIdToPlacementsVideoFormats,
                                                           Map<Long, Map<Long, Set<OutdoorVideoFormat>>> adGroupCreativesVideoFormats);


    /**
     * Преобразовать rawRecommendations к списку GdiRecommendation
     *
     * @param clientId                - клиент
     * @param campaignIdsByAdGroupIds - campaignId->adGroupIds
     * @param rawRecommendations      - рекомендации в сыром виде
     * @return - список преобразованных рекомендаций
     */
    private List<GdiRecommendation> rawsToGdiRecommendations(List<RawRecommendation> rawRecommendations,
                                                             Long clientId,
                                                             Map<Long, Long> campaignIdsByAdGroupIds) {
        return rawRecommendations.stream()
                .map(rawRecommendation -> {

                    var gdKpi = createSpecificKpi(rawRecommendation);
                    addUnnecessaryFields(gdKpi);
                    String kpi = Checked.get(() -> OBJECT_MAPPER.writeValueAsString(gdKpi));

                    Long adGroupId = rawRecommendation.getAdGroupId();

                    return new GdiRecommendation()
                            .withClientId(clientId)
                            .withType(recommendationType)
                            .withCid(campaignIdsByAdGroupIds.get(adGroupId))
                            .withPid(adGroupId)
                            .withBid(rawRecommendation.getBannerId())
                            .withUserKey1("")
                            .withUserKey2("")
                            .withUserKey3("")
                            .withKpi(kpi)
                            .withTimestamp(now().toEpochSecond(UTC));
                })
                .collect(toList());
    }

    /**
     * Создать потомка GdRecommendationKpi с заполненными специфическими полями.
     * Метод должен быть реализован в потомке.
     */
    abstract GdRecommendationKpi createSpecificKpi(RawRecommendation rawRecommendation);

    /**
     * Для каждого adGroupId и bannerId получить допустимые разрешения и длительности креативов
     *
     * @param shard      - шард
     * @param adGroupIds - идентификаторы групп
     * @return adGroupId -> { bannerId -> множество OutdoorVideoFormat }
     */
    private Map<Long, Map<Long, Set<OutdoorVideoFormat>>> getCreativesVideoFormats(int shard,
                                                                                   Collection<Long> adGroupIds) {

        Multimap<Long, Long> adGroupIdToBannerIds =
                bannerRelationsRepository.getAdGroupIdToBannerIds(shard, adGroupIds);

        Set<Long> allBannerIds = new HashSet<>(adGroupIdToBannerIds.values());
        Map<Long, Creative> creativesByBannerIds = creativeRepository.getCreativesByBannerIds(shard, allBannerIds);

        checkState(creativesByBannerIds.keySet().equals(allBannerIds), "banners with id %s must have creatives",
                Sets.difference(allBannerIds, creativesByBannerIds.keySet()));

        Map<Long, Map<Long, Set<OutdoorVideoFormat>>> result = new HashMap<>();

        adGroupIdToBannerIds.asMap().forEach((adGroupId, bannerIds) ->
                bannerIds.forEach(bannerId -> {

                    Creative creative = creativesByBannerIds.get(bannerId);

                    AdditionalData creativeAdditionalData = creative.getAdditionalData();
                    double duration = toRoundedSeconds(creativeAdditionalData.getDuration());

                    Set<OutdoorVideoFormat> outdoorVideoFormats = creativeAdditionalData.getFormats().stream()
                            .filter(x -> x.getHeight() != null && x.getWidth() != null)
                            .map(x -> new OutdoorVideoFormat(x, duration))
                            .collect(toSet());

                    Map<Long, Set<OutdoorVideoFormat>> bannerIdToVideoFormats =
                            result.computeIfAbsent(adGroupId, x -> new HashMap<>());

                    bannerIdToVideoFormats.put(bannerId, outdoorVideoFormats);
                }));

        return result;
    }

    /**
     * Для каждого adGroupId получить допустимые разрешения и длительности щитов
     *
     * @param shard      - шард
     * @param adGroupIds - идентификаторы групп
     * @return мапа adGroupId -> множество OutdoorVideoFormat
     */
    private Map<Long, Set<OutdoorVideoFormat>> getPlacementsVideoFormats(int shard,
                                                                         Collection<Long> adGroupIds) {

        Map<Long, List<PageBlock>> adGroupIdToPageBlocks =
                adGroupRepository.getAdGroupsPageTargetByAdGroupId(shard, adGroupIds);

        List<PlacementBlockKey> allPlacementBlockKeys = adGroupIdToPageBlocks.values().stream()
                .flatMap(Collection::stream)
                .map(x -> PlacementBlockKey.of(x.getPageId(), x.getImpId()))
                .collect(toList());

        List<PlacementBlock> placementBlocks = blockRepository.getPlacementBlocks(allPlacementBlockKeys);

        Map<Long, Map<Long, OutdoorBlock>> pageIdToBlockIdToOutdoorVideoFormat =
                placementBlocks.stream()
                        .map(OutdoorBlock.class::cast)
                        .collect(Collectors.groupingBy(OutdoorBlock::getPageId,
                                Collectors.toMap(OutdoorBlock::getBlockId, identity())));

        return EntryStream.of(adGroupIdToPageBlocks)
                .mapValues(pageBlocks ->
                        listToSet(pageBlocks, pageBlock -> {

                            OutdoorBlock outdoorBlock = pageIdToBlockIdToOutdoorVideoFormat
                                    .get(pageBlock.getPageId())
                                    .get(pageBlock.getImpId());
                            double duration = toRoundedSeconds(outdoorBlock.getDuration());
                            return new OutdoorVideoFormat(outdoorBlock.getResolution(), duration);
                        })
                )
                .toMap();
    }

    GdOutdoorVideoFormat convertToGdVideoFormat(OutdoorVideoFormat outdoorVideoFormat) {
        return new GdOutdoorVideoFormat()
                .withHeight(outdoorVideoFormat.getHeight())
                .withWidth(outdoorVideoFormat.getWidth())
                .withDuration(outdoorVideoFormat.getDuration());
    }

    /**
     * Округлить до одного знака после запятой
     *
     * @param decimalSeconds - время в секундах
     * @return - секунды округленные до одного знака после запятой
     */
    private static double toRoundedSeconds(Number decimalSeconds) {
        checkNotNull(decimalSeconds);
        return Math.round(decimalSeconds.doubleValue() * 10) / 10.0;
    }

    abstract static class RawRecommendation {
        private final Long adGroupId;
        private final Long bannerId;

        RawRecommendation(Long adGroupId, Long bannerId) {
            this.adGroupId = adGroupId;
            this.bannerId = bannerId;
        }

        RawRecommendation(Long adGroupId) {
            this.adGroupId = adGroupId;
            this.bannerId = 0L;
        }

        Long getBannerId() {
            return bannerId;
        }

        Long getAdGroupId() {
            return adGroupId;
        }
    }

}
