package ru.yandex.direct.grid.core.entity.recommendation.service.outdoor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.grid.processing.model.recommendation.GdOutdoorVideoFormat;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiChooseAppropriatePlacementsForAdGroup;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationPlacementRatioInfo;

import static com.google.common.math.IntMath.gcd;
import static java.util.Collections.disjoint;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.chooseAppropriatePlacementsForAdGroup;

/**
 * Рекомендация: "Выберите щиты следующих форматов" (chooseAppropriatePlacementsForAdGroup)
 */
@Service
@ParametersAreNonnullByDefault
public class GridOutdoorVideoRecommendationForAdGroupService extends AbstractGridOutdoorVideoRecommendationService {

    public GridOutdoorVideoRecommendationForAdGroupService(PlacementBlockRepository blockRepository,
                                                           CreativeRepository creativeRepository,
                                                           BannerRelationsRepository bannerRelationsRepository,
                                                           AdGroupRepository adGroupRepository) {
        super(blockRepository, creativeRepository, bannerRelationsRepository, adGroupRepository,
                chooseAppropriatePlacementsForAdGroup);
    }

    /**
     * Рекомендация: "Выберите щиты следующих форматов"
     * <p>
     * Если хотя бы одно разрешение креатива подходит хотя бы под один щит, то для такого баннера не нужно ничего
     * рекомендовать.
     * Если подходящего щита нет, то определяем соотношение креатива.
     * В рекомендацию попадают соотношения креативов для которых нет подходящих щитов + их количество (группировка по
     * соотношению)
     *
     * @param adGroupIdToPlacementsVideoFormats разрешения креативов (adGroupId->bannerId->[resolutions])
     * @param adGroupCreativesVideoFormats      разрешения щитов (adGroupId->[resolutions])
     * @return недостающие соотношения щитов для групп
     */
    @Override
    List<RawRecommendation> getRawRecommendations(Map<Long, Set<OutdoorVideoFormat>> adGroupIdToPlacementsVideoFormats,
                                                  Map<Long, Map<Long, Set<OutdoorVideoFormat>>> adGroupCreativesVideoFormats) {
        List<RawRecommendation> result = new ArrayList<>();

        adGroupCreativesVideoFormats.forEach((adGroupId, bannerIdToCreativesVideoFormats) -> {
                    Map<OutdoorVideoFormat, Integer> ratioToCount = new HashMap<>();

                    bannerIdToCreativesVideoFormats.forEach((bannerId, creativeVideoFormats) -> {

                        Set<OutdoorVideoFormat> placementsVideoFormats =
                                adGroupIdToPlacementsVideoFormats.getOrDefault(adGroupId, emptySet());

                        boolean noIntersection = disjoint(creativeVideoFormats, placementsVideoFormats);
                        if (noIntersection) {
                            Optional<OutdoorVideoFormat> creativeRatio = getCreativeRatio(creativeVideoFormats);
                            creativeRatio.ifPresent(format -> ratioToCount.merge(format, 1, Integer::sum));
                        }
                    });

                    if (!ratioToCount.isEmpty()) {
                        result.add(new RawRecommendationForAdGroup(adGroupId, ratioToCount));
                    }
                }
        );

        return result;
    }

    /**
     * Получить соотношение креатива (с длительностью).
     * У разрешений одного креатива могут быть разные соотношения, потому что креатив может нарезаться на
     * нестандартный щит. Поэтому берем минимальное соотношение.
     * <p>
     * Используем OutdoorVideoFormat для хранения соотношений сторон вместе с длительностью.
     *
     * @param creativeVideoFormats - список форматов креатива
     * @return соотношение сторон и длительность креатива
     */
    private static Optional<OutdoorVideoFormat> getCreativeRatio(Collection<OutdoorVideoFormat> creativeVideoFormats) {
        return creativeVideoFormats.stream()
                .map(format -> {
                    int g = gcd(format.getWidth(), format.getHeight());
                    if (g == 0) {
                        g = 1;
                    }

                    int width = format.getWidth() / g;
                    int height = format.getHeight() / g;
                    return new OutdoorVideoFormat(width, height, format.getDuration());
                })
                .min(Comparator.comparingInt(x -> x.getWidth() * x.getHeight()));
    }

    @Override
    GdRecommendationKpi createSpecificKpi(RawRecommendation rawRecommendation) {
        RawRecommendationForAdGroup rawRecommendationForAdGroup = (RawRecommendationForAdGroup) rawRecommendation;

        List<GdRecommendationPlacementRatioInfo> ratioInfoList =
                EntryStream.of(rawRecommendationForAdGroup.getRatioToCount())
                        .mapKeyValue((format, count) ->
                                new GdRecommendationPlacementRatioInfo()
                                        .withVideoFormat(convertToGdVideoFormat(format))
                                        .withCreativesCount(count))
                        .toList();


        //todo: для того чтобы не падал фронт (DIRECT-104138). После релиза фронта можно будет удалить
        List<GdOutdoorVideoFormat> gdVideoFormats = rawRecommendationForAdGroup.getRatioToCount()
                .keySet()
                .stream()
                .map(this::convertToGdVideoFormat)
                .collect(toList());


        return new GdRecommendationKpiChooseAppropriatePlacementsForAdGroup()
                .withRatioInfoList(ratioInfoList)
                .withVideoFormats(gdVideoFormats);
    }

    private static class RawRecommendationForAdGroup extends RawRecommendation {
        private final Map<OutdoorVideoFormat, Integer> ratioToCount;

        RawRecommendationForAdGroup(Long adGroupId, Map<OutdoorVideoFormat, Integer> ratioToCount) {
            super(adGroupId);
            this.ratioToCount = ratioToCount;
        }

        Map<OutdoorVideoFormat, Integer> getRatioToCount() {
            return ratioToCount;
        }
    }

}
