package ru.yandex.direct.grid.core.entity.recommendation.service.outdoor;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiChooseAppropriatePlacementsForBanner;

import static java.util.Collections.disjoint;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.chooseAppropriatePlacementsForBanner;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Рекомендация: "Выберите щит для баннера на группе этого баннера" (chooseAppropriatePlacementsForBanner)
 */
@Service
@ParametersAreNonnullByDefault
public class GridOutdoorVideoRecommendationForBannersService extends AbstractGridOutdoorVideoRecommendationService {

    public GridOutdoorVideoRecommendationForBannersService(PlacementBlockRepository blockRepository,
                                                           CreativeRepository creativeRepository,
                                                           BannerRelationsRepository bannerRelationsRepository,
                                                           AdGroupRepository adGroupRepository) {
        super(blockRepository, creativeRepository, bannerRelationsRepository, adGroupRepository,
                chooseAppropriatePlacementsForBanner);
    }

    /**
     * Рекомендация: "Выберите щит для баннера на группе этого баннера"
     * <p>
     * Если хотя бы одно разрешение креатива подходит хотя бы под один щит, то для такого баннера не нужно ничего
     * рекомендовать.
     * Если подходящего щита нет, то в рекомендацию попадают все разрешения креатива.
     *
     * @param adGroupIdToPlacementsVideoFormats разрешения креативов (adGroupId->bannerId->[resolutions])
     * @param adGroupCreativesVideoFormats      разрешения щитов (adGroupId->[resolutions])
     * @return недостающие форматы щитов для креативов
     */
    @Override
    List<RawRecommendation> getRawRecommendations(Map<Long, Set<OutdoorVideoFormat>> adGroupIdToPlacementsVideoFormats,
                                                  Map<Long, Map<Long, Set<OutdoorVideoFormat>>> adGroupCreativesVideoFormats) {
        List<RawRecommendation> result = new ArrayList<>();

        adGroupCreativesVideoFormats.forEach((adGroupId, bannerIdToCreativesVideoFormats) ->
                bannerIdToCreativesVideoFormats.forEach((bannerId, creativeVideoFormats) -> {

                            Set<OutdoorVideoFormat> placementsVideoFormats =
                                    adGroupIdToPlacementsVideoFormats.getOrDefault(adGroupId, emptySet());

                            boolean noIntersection = disjoint(creativeVideoFormats, placementsVideoFormats);

                            if (noIntersection) {
                                result.add(new RawRecommendationForBanners(adGroupId, bannerId, creativeVideoFormats));
                            }
                        }
                )
        );

        return result;
    }

    @Override
    GdRecommendationKpi createSpecificKpi(RawRecommendation rawRecommendation) {
        RawRecommendationForBanners rawRecommendationForBanners = (RawRecommendationForBanners) rawRecommendation;

        return new GdRecommendationKpiChooseAppropriatePlacementsForBanner()
                .withVideoFormats(mapList(rawRecommendationForBanners.getVideoFormats(), this::convertToGdVideoFormat));
    }

    private static class RawRecommendationForBanners extends RawRecommendation {
        private final Set<OutdoorVideoFormat> videoFormats;

        RawRecommendationForBanners(Long adGroupId, Long bannerId, Set<OutdoorVideoFormat> videoFormats) {
            super(adGroupId, bannerId);
            this.videoFormats = videoFormats;
        }

        Set<OutdoorVideoFormat> getVideoFormats() {
            return videoFormats;
        }
    }

}
