package ru.yandex.direct.grid.core.entity.recommendation.service.outdoor;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.stereotype.Service;

import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.canvas.client.model.ffmpegresolutions.FfmpegResolutionsResponse;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpi;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiUploadAppropriateCreatives;

import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType.uploadAppropriateCreatives;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Рекомендация: "Загрузите креативы следующих форматов" (uploadAppropriateCreatives)
 */
@Service
@ParametersAreNonnullByDefault
public class GridOutdoorVideoRecommendationForPlacementsService extends AbstractGridOutdoorVideoRecommendationService {
    private final CanvasClient canvasClient;

    public GridOutdoorVideoRecommendationForPlacementsService(PlacementBlockRepository blockRepository,
                                                              CreativeRepository creativeRepository,
                                                              BannerRelationsRepository bannerRelationsRepository,
                                                              AdGroupRepository adGroupRepository,
                                                              CanvasClient canvasClient) {
        super(blockRepository, creativeRepository, bannerRelationsRepository, adGroupRepository, uploadAppropriateCreatives);
        this.canvasClient = canvasClient;
    }

    /**
     * Рекомендация: "Загрузите креативы следующих форматов"
     * <p>
     * Определить щиты для которых нет подходящих креативов.
     * Определить соотношение сторон этих щитов.
     * Определить по списку нарезаемых разрешений максимальное разрешение для каждого соотношения.
     * В рекомендацию попадают максимальные разрешения для соотношений щитов.
     *
     * @param adGroupIdToPlacementsVideoFormats разрешения креативов (adGroupId->bannerId->[resolutions])
     * @param adGroupCreativesVideoFormats      разрешения щитов (adGroupId->[resolutions])
     * @return максимальные разрешения для соотношений щитов у которых нет подходящих креативов.
     */
    @Override
    List<RawRecommendation> getRawRecommendations(Map<Long, Set<OutdoorVideoFormat>> adGroupIdToPlacementsVideoFormats,
                                                  Map<Long, Map<Long, Set<OutdoorVideoFormat>>> adGroupCreativesVideoFormats) {
        List<RawRecommendation> result = new ArrayList<>();

        List<FfmpegResolutionsResponse> ffmpegResolutions = canvasClient.getCachedOutdoorFfmpegResolutions();

        adGroupIdToPlacementsVideoFormats.forEach((adGroupId, placementsVideoFormats) -> {

            var bannerToCreatives = adGroupCreativesVideoFormats.getOrDefault(adGroupId, emptyMap()).values();
            Set<OutdoorVideoFormat> creativeVideoFormats = flatMapToSet(bannerToCreatives, identity());

            Set<OutdoorVideoFormat> difference = Sets.difference(placementsVideoFormats, creativeVideoFormats);
            boolean adGroupHahShows = placementsVideoFormats.size() > difference.size();

            Set<OutdoorVideoFormat> maxFormats = difference.stream()
                    .map(x -> getMaxFormat(x, ffmpegResolutions))
                    .flatMap(Optional::stream)
                    .collect(toSet());

            if (!maxFormats.isEmpty()) {
                result.add(new RawRecommendationForPlacements(adGroupId, maxFormats, adGroupHahShows));
            }
        });

        return result;
    }

    /**
     * Получить разрешение которое нарежется на максимальное количество щитов, включая этот щит.
     * Определяем соотношение сторон placementVideoFormat. Затем определяем максимальное разрешение для этого
     * соотношения.
     * Если соотношение сторон максимального разрешения не соответствует ratio, то увеличиваем стороны.
     * Считаем максимальным то разрешение у которого площадь больше.
     */
    private Optional<OutdoorVideoFormat> getMaxFormat(OutdoorVideoFormat placementVideoFormat,
                                                      List<FfmpegResolutionsResponse> ffmpegResolutions) {
        int width = placementVideoFormat.getWidth();
        int height = placementVideoFormat.getHeight();

        Optional<FfmpegResolutionsResponse> placementFfmpegResolution = ffmpegResolutions
                .stream()
                .filter(v -> v.getResolutionWidth() == width && v.getResolutionHeight() == height)
                .findAny();

        if (placementFfmpegResolution.isEmpty()) {
            return Optional.empty();
        }

        int ratioWidth = placementFfmpegResolution.get().getRatioWidth();
        int ratioHeight = placementFfmpegResolution.get().getRatioHeight();

        // Находим разрешения с таким же ratio, определяем максимальное, увеличиваем стороны, если оно нестандартное.
        return ffmpegResolutions.stream()
                .filter(x -> x.getRatioWidth() == ratioWidth && x.getRatioHeight() == ratioHeight)
                .max(Comparator.comparingInt(x -> x.getResolutionWidth() * x.getResolutionHeight()))
                .map(max -> increaseSidesToExactRatio(max.getResolutionWidth(), max.getResolutionHeight(),
                        max.getRatioWidth(), max.getRatioHeight(), placementVideoFormat.getDuration()));
    }

    /**
     * Увеличиваем стороны, чтобы соотношение сторон было равно ratio.
     * Если соотношение уже равно ratio, то ничего не изменится.
     * <p>
     * 1) Увеличиваем стороны так, чтобы они стали кратными соответствующей стороне ratio.
     * 2) Увеличиваем одну сторону, чтобы соотношение сторон равнялось ratio.
     * (https://st.yandex-team.ru/DIRECT-103416#5d7f6481122154001c383d1b)
     */
    static OutdoorVideoFormat increaseSidesToExactRatio(int currentWidth, int currentHeight, int ratioWidth,
                                                        int ratioHeight, double duration) {

        // обе стороны в любом случае должны быть кратны соотвуствующей стороне ratio
        int remainderWidth = currentWidth % ratioWidth;
        int remainderHeight = currentHeight % ratioHeight;
        if (remainderWidth != 0) {
            currentWidth += ratioWidth - remainderWidth;
        }
        if (remainderHeight != 0) {
            currentHeight += ratioHeight - remainderHeight;
        }

        // определяем множители для сторон (если соотношение сторон точно равно ratio, то множители не будут отличаться)
        int widthMultiplier = currentWidth / ratioWidth;
        int heightMultiplier = currentHeight / ratioHeight;
        // Увеличиваем сторону, множитель которой оказался меньше (в обратном случае одна из сторон уменьшится).
        if (widthMultiplier > heightMultiplier) {
            currentHeight = widthMultiplier * ratioHeight;
        } else {
            currentWidth = heightMultiplier * ratioWidth;
        }

        return new OutdoorVideoFormat(currentWidth, currentHeight, duration);
    }

    @Override
    GdRecommendationKpi createSpecificKpi(RawRecommendation rawRecommendation) {
        RawRecommendationForPlacements rawRecommendationForPlacements =
                (RawRecommendationForPlacements) rawRecommendation;

        return new GdRecommendationKpiUploadAppropriateCreatives()
                .withIsAdGroupHasShows(rawRecommendationForPlacements.getAdGroupHasShows())
                .withVideoFormats(
                        mapList(rawRecommendationForPlacements.getVideoFormats(), this::convertToGdVideoFormat));
    }

    private static class RawRecommendationForPlacements extends RawRecommendation {
        private final Set<OutdoorVideoFormat> videoFormats;
        private final Boolean adGroupHasShows;

        RawRecommendationForPlacements(Long adGroupId, Set<OutdoorVideoFormat> videoFormats, Boolean adGroupHasShows) {
            super(adGroupId);
            this.videoFormats = videoFormats;
            this.adGroupHasShows = adGroupHasShows;
        }

        Set<OutdoorVideoFormat> getVideoFormats() {
            return videoFormats;
        }

        Boolean getAdGroupHasShows() {
            return adGroupHasShows;
        }
    }

}
