package ru.yandex.direct.grid.core.entity.showcondition.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectSelectStep;
import org.jooq.TableField;
import org.jooq.TableOnConditionStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.keyword.repository.KeywordMapping;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionOrderByField;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.schema.yt.tables.BidstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.grid.schema.yt.tables.records.BidstableDirectRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.grid.schema.yt.Tables.BIDSTABLE_DIRECT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для получения аггрегированной по фразам информации об условиях показа клиентов
 * Нужно для UAC, потому что в разных группах используются одинаковые фразы
 */
@Repository
@ParametersAreNonnullByDefault
public class GridAggregatedShowConditionYtRepository {
    private static final BidstableDirect SHOW_CONDITIONS = BIDSTABLE_DIRECT.as("K");
    private static final List<TableField<?, ?>> DEFAULT_ORDER =
            ImmutableList.of(SHOW_CONDITIONS.CID, SHOW_CONDITIONS.PHRASE);
    private static final Condition NON_DELETED_SHOW_CONDITIONS =
            SHOW_CONDITIONS.IS_DELETED.eq(RepositoryUtils.booleanToLong(false)).or(YtDSL.isNull(SHOW_CONDITIONS.IS_DELETED));

    private static final JooqFilterProcessor<GdiAggregatedShowConditionFilter> FILTER_PROCESSOR =
            JooqFilterProcessor.<GdiAggregatedShowConditionFilter>builder()
                .withFilter(GdiAggregatedShowConditionFilter::getCampaignIdIn, SHOW_CONDITIONS.CID::in)
                .build();


    private final YtDynamicSupport ytSupport;
    private final YtFieldMapper<GdiAggregatedShowCondition, BidstableDirectRecord> showConditionsMapper;
    private final List<OrderField<?>> defaultOrderBy;
    private final Map<GdiAggregatedShowConditionOrderByField, Field<?>> sortingMap;
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);

    @Autowired
    public GridAggregatedShowConditionYtRepository(YtDynamicSupport ytSupport, GridKeywordsParser keywordsParser) {
        this.ytSupport = ytSupport;

        JooqReaderWithSupplier<GdiAggregatedShowCondition> internalReader =
                JooqReaderWithSupplierBuilder.builder(GdiAggregatedShowCondition::new)
                        .readProperty(GdiAggregatedShowCondition.CAMPAIGN_ID, fromField(SHOW_CONDITIONS.CID))
                        .readProperty(GdiAggregatedShowCondition.KEYWORD, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(keywordsParser::getPlainKeyword))
                        .readProperty(GdiAggregatedShowCondition.IS_AUTOTARGETING, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(KeywordMapping::isAutotargetingFromDb))
                        .readProperty(GdiAggregatedShowCondition.MINUS_KEYWORDS, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(keywordsParser::getMinusKeywords))
                        .readProperty(GdiAggregatedShowCondition.TYPE, fromField(SHOW_CONDITIONS.BID_TYPE)
                                .by(GridShowConditionMapping::typeFromString))
                        .build();

        showConditionsMapper = new YtFieldMapper<>(internalReader, SHOW_CONDITIONS);

        defaultOrderBy = mapList(DEFAULT_ORDER, showConditionsMapper::fieldAlias);

        sortingMap = new HashMap<>();

        gridStat.addStatOrderingToMap(sortingMap, GdiAggregatedShowConditionOrderByField.class);
    }

    /**
     * Получить данные об условиях показа
     *
     * @param shard                              шард, в котором хранятся условия показа (исключительно для улучшения
     *                                           результатов запроса)
     * @param filter                             фильтр для выборки данных
     * @param showConditionOrderByList           список полей с порядком сортировки данных
     * @param statFrom                           начала периода, за который нужно получать статистику
     * @param statTo                             конец периода, за который нужно получать статистику (включительно)
     * @param limitOffset                        верхний предел количества полученных условий показа и смещение
     *                                           относительно начала выборки
     * @param goalIds                            идентификаторы целей
     * @param goalIdsForRevenue                  идентификаторы целей, по которым считается доход
     */
    public List<GdiAggregatedShowCondition> getAggregatedShowConditions(int shard, GdiAggregatedShowConditionFilter filter,
                                                                        List<GdiAggregatedShowConditionOrderBy> showConditionOrderByList,
                                                                        LocalDate statFrom, LocalDate statTo, LimitOffset limitOffset,
                                                                        Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        if (CollectionUtils.isAllEmpty(filter.getCampaignIdIn())) {
            return Collections.emptyList();
        }

        Directphrasestatv2Bs statTable = gridStat.getTableData().table();
        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(showConditionsMapper.getFieldsToRead());

        TableOnConditionStep<Record> joinStep = gridStat.joinStatColumns(SHOW_CONDITIONS,
                row(SHOW_CONDITIONS.CID, SHOW_CONDITIONS.PID, YtDSL.toULong(SHOW_CONDITIONS.ID))
                        .eq(statTable.EXPORT_ID, statTable.GROUP_EXPORT_ID, statTable.PHRASE_EXPORT_ID),
                statFrom, statTo, null);
        joinStep = gridStat.joinGoalStatColumns(joinStep, goalIds);

        selectStep.select(gridStat.getStatSelectFields());
        selectStep.select(gridStat.getGoalStatFields(goalIds, goalIdsForRevenue));
        selectStep.from(joinStep);

        Condition baseCondition = FILTER_PROCESSOR.apply(filter).and(NON_DELETED_SHOW_CONDITIONS);

        List<Field<?>> groupBy = new ArrayList<>(showConditionsMapper.getFieldsToRead());
        // И добавляем ключи для ускорения
        groupBy.add(SHOW_CONDITIONS.__HASH__);
        groupBy.add(SHOW_CONDITIONS.__SHARD__);

        selectStep.where(baseCondition)
                .groupBy(groupBy);

        List<OrderField<?>> orderFields = gridStat.getOrderFields(sortingMap, showConditionOrderByList, defaultOrderBy);

        Select query = selectStep
                .orderBy(orderFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        return ytSupport
                .selectRows(shard, query, false).getYTreeRows().stream()
                .map(n -> showConditionsMapper.fromNode(n)
                        .withStat(GridStatNew.addZeros(goalIdsForRevenue != null ?
                                gridStat.extractStatsEntryWithGoalsRevenue(n, goalIds, goalIdsForRevenue) :
                                gridStat.extractStatsEntry(n)))
                        .withGoalStats(gridStat.extractGoalStatEntries(n, goalIds)))
                .collect(Collectors.toList());
    }
}
