package ru.yandex.direct.grid.core.entity.showcondition.repository;

import java.util.List;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.springframework.stereotype.Component;

import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.parser.KeywordParser;

import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.phraseWithoutAutotargetingPrefix;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Компонент, производящий парсинг ключевых фраз для разделения их на собственно фразы и минус-слова
 */
// TODO(bzzzz): кажется, что использование здесь KeywordParser - это overkill, но пока что это самое быстрое решение
@Component
@ParametersAreNonnullByDefault
public class GridKeywordsParser {
    private static final int CACHE_SIZE = 1_000;

    private final LoadingCache<String, KeywordWithMinuses> keywordCache = CacheBuilder.newBuilder()
            .recordStats()
            .maximumSize(CACHE_SIZE)
            .build(
                    new CacheLoader<String, KeywordWithMinuses>() {
                        public KeywordWithMinuses load(String key) {
                            return KeywordParser.parseWithMinuses(key);
                        }
                    });

    /**
     * Получить плюс-часть ключевой фразы
     *
     * @param keywordString строка с ключевой фразой
     */
    public String getPlainKeyword(@Nullable String keywordString) {
        KeywordWithMinuses keyword = parseKeyword(keywordString);

        return keyword != null ? keyword.getKeyword().toString() : null;
    }

    /**
     * Получить список минус-фраз ключевой фразы
     *
     * @param keywordString строка с ключевой фразой
     */
    public List<String> getMinusKeywords(@Nullable String keywordString) {
        KeywordWithMinuses keyword = parseKeyword(keywordString);

        return keyword != null ? mapList(keyword.getMinusKeywords(), Keyword::toString) : null;
    }

    public KeywordWithMinuses parseKeyword(@Nullable String keywordString) {
        if (keywordString == null) {
            return null;
        }

        keywordString = phraseWithoutAutotargetingPrefix(keywordString);
        try {
            return keywordCache.get(keywordString);
        } catch (ExecutionException e) {
            return null;
        }
    }
}
