package ru.yandex.direct.grid.core.entity.showcondition.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.SelectSelectStep;
import org.jooq.SelectWindowStep;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.RetargetingFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingCondition;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionType;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingOrderBy;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingOrderByField;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargeting;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargetingStatusBsSynced;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargetingWithTotals;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.ordering.JooqOrderingProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.stats_from_query.WithTotalsUtils;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.schema.yt.tables.BidsRetargetingtableDirect;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.grid.schema.yt.tables.RetargetingConditionstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.records.BidsRetargetingtableDirectRecord;
import ru.yandex.direct.grid.schema.yt.tables.records.RetargetingConditionstableDirectRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromStringFieldToEnum;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.not;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCase;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;

@Repository
@ParametersAreNonnullByDefault
public class GridRetargetingYtRepository {

    private static final BidsRetargetingtableDirect BIDS =
            BidsRetargetingtableDirect.BIDS_RETARGETINGTABLE_DIRECT.as("R");
    private static final RetargetingConditionstableDirect CONDITIONS =
            RetargetingConditionstableDirect.RETARGETING_CONDITIONSTABLE_DIRECT.as("C");
    private static final String RET_COND_ID_FIELD_NAME = CONDITIONS.RET_COND_ID.getName();

    private final YtFieldMapper<GdiRetargetingCondition, RetargetingConditionstableDirectRecord>
            retargetingConditionYtMapper;

    /**
     * Все поля кроме "ret_cond_id", чтобы не было дублирования при джойне с bids
     */
    private final List<Field<?>> retargetingConditionFieldsForJoin;

    private final YtFieldMapper<GdiBidsRetargeting, BidsRetargetingtableDirectRecord> bidsRetargetingYtMapper;
    private final JooqOrderingProcessor<GdiRetargetingOrderByField> orderingProcessor;
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);

    private final YtDynamicSupport ytSupport;

    private final Collection<OrderField<?>> defaultOrderByFields;

    private static final LimitOffset DEFAULT_LIMIT_OFFSET = new LimitOffset(10000, 0);

    private static final JooqFilterProcessor<GdiRetargetingFilter> BIDS_FILTER_PROCESSOR =
            getFilterProcessor(true);
    private static final JooqFilterProcessor<GdiRetargetingFilter> BIDS_FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS =
            getFilterProcessor(false);

    private static JooqFilterProcessor<GdiRetargetingFilter> getFilterProcessor(boolean withFilterByStatus) {
        var builder = JooqFilterProcessor.<GdiRetargetingFilter>builder()
                .withFilter(GdiRetargetingFilter::getRetargetingIdIn, BIDS.RET_ID::in)
                .withFilter(GdiRetargetingFilter::getRetargetingIdNotIn, not(BIDS.RET_ID::in))
                .withFilter(GdiRetargetingFilter::getRetargetingConditionIdIn, BIDS.RET_COND_ID::in)
                .withFilter(GdiRetargetingFilter::getRetargetingConditionIdNotIn, not(BIDS.RET_COND_ID::in))
                .withFilter(GdiRetargetingFilter::getAdGroupIdIn, BIDS.PID::in)
                .withFilter(GdiRetargetingFilter::getCampaignIdIn, BIDS.CID::in)
                .withFilter(GdiRetargetingFilter::getMaxPriceContext,
                        mp -> BIDS.PRICE_CONTEXT.le(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiRetargetingFilter::getMinPriceContext,
                        mp -> BIDS.PRICE_CONTEXT.ge(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiRetargetingFilter::getNameContains, substringIgnoreCase(CONDITIONS.CONDITION_NAME))
                .withFilter(filter -> RepositoryUtils.booleanToLong(filter.getInterest()), CONDITIONS.INTEREST::eq)
                .withFilter(GdiRetargetingFilter::getNameNotContains,
                        not(substringIgnoreCase(CONDITIONS.CONDITION_NAME)))
                .withFilter(GdiRetargetingFilter::getTypeIn,
                        typeIn -> CONDITIONS.RETARGETING_CONDITIONS_TYPE.in(mapSet(typeIn,
                                GdiRetargetingConditionType::toSource)))
                .withFilter(GdiRetargetingFilter::getTypeNotIn,
                        typeNotIn -> DSL.not(CONDITIONS.RETARGETING_CONDITIONS_TYPE.in(mapSet(typeNotIn,
                                GdiRetargetingConditionType::toSource))));
        if (withFilterByStatus) {
            builder.withFilter(GdiRetargetingFilter::getStatusIn,
                    statuses -> GridRetargetingMapping.buildFromBaseStatuses(BIDS, statuses));
        }
        return builder.build();
    }

    private static final JooqFilterProcessor<GdiRetargetingConditionFilter> CONDITIONS_FILTER_PROCESSOR =
            JooqFilterProcessor.<GdiRetargetingConditionFilter>builder()
                    .withFilter(GdiRetargetingConditionFilter::getClientId, CONDITIONS.CLIENT_ID::eq)
                    .withFilter(GdiRetargetingConditionFilter::getRetargetingConditionIdIn, CONDITIONS.RET_COND_ID::in)
                    .withFilter(GdiRetargetingConditionFilter::getRetargetingConditionIdNotIn,
                            not(CONDITIONS.RET_COND_ID::in))
                    .withFilter(GdiRetargetingConditionFilter::getRetargetingConditionTypeIn,
                            types -> CONDITIONS.RETARGETING_CONDITIONS_TYPE.in(mapSet(types,
                                    GdiRetargetingConditionType::toSource)))
                    .withFilter(GdiRetargetingConditionFilter::getNameIn, CONDITIONS.CONDITION_NAME::in)
                    .withFilter(GdiRetargetingConditionFilter::getNameNotIn, not(CONDITIONS.CONDITION_NAME::in))
                    .withFilter(filter -> RepositoryUtils.booleanToLong(filter.getIsNegative()),
                            CONDITIONS.NEGATIVE::eq)
                    .withFilter(filter -> RepositoryUtils.booleanToLong(filter.getIsInterest()),
                            CONDITIONS.INTEREST::eq)
                    .withFilter(filter -> RepositoryUtils.booleanToLong(filter.getIsAutoRetargeting()),
                            CONDITIONS.AUTORETARGETING::eq)
                    .build();

    private static final Condition NON_DELETED_RETARGETING_CONDITION = CONDITIONS.IS_DELETED.eq(0L);

    @Autowired
    public GridRetargetingYtRepository(YtDynamicSupport ytSupport) {
        this.ytSupport = ytSupport;

        JooqReaderWithSupplier<GdiRetargetingCondition> retargetingConditionsReader =
                JooqReaderWithSupplierBuilder.builder(GdiRetargetingCondition::new)
                        .readProperty(GdiRetargetingCondition.RETARGETING_CONDITION_ID,
                                fromField(CONDITIONS.RET_COND_ID))
                        .readProperty(GdiRetargetingCondition.CLIENT_ID, fromField(CONDITIONS.CLIENT_ID))
                        .readProperty(GdiRetargetingCondition.NAME, fromField(CONDITIONS.CONDITION_NAME))
                        .readProperty(GdiRetargetingCondition.DESCRIPTION, fromField(CONDITIONS.CONDITION_DESC))
                        .readProperty(GdiRetargetingCondition.TYPE,
                                fromStringFieldToEnum(CONDITIONS.RETARGETING_CONDITIONS_TYPE,
                                        GdiRetargetingConditionType.class))
                        .readProperty(GdiRetargetingCondition.IS_DELETED, fromLongFieldToBoolean(CONDITIONS.IS_DELETED))
                        .readProperty(GdiRetargetingCondition.NEGATIVE, fromLongFieldToBoolean(CONDITIONS.NEGATIVE))
                        .readProperty(GdiRetargetingCondition.INTEREST, fromLongFieldToBoolean(CONDITIONS.INTEREST))
                        .readProperty(GdiRetargetingCondition.CONDITION_RULES,
                                fromField(CONDITIONS.CONDITION_JSON)
                                        .by(GridRetargetingMapping::retargetingConditionJsonReader))
                        .build();

        retargetingConditionYtMapper = new YtFieldMapper<>(retargetingConditionsReader, CONDITIONS);

        retargetingConditionFieldsForJoin = filterList(retargetingConditionYtMapper.getFieldsToRead(),
                s -> !s.getName().equals(RET_COND_ID_FIELD_NAME));

        JooqReaderWithSupplier<GdiBidsRetargeting> bidsRetargetingReader =
                JooqReaderWithSupplierBuilder.builder(GdiBidsRetargeting::new)
                        .readProperty(GdiBidsRetargeting.RETARGETING_ID, fromField(BIDS.RET_ID))
                        .readProperty(GdiBidsRetargeting.RETARGETING_CONDITION_ID, fromField(BIDS.RET_COND_ID))
                        .readProperty(GdiBidsRetargeting.CAMPAIGN_ID, fromField(BIDS.CID))
                        .readProperty(GdiBidsRetargeting.AD_GROUP_ID, fromField(BIDS.PID))
                        .readProperty(GdiBidsRetargeting.AD_ID, fromField(BIDS.BID))
                        .readPropertyForFirst(GdiBidsRetargeting.STATUS_BS_SYNCED,
                                fromStringFieldToEnum(BIDS.STATUS_BS_SYNCED,
                                        GdiBidsRetargetingStatusBsSynced.class))
                        .readPropertyForFirst(GdiBidsRetargeting.AUTO_BUDGET_PRIORITY,
                                fromField(BIDS.AUTOBUDGET_PRIORITY)
                                        .by(GridShowConditionMapping::autobudgetPriorityFromNum))
                        .readPropertyForFirst(GdiBidsRetargeting.IS_SUSPENDED,
                                fromLongFieldToBoolean(BIDS.IS_SUSPENDED))
                        .readPropertyForFirst(GdiBidsRetargeting.PRICE_CONTEXT, fromField(BIDS.PRICE_CONTEXT).by(
                                YtMappingUtils::fromMicros))
                        .readPropertyForFirst(GdiBidsRetargeting.REACH, fromField(BIDS.REACH))
                        .build();

        bidsRetargetingYtMapper = new YtFieldMapper<>(bidsRetargetingReader, BIDS);

        JooqOrderingProcessor.Builder<GdiRetargetingOrderByField> orderingBuilder =
                JooqOrderingProcessor.<GdiRetargetingOrderByField>builder()
                        .withField(GdiRetargetingOrderByField.ID, bidsRetargetingYtMapper.fieldAlias(BIDS.RET_ID))
                        .withField(GdiRetargetingOrderByField.CAMPAIGN_ID, bidsRetargetingYtMapper.fieldAlias(BIDS.CID))
                        .withField(GdiRetargetingOrderByField.GROUP_ID, bidsRetargetingYtMapper.fieldAlias(BIDS.PID))
                        .withField(GdiRetargetingOrderByField.AUTOBUDGET_PRIORITY,
                                bidsRetargetingYtMapper.fieldAlias(BIDS.AUTOBUDGET_PRIORITY))
                        .withField(GdiRetargetingOrderByField.BANNER_ID, bidsRetargetingYtMapper.fieldAlias(BIDS.BID))
                        .withField(GdiRetargetingOrderByField.RETARGETING_CONDITION_ID,
                                bidsRetargetingYtMapper.fieldAlias(BIDS.RET_COND_ID))
                        .withField(GdiRetargetingOrderByField.PRICE_CONTEXT,
                                bidsRetargetingYtMapper.fieldAlias(BIDS.PRICE_CONTEXT))
                        .withField(GdiRetargetingOrderByField.NAME,
                                bidsRetargetingYtMapper.fieldAlias(CONDITIONS.CONDITION_NAME));

        orderingProcessor = gridStat.addStatOrdering(orderingBuilder, GdiRetargetingOrderByField.class)
                .build();

        defaultOrderByFields =
                mapSet(ImmutableSet.of(BIDS.CID, BIDS.PID, BIDS.BID, BIDS.RET_ID), bidsRetargetingYtMapper::fieldAlias);
    }

    public List<GdiRetargetingCondition> getRetargetingConditions(int shard, GdiRetargetingConditionFilter filter) {
        Condition where = NON_DELETED_RETARGETING_CONDITION
                .and(CONDITIONS_FILTER_PROCESSOR.apply(filter));

        SelectConditionStep<Record> baseSelectStep = YtDSL.ytContext()
                .select(retargetingConditionYtMapper.getFieldsToRead())
                .from(CONDITIONS)
                .where(where);

        Collection<OrderField<?>> defaultOrderBy =
                mapSet(ImmutableSet.of(CONDITIONS.CLIENT_ID, CONDITIONS.RET_COND_ID, CONDITIONS.CONDITION_NAME),
                        retargetingConditionYtMapper::fieldAlias);

        Select query = baseSelectStep
                .orderBy(defaultOrderBy)
                .limit(DEFAULT_LIMIT_OFFSET.limit())
                .offset(DEFAULT_LIMIT_OFFSET.offset());

        return ytSupport.selectRows(shard, query).getYTreeRows().stream()
                .map(retargetingConditionYtMapper::fromNode)
                .collect(Collectors.toList());
    }

    public GdiBidsRetargetingWithTotals getRetargetings(int shard, ClientId clientId, GdiRetargetingFilter filter,
                                                        RetargetingFetchedFieldsResolver fetchingFieldsResolver,
                                                        List<GdiRetargetingOrderBy> retargetingOrderByList,
                                                        LocalDate statFrom, LocalDate statTo,
                                                        LimitOffset limitOffset,
                                                        boolean disableStatusFilter, boolean addWithTotalsToQuery) {
        if (CollectionUtils
                .isAllEmpty(filter.getCampaignIdIn(), filter.getAdGroupIdIn(), filter.getRetargetingIdIn())) {
            return new GdiBidsRetargetingWithTotals()
                    .withGdiBidsRetargetings(Collections.emptyList());
        }

        SelectSelectStep<Record> baseSelectStep = YtDSL.ytContext()
                .select(bidsRetargetingYtMapper.getFieldsToRead())
                .select(bidsRetargetingYtMapper.getFieldsToReadForFirst());

        //query для статистики (если она нужна)
        Directphrasestatv2Bs table = gridStat.getTableData().table();

        //фильтр
        Condition where =
                (disableStatusFilter ? BIDS_FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS : BIDS_FILTER_PROCESSOR).apply(filter);

        //group by
        List<Field<?>> groupBy = new ArrayList<>(bidsRetargetingYtMapper.getFieldsToRead());
        groupBy.add(BIDS.__HASH__);
        groupBy.add(BIDS.__SHARD__);

        boolean hasStatFieldsSort = gridStat.hasStatFieldsSort(retargetingOrderByList);
        boolean needStats = fetchingFieldsResolver.getStats() || filter.getStats() != null || hasStatFieldsSort;
        boolean needRetargetingConditions =
                fetchingFieldsResolver.getRetargetingCondition()
                        || filter.getNameContains() != null || filter.getInterest() != null
                        || filter.getTypeIn() != null || filter.getTypeNotIn() != null;

        TableOnConditionStep<Record> joinStep = null;

        if (needRetargetingConditions) {
            joinStep = nvl(joinStep, BIDS)
                    .join(CONDITIONS)
                    .on(row(BIDS.RET_COND_ID, BIDS.__SHARD__, DSL.val(clientId.asLong()))
                            .eq(CONDITIONS.RET_COND_ID, CONDITIONS.__SHARD__, CONDITIONS.CLIENT_ID));

            baseSelectStep.select(retargetingConditionFieldsForJoin);
            where = where.and(CONDITIONS.IS_DELETED.eq(RepositoryUtils.booleanToLong(false)));
            groupBy.addAll(retargetingConditionFieldsForJoin);
        }

        if (needStats) {
            joinStep = gridStat.joinStatColumns(nvl(joinStep, BIDS),
                    row(BIDS.CID, BIDS.PID, YtDSL.toULong(BIDS.RET_COND_ID))
                            .eq(table.EXPORT_ID, table.GROUP_EXPORT_ID, table.PHRASE_ID),
                    statFrom, statTo, null);

            baseSelectStep.select(gridStat.getStatSelectFields());
        }

        baseSelectStep.from(nvl(joinStep, BIDS));
        baseSelectStep.where(where).groupBy(groupBy);

        SelectWindowStep<Record> pageStep = baseSelectStep;

        Condition statHavingCondition = gridStat.getStatHavingCondition(filter.getStats());
        if (statHavingCondition != null) {
            pageStep = baseSelectStep.having(statHavingCondition);
        }

        //order_by, limit, offset
        Collection<OrderField<?>> orderByFields;
        if (!retargetingOrderByList.isEmpty()) {
            orderByFields = orderingProcessor.construct(retargetingOrderByList);
        } else {
            orderByFields = defaultOrderByFields;
        }

        Select query = pageStep
                .orderBy(orderByFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        List<GdiBidsRetargeting> bidsRetargetings = mapList(ytSupport
                        .selectRows(shard, query, addWithTotalsToQuery && needStats).getYTreeRows(),
                (n -> {
                    GdiBidsRetargeting gdiBidsRetargeting = bidsRetargetingYtMapper.fromNode(n);
                    if (needRetargetingConditions && gdiBidsRetargeting.getRetargetingId() != null) {
                        gdiBidsRetargeting.setRetargetingCondition(retargetingConditionYtMapper.fromNode(n));
                    }
                    if (needStats) {
                        gdiBidsRetargeting.setStat(GridStatNew.addZeros(gridStat.extractStatsEntry(n)));
                    }
                    return gdiBidsRetargeting;
                }));

        // Возвращаем totals из запроса если кол. условий показа = лимиту (showConditions.size == limit + total row)
        boolean withTotalStats = WithTotalsUtils
                .withTotalStats(addWithTotalsToQuery, needStats, bidsRetargetings, limitOffset);

        GdiBidsRetargeting bidsRetargetingWithTotals = !withTotalStats ? null : StreamEx.of(bidsRetargetings)
                .findFirst(bidsRetargeting -> Objects.isNull(bidsRetargeting.getRetargetingId()))
                .orElse(null);
        return new GdiBidsRetargetingWithTotals()
                .withGdiBidsRetargetings(StreamEx.of(bidsRetargetings)
                        .filter(bidsRetargeting -> Objects.nonNull(bidsRetargeting.getRetargetingId()))
                        .toList())
                .withTotalStats(bidsRetargetingWithTotals != null ? bidsRetargetingWithTotals.getStat() : null);
    }
}
