package ru.yandex.direct.grid.core.entity.showcondition.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectSelectStep;
import org.jooq.SelectWindowStep;
import org.jooq.TableField;
import org.jooq.TableOnConditionStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.keyword.repository.KeywordMapping;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.ShowConditionFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionOrderByField;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionPrimaryStatus;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionStatusModerate;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionWithTotals;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.stats_from_query.WithTotalsUtils;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.core.util.yt.mapping.YtFieldMapper;
import ru.yandex.direct.grid.schema.yt.tables.BidstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.grid.schema.yt.tables.records.BidstableDirectRecord;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromStringFieldToEnum;
import static ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionStatusModerate.toSource;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inAllSetsOfSetSubstringIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.inSetIgnoreCase;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.not;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.substringIgnoreCase;
import static ru.yandex.direct.grid.schema.yt.Tables.BIDSTABLE_DIRECT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;

/**
 * Репозиторий для получения базовой информации об условиях показа клиентов
 */
@Repository
@ParametersAreNonnullByDefault
public class GridShowConditionYtRepository {
    private static final BidstableDirect SHOW_CONDITIONS = BIDSTABLE_DIRECT.as("K");
    private static final List<TableField<?, ?>> DEFAULT_ORDER =
            ImmutableList.of(SHOW_CONDITIONS.CID, SHOW_CONDITIONS.PID, SHOW_CONDITIONS.ID);
    private static final Condition NON_DELETED_SHOW_CONDITIONS =
            SHOW_CONDITIONS.IS_DELETED.eq(RepositoryUtils.booleanToLong(false)).or(YtDSL.isNull(SHOW_CONDITIONS.IS_DELETED));
    private static final Condition NON_ARCHIVED_SHOW_CONDITIONS = SHOW_CONDITIONS.IS_ARCHIVED.eq(YtDSL.ytFalse());

    private static final JooqFilterProcessor<GdiShowConditionFilter> FILTER_PROCESSOR =
            getFilterProcessor(true);
    private static final JooqFilterProcessor<GdiShowConditionFilter> FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS =
            getFilterProcessor(false);

    private static JooqFilterProcessor<GdiShowConditionFilter> getFilterProcessor(boolean withFilterByStatus) {
        return JooqFilterProcessor.<GdiShowConditionFilter>builder()
                .withFilter(GdiShowConditionFilter::getCampaignIdIn, SHOW_CONDITIONS.CID::in)
                .withFilter(GdiShowConditionFilter::getAdGroupIdIn, SHOW_CONDITIONS.PID::in)
                .withFilter(GdiShowConditionFilter::getShowConditionIdIn, SHOW_CONDITIONS.ID::in)
                .withFilter(GdiShowConditionFilter::getShowConditionIdNotIn, not(SHOW_CONDITIONS.ID::in))
                .withFilter(GdiShowConditionFilter::getKeywordContains, substringIgnoreCase(SHOW_CONDITIONS.PHRASE))
                .withFilter(GdiShowConditionFilter::getKeywordIn, inSetIgnoreCase(SHOW_CONDITIONS.PHRASE))
                .withFilter(GdiShowConditionFilter::getKeywordNotContains,
                        not(substringIgnoreCase(SHOW_CONDITIONS.PHRASE)))
                .withFilter(GdiShowConditionFilter::getKeywordNotIn, not(inSetIgnoreCase(SHOW_CONDITIONS.PHRASE)))
                .withFilter(GdiShowConditionFilter::getKeywordWithoutMinusWordsContains,
                        inAllSetsOfSetSubstringIgnoreCase(SHOW_CONDITIONS.KEYWORD))
                .withFilter(GdiShowConditionFilter::getMinusWordsContains,
                        inAllSetsOfSetSubstringIgnoreCase(SHOW_CONDITIONS.MINUS_WORDS))
                .withFilter(GdiShowConditionFilter::getMaxPrice,
                        mp -> SHOW_CONDITIONS.PRICE.le(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiShowConditionFilter::getMinPrice,
                        mp -> SHOW_CONDITIONS.PRICE.ge(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiShowConditionFilter::getMaxPriceContext,
                        mp -> SHOW_CONDITIONS.PRICE_CONTEXT.le(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiShowConditionFilter::getMinPriceContext,
                        mp -> SHOW_CONDITIONS.PRICE_CONTEXT.ge(mp.multiply(DECIMAL_MULT).longValue()))
                .withFilter(GdiShowConditionFilter::getAutobudgetPriorityIn,
                        aps -> SHOW_CONDITIONS.AUTOBUDGET_PRIORITY
                                .in(mapList(aps, GridShowConditionMapping::numFromAutobudgetPriority)))
                .withFilter(GdiShowConditionFilter::getTypeIn,
                        aps -> SHOW_CONDITIONS.BID_TYPE
                                .in(mapList(aps, GridShowConditionMapping::stringFromType)))
                .withFilter(GdiShowConditionFilter::getShowConditionStatusIn,
                        s -> GridShowConditionYtRepository.buildFromPrimaryStatuses(s, withFilterByStatus))
                .build();
    }

    private final YtDynamicSupport ytSupport;
    private final YtFieldMapper<GdiShowCondition, BidstableDirectRecord> showConditionsMapper;
    private final List<OrderField<?>> defaultOrderBy;
    private final Map<GdiShowConditionOrderByField, Field<?>> sortingMap;
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);

    @Autowired
    public GridShowConditionYtRepository(YtDynamicSupport ytSupport, GridKeywordsParser keywordsParser) {
        this.ytSupport = ytSupport;

        JooqReaderWithSupplier<GdiShowCondition> internalReader =
                JooqReaderWithSupplierBuilder.builder(GdiShowCondition::new)
                        .readProperty(GdiShowCondition.ID, fromField(SHOW_CONDITIONS.ID))
                        .readProperty(GdiShowCondition.GROUP_ID, fromField(SHOW_CONDITIONS.PID))
                        .readProperty(GdiShowCondition.CAMPAIGN_ID, fromField(SHOW_CONDITIONS.CID))
                        .readPropertyForFirst(GdiShowCondition.BS_ID, fromField(SHOW_CONDITIONS.PHRASE_ID))
                        .readPropertyForFirst(GdiShowCondition.KEYWORD, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(keywordsParser::getPlainKeyword))
                        .readPropertyForFirst(GdiShowCondition.IS_AUTOTARGETING, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(KeywordMapping::isAutotargetingFromDb))
                        .readPropertyForFirst(GdiShowCondition.MINUS_KEYWORDS, fromField(SHOW_CONDITIONS.PHRASE)
                                .by(keywordsParser::getMinusKeywords))
                        .readPropertyForFirst(GdiShowCondition.PRICE, fromField(SHOW_CONDITIONS.PRICE)
                                .by(YtMappingUtils::fromMicros))
                        .readPropertyForFirst(GdiShowCondition.PRICE_CONTEXT, fromField(SHOW_CONDITIONS.PRICE_CONTEXT)
                                .by(YtMappingUtils::fromMicros))
                        .readPropertyForFirst(GdiShowCondition.SHOWS_FORECAST,
                                fromField(SHOW_CONDITIONS.SHOWS_FORECAST))
                        .readPropertyForFirst(GdiShowCondition.AUTOBUDGET_PRIORITY,
                                fromField(SHOW_CONDITIONS.AUTOBUDGET_PRIORITY)
                                        .by(GridShowConditionMapping::autobudgetPriorityFromNum))
                        .readPropertyForFirst(GdiShowCondition.TYPE, fromField(SHOW_CONDITIONS.BID_TYPE)
                                .by(GridShowConditionMapping::typeFromString))
                        .readPropertyForFirst(GdiShowCondition.STATUS_MODERATE,
                                fromStringFieldToEnum(SHOW_CONDITIONS.STATUS_MODERATE,
                                        GdiShowConditionStatusModerate.class))
                        .readPropertyForFirst(GdiShowCondition.SUSPENDED,
                                fromLongFieldToBoolean(SHOW_CONDITIONS.IS_SUSPENDED))
                        .readPropertyForFirst(GdiShowCondition.DELETED,
                                fromLongFieldToBoolean(SHOW_CONDITIONS.IS_DELETED))
                        .readPropertyForFirst(GdiShowCondition.SEARCH_STOP,
                                fromLongFieldToBoolean(SHOW_CONDITIONS.SEARCH_STOP))
                        .readPropertyForFirst(GdiShowCondition.NET_STOP,
                                fromLongFieldToBoolean(SHOW_CONDITIONS.NET_STOP))
                        .readPropertyForFirst(GdiShowCondition.ARCHIVED,
                                fromLongFieldToBoolean(SHOW_CONDITIONS.IS_ARCHIVED))
                        .build();

        showConditionsMapper = new YtFieldMapper<>(internalReader, SHOW_CONDITIONS);

        defaultOrderBy = mapList(DEFAULT_ORDER, showConditionsMapper::fieldAlias);

        sortingMap = new HashMap<>();
        sortingMap.put(GdiShowConditionOrderByField.GROUP_ID, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.PID));
        sortingMap.put(GdiShowConditionOrderByField.CAMPAIGN_ID, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.CID));
        sortingMap.put(GdiShowConditionOrderByField.ID, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.ID));
        sortingMap.put(GdiShowConditionOrderByField.KEYWORD, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.PHRASE));
        sortingMap.put(GdiShowConditionOrderByField.PRICE, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.PRICE));
        sortingMap.put(GdiShowConditionOrderByField.PRICE_CONTEXT,
                showConditionsMapper.fieldAlias(SHOW_CONDITIONS.PRICE_CONTEXT));
        sortingMap.put(GdiShowConditionOrderByField.AUTOBUDGET_PRIORITY,
                showConditionsMapper.fieldAlias(SHOW_CONDITIONS.AUTOBUDGET_PRIORITY));
        sortingMap.put(GdiShowConditionOrderByField.TYPE, showConditionsMapper.fieldAlias(SHOW_CONDITIONS.BID_TYPE));

        gridStat.addStatOrderingToMap(sortingMap, GdiShowConditionOrderByField.class);
    }

    private static Condition buildFromPrimaryStatuses(Collection<GdiShowConditionPrimaryStatus> statuses,
                                                      boolean withFilterByStatus) {
        if (!withFilterByStatus) {
            return NON_DELETED_SHOW_CONDITIONS;
        } else {
            return statuses.stream()
                    .map(GridShowConditionYtRepository::buildFromPrimaryStatus)
                    .reduce(Condition::or)
                    .orElse(NON_DELETED_SHOW_CONDITIONS);
        }
    }

    private static Condition buildFromPrimaryStatus(GdiShowConditionPrimaryStatus status) {
        if (status == GdiShowConditionPrimaryStatus.ACTIVE) {
            return SHOW_CONDITIONS.STATUS_MODERATE.eq(toSource(GdiShowConditionStatusModerate.YES).getLiteral())
                    .and(SHOW_CONDITIONS.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(false)))
                    .and(NON_ARCHIVED_SHOW_CONDITIONS)
                    .and(NON_DELETED_SHOW_CONDITIONS);
        } else if (status == GdiShowConditionPrimaryStatus.STOPPED) {
            return SHOW_CONDITIONS.STATUS_MODERATE.eq(toSource(GdiShowConditionStatusModerate.YES).getLiteral())
                    .and(SHOW_CONDITIONS.IS_SUSPENDED.eq(RepositoryUtils.booleanToLong(true)))
                    .and(NON_ARCHIVED_SHOW_CONDITIONS)
                    .and(NON_DELETED_SHOW_CONDITIONS);
        } else if (status == GdiShowConditionPrimaryStatus.DRAFT) {
            return SHOW_CONDITIONS.STATUS_MODERATE.eq(toSource(GdiShowConditionStatusModerate.NEW).getLiteral())
                    .and(NON_ARCHIVED_SHOW_CONDITIONS)
                    .and(NON_DELETED_SHOW_CONDITIONS);
        } else if (status == GdiShowConditionPrimaryStatus.MODERATION_REJECTED) {
            return SHOW_CONDITIONS.STATUS_MODERATE.eq(toSource(GdiShowConditionStatusModerate.NO).getLiteral())
                    .and(NON_ARCHIVED_SHOW_CONDITIONS)
                    .and(NON_DELETED_SHOW_CONDITIONS);
        } else if (status == GdiShowConditionPrimaryStatus.ARCHIVED) {
            return SHOW_CONDITIONS.IS_ARCHIVED.eq(YtDSL.ytTrue())
                    .and(NON_DELETED_SHOW_CONDITIONS);
        }
        throw new IllegalArgumentException("Unknown status: " + status);
    }

    /**
     * Получить данные об условиях показа
     *
     * @param shard                              шард, в котором хранятся условия показа (исключительно для улучшения
     *                                           результатов запроса)
     * @param filter                             фильтр для выборки данных
     * @param showConditionOrderByList           список полей с порядком сортировки данных
     * @param statFrom                           начала периода, за который нужно получать статистику
     * @param statTo                             конец периода, за который нужно получать статистику (включительно)
     * @param limitOffset                        верхний предел количества полученных условий показа и смещение
     *                                           относительно начала выборки
     * @param goalIds                            идентификаторы целей
     * @param goalIdsForRevenue                  идентификаторы целей, по которым считается доход
     * @param showConditionFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля
     *                                           запрошены на верхнем уровне
     * @param disableStatusFilter                не фильтровать по статусам
     * @param addWithTotalsToQuery               добавляет в запросе with totals с получением общей сатистики
     */
    public GdiShowConditionWithTotals getShowConditions(int shard, GdiShowConditionFilter filter,
                                                        List<GdiShowConditionOrderBy> showConditionOrderByList,
                                                        LocalDate statFrom, LocalDate statTo, LimitOffset limitOffset,
                                                        Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue,
                                                        ShowConditionFetchedFieldsResolver showConditionFetchedFieldsResolver,
                                                        boolean disableStatusFilter, boolean addWithTotalsToQuery) {
        if (CollectionUtils.isAllEmpty(filter.getCampaignIdIn(), filter.getAdGroupIdIn(),
                filter.getShowConditionIdIn(), filter.getShowConditionIdNotIn())) {
            return new GdiShowConditionWithTotals()
                    .withGdiShowConditions(Collections.emptyList());
        }

        Directphrasestatv2Bs statTable = gridStat.getTableData().table();
        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(showConditionsMapper.getFieldsToRead())
                .select(showConditionsMapper.getFieldsToReadForFirst());


        //Читаем статистику, только если она запрошена или есть фильтры по ней
        boolean hasStatFieldsSort = gridStat.hasStatFieldsSort(showConditionOrderByList);
        boolean withStatistic = showConditionFetchedFieldsResolver.getStats() || filter.getStats() != null
                || filter.getGoalStats() != null || hasStatFieldsSort;
        if (withStatistic) {
            TableOnConditionStep<Record> joinStep = gridStat.joinStatColumns(SHOW_CONDITIONS,
                    row(SHOW_CONDITIONS.CID, SHOW_CONDITIONS.PID, YtDSL.toULong(SHOW_CONDITIONS.ID))
                            .eq(statTable.EXPORT_ID, statTable.GROUP_EXPORT_ID, statTable.PHRASE_EXPORT_ID),
                    statFrom, statTo, null);
            joinStep = gridStat.joinGoalStatColumns(joinStep, goalIds);

            selectStep.select(gridStat.getStatSelectFields());
            selectStep.select(gridStat.getGoalStatFields(goalIds, goalIdsForRevenue));
            selectStep.from(joinStep);
        } else {
            selectStep.from(SHOW_CONDITIONS);
        }
        Condition baseCondition =
                (disableStatusFilter ? FILTER_PROCESSOR_WITHOUT_FILTER_BY_STATUS : FILTER_PROCESSOR).apply(filter);

        // Группируем по всем выбираемым полям, так как они не аггрегируемые
        List<Field<?>> groupBy = new ArrayList<>(showConditionsMapper.getFieldsToRead());
        // И добавляем ключи для ускорения
        groupBy.add(SHOW_CONDITIONS.__HASH__);
        groupBy.add(SHOW_CONDITIONS.__SHARD__);

        selectStep.where(baseCondition)
                .groupBy(groupBy);

        SelectWindowStep<Record> windowStep = selectStep;

        Condition havingCondition = gridStat.getStatHavingCondition(filter.getStats());
        if (filter.getGoalStats() != null) {
            havingCondition = gridStat.addGoalStatHavingCondition(havingCondition, filter.getGoalStats());
        }

        if (havingCondition != null) {
            windowStep = selectStep.having(havingCondition);
        }

        List<OrderField<?>> orderFields = gridStat.getOrderFields(sortingMap, showConditionOrderByList, defaultOrderBy);

        Select query = windowStep
                .orderBy(orderFields)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        List<GdiShowCondition> showConditions = ytSupport
                .selectRows(shard, query, addWithTotalsToQuery && withStatistic).getYTreeRows().stream()
                .map(n -> showConditionsMapper.fromNode(n)
                        .withStat(GridStatNew.addZeros(goalIdsForRevenue != null ?
                                gridStat.extractStatsEntryWithGoalsRevenue(n, goalIds, goalIdsForRevenue) :
                                gridStat.extractStatsEntry(n)))
                        .withGoalStats(gridStat.extractGoalStatEntries(n, goalIds)))
                .map(condition -> condition
                        .withSuspended(nvl(condition.getSuspended(), false))
                        .withDeleted(nvl(condition.getDeleted(), false)))
                .collect(Collectors.toList());


        // Возвращаем totals из запроса если кол. условий показа = лимиту (showConditions.size == limit + total row)
        boolean withTotalStats = WithTotalsUtils
                .withTotalStats(addWithTotalsToQuery, withStatistic, showConditions, limitOffset);

        GdiShowCondition showConditionWithTotals = !withTotalStats ? null : StreamEx.of(showConditions)
                .findFirst(showCondition -> Objects.isNull(showCondition.getId()))
                .orElse(null);

        return new GdiShowConditionWithTotals()
                .withGdiShowConditions(StreamEx.of(showConditions)
                        .filter(showCondition -> Objects.nonNull(showCondition.getId()))
                        .toList())
                .withTotalStats(showConditionWithTotals != null ? showConditionWithTotals.getStat() : null);
    }
}
